package ru.yandex.direct.core.entity.deal.service.validation;

import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.jooq.DSLContext;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.campaign.model.CampaignSimple;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.client.model.Client;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.deal.container.CampaignDeal;
import ru.yandex.direct.core.entity.deal.container.UpdateDealContainer;
import ru.yandex.direct.core.entity.deal.model.Deal;
import ru.yandex.direct.core.entity.deal.model.DealAdfox;
import ru.yandex.direct.core.entity.deal.model.StatusDirect;
import ru.yandex.direct.core.entity.deal.model.UpdatableDeal;
import ru.yandex.direct.core.entity.deal.repository.DealRepository;
import ru.yandex.direct.core.entity.deal.service.DealTransitionsService;
import ru.yandex.direct.currency.CurrencyCode;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.rbac.RbacService;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.defect.CommonDefects;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static java.util.Collections.emptyList;
import static java.util.Collections.singletonList;
import static java.util.function.Function.identity;
import static org.apache.commons.collections4.CollectionUtils.isEmpty;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignDefects.archivedCampaignModification;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignDefects.campaignTypeNotSupported;
import static ru.yandex.direct.core.entity.deal.service.validation.DealConstraints.dealExists;
import static ru.yandex.direct.core.entity.deal.service.validation.DealDefects.dealCurrencyShouldMatchCampaign;
import static ru.yandex.direct.core.entity.deal.service.validation.DealDefects.dealIsNotActive;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;
import static ru.yandex.direct.validation.constraint.CommonConstraints.eachNotNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.inSet;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notInSet;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.validId;
import static ru.yandex.direct.validation.constraint.StringConstraints.hasNoForbiddenChars;
import static ru.yandex.direct.validation.constraint.StringConstraints.maxStringLength;
import static ru.yandex.direct.validation.constraint.StringConstraints.notBlank;
import static ru.yandex.direct.validation.constraint.StringConstraints.notControlChars;
import static ru.yandex.direct.validation.defect.CommonDefects.objectNotFound;

@Service
public class DealValidationService {

    /**
     * Типы кампаний, к которым можно привязывать сделки
     */
    static final List<CampaignType> AVAILABLE_CAMPAIGN_TYPES = singletonList(CampaignType.CPM_DEALS);
    /**
     * Максамильное число сделок, которое можно привязать к одной кампании
     */
    private static final int MAX_NUMBER_OF_DEALS_IN_CAMPAIGN = 1;
    private static final int MAX_DEAL_NAME_LENGTH = 255;
    private static final int MAX_DEAL_DESCRIPTION_LENGTH = 255;

    private static final List<String> FORBIDDEN_CHARS_IN_DEAL_NAME = Arrays.asList("<", ">");
    private static final List<String> FORBIDDEN_CHARS_IN_DEAL_DESCRIPTION = Arrays.asList("<", ">");

    private final RbacService rbacService;
    private final DealRepository dealRepository;
    private final CampaignRepository campaignRepository;
    private final DealTransitionsService dealTransitionsService;
    private final ClientService clientService;
    private final ShardHelper shardHelper;
    private final DslContextProvider dslContextProvider;

    @Autowired
    public DealValidationService(RbacService rbacService,
                                 DealRepository dealRepository,
                                 CampaignRepository campaignRepository,
                                 DealTransitionsService dealTransitionsService,
                                 ClientService clientService, ShardHelper shardHelper,
                                 DslContextProvider dslContextProvider) {
        this.rbacService = rbacService;
        this.dealRepository = dealRepository;
        this.campaignRepository = campaignRepository;
        this.dealTransitionsService = dealTransitionsService;
        this.clientService = clientService;
        this.shardHelper = shardHelper;
        this.dslContextProvider = dslContextProvider;
    }

    public ValidationResult<List<Deal>, Defect> validateAddDeal(ClientId clientId, List<Deal> deals) {
        ListValidationBuilder<Deal, Defect> lvb = ListValidationBuilder.of(deals);
        lvb.check(eachNotNull());

        Client client = clientService.getClient(clientId);
        if (lvb.getResult().hasAnyErrors() || client == null) {
            // todo maxlog: подумать, как лучше обработать ошибку "клиент не найден"
            return lvb.getResult();
        }
        CurrencyCode clientCurrency = client.getWorkCurrency();

        lvb.checkEachBy(dealValidator(clientCurrency));
        return lvb.getResult();
    }

    private Validator<Deal, Defect> dealValidator(CurrencyCode clientCurrency) {
        return (deal) -> {
            ModelItemValidationBuilder<Deal> ivb = ModelItemValidationBuilder.of(deal);
            if (deal == null) {
                // null-tolerance
                return ivb.getResult();
            }
            ivb.item(Deal.ID)
                    .check(notNull());
            ivb.item(Deal.CLIENT_ID)
                    .check(notNull());
            ivb.item(Deal.ADFOX_STATUS)
                    .check(notNull());
            ivb.item(Deal.DEAL_TYPE)
                    .check(notNull());
            ivb.item(Deal.CURRENCY_CODE)
                    .check(notNull());
            //todo maxlog: починить с учетом множества валют у агентства
//                    .check(Constraint.fromPredicate(currencyCode -> currencyCode == clientCurrency,
//                            dealCurrencyShouldMatchClient(clientCurrency)));
            ivb.item(Deal.ADFOX_NAME)
                    .check(notNull());

            ivb.item(Deal.DATE_START)
                    .check(notNull());
            ivb.item(Deal.DATE_END)
                    .check(notNull());
            ivb.item(Deal.CPM)
                    .check(notNull());
            ivb.item(Deal.DEAL_JSON)
                    .check(notNull());

            return ivb.getResult();
        };
    }

    /**
     * Проверка заявок на обновление параметров сделки, управляемых Adfox'ом.
     */
    public ValidationResult<List<DealAdfox>, Defect> validateUpdateDealAdfox(List<DealAdfox> deals) {
        ListValidationBuilder<DealAdfox, Defect> lvb =
                ListValidationBuilder.of(deals, Defect.class);
        // Проверяем, что необходимые поля заполнены
        lvb.check(eachNotNull())
                .checkEachBy(updateSingleDealAdfoxValidator());

        if (lvb.getResult().hasAnyErrors()) {
            return lvb.getResult();
        }

        // Проверяем, что указаны ID существующих сделок
        List<Long> dealIds = mapList(deals, DealAdfox::getId);
        List<DealAdfox> existingDeals = dealRepository.getDealsAdfox(dealIds);
        HashSet<Long> existingDealIds = new HashSet<>(mapList(existingDeals, DealAdfox::getId));
        lvb.checkEach(dealExists(existingDealIds));
        // Не проверяем, что изменение статуса соответствует lifecycle'у, так как должны отражать данные Adfox'а "как есть"
        return lvb.getResult();
    }

    /**
     * @return Валидатор для проверки заполненности параметров в заявке на обновление сделки.
     */
    private Validator<DealAdfox, Defect> updateSingleDealAdfoxValidator() {
        return deal -> {
            ModelItemValidationBuilder<DealAdfox> ivb = ModelItemValidationBuilder.of(deal);
            if (deal == null) {
                return ivb.getResult();
            }
            ivb.item(DealAdfox.ID)
                    .check(notNull())
                    .check(validId());
            ivb.item(DealAdfox.ADFOX_STATUS)
                    .check(notNull());
            return ivb.getResult();
        };
    }

    public ValidationResult<List<Long>, Defect> validateChangeStatus(int shard, List<Long> dealIds,
                                                                     StatusDirect status) {
        DSLContext dslContext = dslContextProvider.ppc(shard);
        return validateChangeStatus(dslContext, dealIds, status);
    }

    public ValidationResult<List<Long>, Defect> validateChangeStatus(DSLContext dslContext,
                                                                     List<Long> dealIds, StatusDirect status) {
        ListValidationBuilder<Long, Defect> v = ListValidationBuilder.of(dealIds);
        v.check(notNull());
        if (v.getResult().hasAnyErrors()) {
            return v.getResult();
        }

        Map<Long, StatusDirect> dealsWithStatuses = dealRepository.getDealsStatuses(dslContext, dealIds);

        v.checkEach(notNull());
        v.checkEach(inSet(dealsWithStatuses.keySet()), objectNotFound(), When.isValid());
        v.checkEach(Constraint.fromPredicate(
                dealId -> dealTransitionsService.getAvailableStatuses(dealsWithStatuses.get(dealId)).contains(status),
                DealDefects.transitionIsUnavailable()), When.isValid());
        return v.getResult();
    }

    private ValidationResult<List<UpdatableDeal>, Defect> validateUpdateDeal(int shard,
                                                                             ClientId clientId, List<UpdatableDeal> updatableDealList) {
        ListValidationBuilder<UpdatableDeal, Defect> v =
                ListValidationBuilder.of(updatableDealList);
        if (v.getResult().hasAnyErrors()) {
            return v.getResult();
        }
        List<Long> dealIds = mapList(updatableDealList, UpdatableDeal::getId);

        Set<Long> existsDealIds =
                StreamEx.of(dealRepository.getDeals(shard, clientId, dealIds)).map(Deal::getId).toSet();

        v.checkEach(notNull());
        v.checkEachBy(updateDealValidator(existsDealIds), When.isValid());
        return v.getResult();
    }

    private Validator<UpdatableDeal, Defect> updateDealValidator(Set<Long> existsDealIds) {
        return updatableDeal -> {
            ItemValidationBuilder<UpdatableDeal, Defect> v =
                    ItemValidationBuilder.of(updatableDeal);
            v.item(updatableDeal.getId(), Deal.ID.name())
                    .weakCheck(inSet(existsDealIds), objectNotFound());
            v.item(updatableDeal.getName(), Deal.NAME.name())
                    .weakCheck(notBlank())
                    .weakCheck(maxStringLength(MAX_DEAL_NAME_LENGTH), When.isValid())
                    .weakCheck(hasNoForbiddenChars(FORBIDDEN_CHARS_IN_DEAL_NAME), When.isValid())
                    .weakCheck(notControlChars(), When.isValid());
            v.item(updatableDeal.getDescription(), Deal.DESCRIPTION.name())
                    .weakCheck(notBlank())
                    .weakCheck(maxStringLength(MAX_DEAL_DESCRIPTION_LENGTH), When.isValid())
                    .weakCheck(hasNoForbiddenChars(FORBIDDEN_CHARS_IN_DEAL_DESCRIPTION), When.isValid())
                    .weakCheck(notControlChars(), When.isValid());
            return v.getResult();
        };
    }

    public ValidationResult<UpdateDealContainer, Defect> validateUpdateDeal(int shard,
                                                                            long agencyUid,
                                                                            ClientId agencyId,
                                                                            UpdateDealContainer updateDealContainer) {
        ModelItemValidationBuilder<UpdateDealContainer> v =
                ModelItemValidationBuilder.of(updateDealContainer);

        //один из списков должен быть не пустым
        v.check(fromPredicate(t -> !(isEmpty(t.getAdded()) && isEmpty(t.getRemoved()) && isEmpty(t.getDeals())),
                CommonDefects.requiredButEmpty()));
        if (v.getResult().hasAnyErrors()) {
            return v.getResult();
        }

        Set<Long> requestedCampaigns = getRequestedIds(updateDealContainer, CampaignDeal::getCampaignId);
        Map<Long, Deal> existingDeals = getExistingDeals(shard, agencyId);
        Map<Long, CampaignSimple> existingCampaigns = getExistingCampaigns(agencyUid, requestedCampaigns);

        v.list(updateDealContainer.getAdded(), UpdateDealContainer.ADDED.name())
                .checkEachBy(t -> dealToCampaignValidator(existingCampaigns, existingDeals, t), When.notNull())
                //валюты сделки и кампании должны совпадать
                .checkEachBy(t -> currencyCodeShouldMatch(existingCampaigns, existingDeals, t),
                        When.isValidAnd(When.notNull()));

        v.list(updateDealContainer.getRemoved(), UpdateDealContainer.REMOVED.name())
                .checkEachBy(t -> dealToCampaignValidator(existingCampaigns, existingDeals, t), When.notNull());
        v.list(updateDealContainer.getDeals(), UpdateDealContainer.DEALS.name())
                .checkBy(t -> validateUpdateDeal(shard, agencyId, t), When.notNull());
        return v.getResult();
    }

    /**
     * Вытащить список переданных id
     */
    public Set<Long> getRequestedIds(UpdateDealContainer updateDealContainer,
                                     Function<CampaignDeal, Long> idFunction) {
        Set<Long> requestedIds = new HashSet<>();
        for (CampaignDeal campaignDeal : Optional.ofNullable(updateDealContainer.getAdded())
                .orElse(emptyList())) {
            requestedIds.add(idFunction.apply(campaignDeal));
        }

        for (CampaignDeal campaignDeal : Optional.ofNullable(updateDealContainer.getRemoved())
                .orElse(emptyList())) {
            requestedIds.add(idFunction.apply(campaignDeal));
        }
        return requestedIds;
    }

    /**
     * Проверяет, что валюта сделки матчится с валютой кампании
     * в existingCampaigns обязательно должна быть campaignDeal.getCampaignId(), что должно проверяться валидацией выше
     */
    private ValidationResult<CampaignDeal, Defect> currencyCodeShouldMatch(
            Map<Long, CampaignSimple> existingCampaigns, Map<Long, Deal> existingDeals, CampaignDeal campaignDeal) {
        ModelItemValidationBuilder<CampaignDeal> v = ModelItemValidationBuilder.of(campaignDeal);
        if ((existingCampaigns.get(campaignDeal.getCampaignId()) != null)
                && (existingDeals.get(campaignDeal.getDealId()) != null)) {
            v.weakCheck(fromPredicate(
                    t -> existingCampaigns.get(t.getCampaignId()).getCurrency() ==
                            existingDeals.get(t.getDealId()).getCurrencyCode(),
                    dealCurrencyShouldMatchCampaign(
                            existingCampaigns.get(campaignDeal.getCampaignId()).getCurrency())));
        }
        return v.getResult();
    }

    private ValidationResult<CampaignDeal, Defect> dealToCampaignValidator(
            Map<Long, CampaignSimple> existingCampaigns,
            Map<Long, Deal> existingDeals, CampaignDeal dealCampaign) {
        Map<Long, CampaignType> campaignTypeMap = EntryStream.of(existingCampaigns)
                .mapValues(CampaignSimple::getType)
                .toMap();

        Set<Long> archivedCampaigns = EntryStream.of(existingCampaigns)
                .filterValues(CampaignSimple::getStatusArchived)
                .map(Map.Entry::getKey)
                .toSet();

        Map<Long, StatusDirect> dealsStatusMap = EntryStream.of(existingDeals)
                .mapValues(Deal::getDirectStatus)
                .toMap();

        ModelItemValidationBuilder<CampaignDeal> v = ModelItemValidationBuilder.of(dealCampaign);
        v.item(dealCampaign.getCampaignId(), CampaignDeal.CAMPAIGN_ID.name())
                //оператор имеет доступ на запись
                .weakCheck(inSet(existingCampaigns.keySet()), objectNotFound())
                //кампании не архивны
                .weakCheck(notInSet(archivedCampaigns), archivedCampaignModification(), When.isValid())
                //подключается разрешенный тип кампании
                .weakCheck(
                        fromPredicate(campaignId -> AVAILABLE_CAMPAIGN_TYPES.contains(campaignTypeMap.get(campaignId)),
                                campaignTypeNotSupported()),
                        When.isValidAnd(When.valueIs(existingCampaigns::containsKey)));
        v.item(dealCampaign.getDealId(), CampaignDeal.DEAL_ID.name())
                //сделки существуют на клиенте
                .weakCheck(inSet(existingDeals.keySet()), objectNotFound())
                //сделки активны
                .weakCheck(fromPredicate(t -> dealsStatusMap.get(t) == StatusDirect.ACTIVE, dealIsNotActive()),
                        When.isValid());
        return v.getResult();
    }

    /**
     * Существующие кампании и доступные на запись переданному agencyUid
     */
    Map<Long, CampaignSimple> getExistingCampaigns(long agencyUid, Set<Long> requestedCampaigns) {
        Set<Long> writableCampaigns = rbacService.getWritableCampaigns(agencyUid, requestedCampaigns);
        Map<Long, CampaignSimple> campaignSimpleMap = new HashMap<>();
        shardHelper.groupByShard(writableCampaigns, ShardKey.CID)
                .chunkedByDefault()
                .forEach(
                        (shard, campaignIds) -> campaignSimpleMap
                                .putAll(campaignRepository.getCampaignsSimple(shard, campaignIds))
                );
        return campaignSimpleMap;
    }

    Map<Long, Deal> getExistingDeals(int shard, ClientId clientId) {
        return StreamEx.of(dealRepository.getDealsBriefByClientId(shard, clientId))
                .toMap(Deal::getId, identity());
    }
}
