package ru.yandex.direct.core.entity.dialogs.repository;

import java.time.LocalDateTime;
import java.util.Collection;
import java.util.List;
import java.util.Map;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.common.util.RepositoryUtils;
import ru.yandex.direct.core.entity.campaign.model.Dialog;
import ru.yandex.direct.dbschema.ppc.tables.records.ClientDialogsRecord;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.JooqUpdateBuilder;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.multitype.entity.LimitOffset;

import static ru.yandex.direct.common.jooqmapperex.ReaderWriterBuildersEx.booleanProperty;
import static ru.yandex.direct.common.util.RepositoryUtils.booleanToLong;
import static ru.yandex.direct.dbschema.ppc.Tables.CLIENT_DIALOGS;
import static ru.yandex.direct.dbschema.ppc.tables.CampDialogs.CAMP_DIALOGS;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;

@Repository
public class ClientDialogsRepository {
    private final DslContextProvider dslContextProvider;
    private final JooqMapperWithSupplier<Dialog> jooqMapper;
    private final ShardHelper shardHelper;

    @Autowired
    public ClientDialogsRepository(DslContextProvider dslContextProvider, ShardHelper shardHelper) {
        this.dslContextProvider = dslContextProvider;
        this.shardHelper = shardHelper;
        this.jooqMapper = JooqMapperWithSupplierBuilder.builder(Dialog::new)
                .map(property(Dialog.ID, CLIENT_DIALOGS.CLIENT_DIALOG_ID))
                .map(property(Dialog.CLIENT_ID, CLIENT_DIALOGS.CLIENT_ID))
                .map(property(Dialog.SKILL_ID, CLIENT_DIALOGS.SKILL_ID))
                .map(property(Dialog.BOT_GUID, CLIENT_DIALOGS.BOT_GUID))
                .map(property(Dialog.NAME, CLIENT_DIALOGS.NAME))
                .map(booleanProperty(Dialog.IS_ACTIVE, CLIENT_DIALOGS.IS_ACTIVE))
                .map(property(Dialog.LAST_SYNC_TIME, CLIENT_DIALOGS.LAST_SYNC_TIME))
                .build();
    }

    public List<Dialog> getDialogs(int shard, LimitOffset limitOffset) {
        return dslContextProvider.ppc(shard)
                .select(jooqMapper.getFieldsToRead())
                .from(CLIENT_DIALOGS)
                .orderBy(CLIENT_DIALOGS.CLIENT_DIALOG_ID)
                .limit(limitOffset.limit())
                .offset(limitOffset.offset())
                .fetch(jooqMapper::fromDb);
    }

    /**
     * Получает все диалоги, которые были обновлены раньше, чем lastSyncTime
     */
    public List<Dialog> getDialogs(int shard, LocalDateTime lastSyncTime) {
        return dslContextProvider.ppc(shard)
                .select(jooqMapper.getFieldsToRead())
                .from(CLIENT_DIALOGS)
                .where(CLIENT_DIALOGS.LAST_SYNC_TIME.le(lastSyncTime))
                .orderBy(CLIENT_DIALOGS.CLIENT_DIALOG_ID)
                .fetch(jooqMapper::fromDb);
    }

    /**
     * Возвращает все чаты клиента (вне зависимости от привязки к кампании)
     */
    public List<Dialog> getDialogsByClientId(ClientId clientId) {
        int shard = shardHelper.getShardByClientId(clientId);
        return dslContextProvider.ppc(shard)
                .select(jooqMapper.getFieldsToRead())
                .from(CLIENT_DIALOGS)
                .where(CLIENT_DIALOGS.CLIENT_ID.eq(clientId.asLong()))
                .fetch(jooqMapper::fromDb);
    }

    public Map<String, List<Long>> getCidByDialogSkillIds(int shard, Collection<String> skillIds) {
        return dslContextProvider.ppc(shard)
                .select(CLIENT_DIALOGS.SKILL_ID, CAMP_DIALOGS.CID)
                .from(CAMP_DIALOGS)
                .leftJoin(CLIENT_DIALOGS).on(CLIENT_DIALOGS.CLIENT_DIALOG_ID.eq(CAMP_DIALOGS.CLIENT_DIALOG_ID))
                .where(CLIENT_DIALOGS.SKILL_ID.in(skillIds))
                .fetchGroups(CLIENT_DIALOGS.SKILL_ID, CAMP_DIALOGS.CID);
    }

    /**
     * Добавляет новый диалог к клиенту
     * @return client_dialog_id
     */
    public Long addOrUpdateDialogToClient(Dialog dialog) {
        LocalDateTime syncTime = dialog.getLastSyncTime() == null ? LocalDateTime.now() : dialog.getLastSyncTime();
        int shard = shardHelper.getShardByClientId(ClientId.fromLong(dialog.getClientId()));
        List<Long> existedDialogIds = dslContextProvider.ppc(shard)
                .select(CLIENT_DIALOGS.CLIENT_DIALOG_ID)
                .from(CLIENT_DIALOGS)
                .where(CLIENT_DIALOGS.SKILL_ID.eq(dialog.getSkillId()))
                .and(CLIENT_DIALOGS.CLIENT_ID.eq(dialog.getClientId()))
                .fetch(CLIENT_DIALOGS.CLIENT_DIALOG_ID);
        long clientDialogId;
        if (existedDialogIds == null || existedDialogIds.isEmpty()) {
            clientDialogId = shardHelper.generateClientDialogIds(1).get(0);
            dslContextProvider.ppc(shard)
                    .insertInto(CLIENT_DIALOGS)
                    .set(CLIENT_DIALOGS.CLIENT_DIALOG_ID, clientDialogId)
                    .set(CLIENT_DIALOGS.CLIENT_ID, dialog.getClientId())
                    .set(CLIENT_DIALOGS.SKILL_ID, dialog.getSkillId())
                    .set(CLIENT_DIALOGS.NAME, dialog.getName())
                    .set(CLIENT_DIALOGS.BOT_GUID, dialog.getBotGuid())
                    .set(CLIENT_DIALOGS.LAST_SYNC_TIME, syncTime)
                    .set(CLIENT_DIALOGS.IS_ACTIVE, booleanToLong(dialog.getIsActive()))
                    .execute();
        } else {
            // client_dialogs не может содержать больше одной записи с одними значениями skill_id и clientId
            clientDialogId = existedDialogIds.get(0);
            dslContextProvider.ppc(shard)
                    .update(CLIENT_DIALOGS)
                    .set(CLIENT_DIALOGS.NAME, dialog.getName())
                    .set(CLIENT_DIALOGS.BOT_GUID, dialog.getBotGuid())
                    .set(CLIENT_DIALOGS.LAST_SYNC_TIME, syncTime)
                    .set(CLIENT_DIALOGS.IS_ACTIVE, booleanToLong(dialog.getIsActive()))
                    .where(CLIENT_DIALOGS.CLIENT_DIALOG_ID.eq(clientDialogId))
                    .execute();
        }
        return clientDialogId;
    }

    public void addDialogToCampaign(int shard, Long campaignId, Long clientDialogId) {
        dslContextProvider.ppc(shard)
                .insertInto(CAMP_DIALOGS)
                .set(CAMP_DIALOGS.CID, campaignId)
                .set(CAMP_DIALOGS.CLIENT_DIALOG_ID, clientDialogId)
                .execute();
    }

    public void addDialog(int shard, Long campaignId, Dialog dialog) {
        long clientDialogId = addOrUpdateDialogToClient(dialog);
        dialog.setId(clientDialogId);
        addDialogToCampaign(shard, campaignId, clientDialogId);
    }

    /**
     * Обновление диалога после синхронизации с Яндекс Диалогами
     */
    public void updateDialogs(int shard, List<AppliedChanges<Dialog>> appliedChangesDialogs) {
        JooqUpdateBuilder<ClientDialogsRecord, Dialog> ub =
                new JooqUpdateBuilder<>(CLIENT_DIALOGS.CLIENT_DIALOG_ID, appliedChangesDialogs);
        ub.processProperty(Dialog.BOT_GUID, CLIENT_DIALOGS.BOT_GUID);
        ub.processProperty(Dialog.NAME, CLIENT_DIALOGS.NAME);
        ub.processProperty(Dialog.IS_ACTIVE, CLIENT_DIALOGS.IS_ACTIVE, RepositoryUtils::booleanToLong);
        ub.processProperty(Dialog.LAST_SYNC_TIME, CLIENT_DIALOGS.LAST_SYNC_TIME);

        dslContextProvider.ppc(shard)
                .update(CLIENT_DIALOGS)
                .set(ub.getValues())
                .where(CLIENT_DIALOGS.CLIENT_DIALOG_ID.in(ub.getChangedIds()))
                .execute();
    }
}
