package ru.yandex.direct.core.entity.dialogs.service;

import java.time.LocalDateTime;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.StatusBsSynced;
import ru.yandex.direct.core.entity.campaign.model.Dialog;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.dialogs.exception.InvalidDialogException;
import ru.yandex.direct.core.entity.dialogs.repository.ClientDialogsRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dialogs.client.DialogsClient;
import ru.yandex.direct.dialogs.client.model.Skill;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.multitype.entity.LimitOffset;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.util.Collections.emptyList;
import static java.util.Collections.singletonList;
import static ru.yandex.direct.core.entity.dialogs.DialogUtils.convertSkillToDialog;
import static ru.yandex.direct.core.entity.dialogs.DialogUtils.normalizeName;
import static ru.yandex.direct.core.entity.dialogs.DialogUtils.skillIsActive;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.validation.defect.CommonDefects.objectNotFound;

@Service
@ParametersAreNonnullByDefault
public class DialogsService {
    private final CampaignRepository campaignRepository;
    private final ClientDialogsRepository clientDialogsRepository;
    private final DialogsClient dialogsClient;

    @Autowired
    public DialogsService(
            CampaignRepository campaignRepository,
            ClientDialogsRepository clientDialogsRepository,
            DialogsClient dialogsClient) {
        this.campaignRepository = campaignRepository;
        this.clientDialogsRepository = clientDialogsRepository;
        this.dialogsClient = dialogsClient;
    }

    public List<Dialog> getDialogs(int shard, LimitOffset limitOffset) {
        return clientDialogsRepository.getDialogs(shard, limitOffset);
    }

    public List<Dialog> getDialogs(int shard, LocalDateTime olderThan) {
        return clientDialogsRepository.getDialogs(shard, olderThan);
    }

    public List<Dialog> getDialogsByClientId(ClientId clientId) {
        return clientDialogsRepository.getDialogsByClientId(clientId);
    }

    public List<Dialog> getDialogsByUserId(ClientId clientId, Long operatorUid) {
        Map<Long, List<Skill>> skillsByUserId = dialogsClient.getSkillsByUserId(singletonList(operatorUid));
        return StreamEx.of(skillsByUserId.values())
                .nonNull()
                .flatMap(Collection::stream)
                .map(skill -> convertSkillToDialog(skill, clientId))
                .toList();
    }

    public Result<Dialog> addDialog(ClientId clientId, String addedSkillId) {
        List<Skill> skillsFromDialogsClient = getSkills(singletonList(addedSkillId));
        Map<String, Skill> skills = listToMap(skillsFromDialogsClient, Skill::getSkillId);
        Skill skill = skills.get(addedSkillId);
        if (!skills.containsKey(addedSkillId) || skill.getError() != null) {
            return Result.broken(ValidationResult.failed(new Dialog().withSkillId(addedSkillId), objectNotFound()));
        }
        if (skill.getBotGuid() == null) {
            throw new InvalidDialogException("Null botGuid for skillId " + addedSkillId);
        }
        Dialog dialog = convertSkillToDialog(skill, clientId);
        Long clientDialogId = clientDialogsRepository.addOrUpdateDialogToClient(dialog);
        dialog.setId(clientDialogId);
        return Result.successful(dialog);
    }

    /**
     * Обновляет диалоги, skillId которых есть в {@code skills}
     */
    public void updateDialogsBySkills(int shard, List<Dialog> dialogs, Map<String, Skill> skills) {
        skills.forEach((id, skill) -> skill.setName(normalizeName(skill.getName())));
        List<AppliedChanges<Dialog>> appliedChangesDialogs = StreamEx.of(dialogs)
                .mapToEntry(Dialog::getSkillId)
                .filterValues(skills::containsKey)
                .mapKeyValue((dialog, skillId) -> appliedChangesDialog(dialog, skills.get(skillId)))
                .toList();
        Map<String, List<Long>> cidBySkillIds = clientDialogsRepository.getCidByDialogSkillIds(shard, skills.keySet());
        Set<Long> cidsToUpdate = StreamEx.of(appliedChangesDialogs)
                .filter(ch -> ch.changed(Dialog.IS_ACTIVE) || ch.changed(Dialog.BOT_GUID))
                .map(AppliedChanges::getModel)
                .map(d -> cidBySkillIds.get(d.getSkillId()))
                .nonNull()
                .toFlatCollection(Function.identity(), HashSet::new);

        clientDialogsRepository.updateDialogs(shard, appliedChangesDialogs);
        campaignRepository.updateStatusBsSynced(shard, cidsToUpdate, StatusBsSynced.NO);
    }

    private AppliedChanges<Dialog> appliedChangesDialog(Dialog dialog, Skill skill) {
        return new ModelChanges<>(dialog.getId(), Dialog.class)
                .processNotNull(skill.getBotGuid(), Dialog.BOT_GUID)
                .processNotNull(skill.getName(), Dialog.NAME)
                .processNotNull(skillIsActive(skill), Dialog.IS_ACTIVE)
                .processNotNull(LocalDateTime.now(), Dialog.LAST_SYNC_TIME)
                .applyTo(dialog);
    }

    public List<Skill> getSkills(List<String> skillsIds) {
        if (skillsIds.isEmpty()) {
            return emptyList();
        }

        return dialogsClient.getSkills(skillsIds);
    }
}
