package ru.yandex.direct.core.entity.domain;

import java.io.UnsupportedEncodingException;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLDecoder;
import java.nio.charset.StandardCharsets;
import java.util.HashSet;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.regex.Pattern;

import javax.annotation.Nullable;

import com.google.common.collect.ImmutableMap;
import one.util.streamex.StreamEx;
import org.apache.commons.lang3.StringUtils;
import org.eclipse.jetty.util.StringUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.common.enums.YandexDomain;
import ru.yandex.direct.core.entity.banner.model.old.OldBanner;
import ru.yandex.direct.core.entity.banner.service.old.BannerUtils;
import ru.yandex.direct.utils.text.StringModifier;

import static java.util.Collections.emptySet;
import static ru.yandex.direct.core.entity.banner.type.href.BannerWithHrefConstants.DOMAIN_PATTERN;


public class AggregatorDomainsUtils {
    private static final Logger logger = LoggerFactory.getLogger(AggregatorDomainsUtils.class);

    private static final String VK_COM_DOMAIN = "vk.com";
    private static final String INSTAGRAM_COM_DOMAIN = "instagram.com";
    private static final String OK_RU_DOMAIN = "ok.ru";
    private static final String YOUTUBE_COM_DOMAIN = "youtube.com";
    private static final String SITES_GOOGLE_COM_DOMAIN = "sites.google.com";
    private static final String TURBO_SITE = "turbo.site";
    private static final String T_ME = "t.me";
    private static final String PROFI_RU = "profi.ru";
    private static final String MAPS_YANDEX = "maps";
    private static final String USLUGI_YANDEX = "uslugi";
    private static final String COLLECTIONS_YANDEX = "collections";
    private static final String ZEN_YANDEX = "zen";
    private static final String POKUPKI_MARKET_YANDEX = "pokupki.market";

    private static final int MIN_SUBDOMAIN_LENGTH = 1;

    private static final Pattern DOMAIN_DISALLOWED_CHARS = Pattern.compile("[^\\p{Alpha}0-9\\-]+",
            Pattern.UNICODE_CHARACTER_CLASS);

    private static final StringModifier SUBDOMAIN_MODIFIER = new StringModifier.Builder()
            .withRegexpReplaceAllRule(DOMAIN_DISALLOWED_CHARS, "-")
            .withRegexpReplaceAllRule("-+", "-")
            .withRegexpReplaceAllRule("^-|-$", "")
            .build();

    private static final Map<String, SubDomainParser> SUBDOMAIN_PARSER_BY_DOMAIN = getSubdomainParserByDomain();

    public static Set<String> registeredDomans() {
        return SUBDOMAIN_PARSER_BY_DOMAIN.keySet();
    }

    private static Map<String, SubDomainParser> getSubdomainParserByDomain() {
        ImmutableMap.Builder<String, SubDomainParser> builder = ImmutableMap.<String, SubDomainParser>builder()
                .put(VK_COM_DOMAIN, new SubDomainParserForVkCom())
                .put(INSTAGRAM_COM_DOMAIN, new SubDomainParserForInstagramCom())
                .put(OK_RU_DOMAIN, new SubDomainParserForOkRu())
                .put(YOUTUBE_COM_DOMAIN, new SubDomainParserForYoutubeCom())
                .put(SITES_GOOGLE_COM_DOMAIN, new SubDomainParserForSitesGoogleCom())
                .put(TURBO_SITE, new SubDomainParserForTurboSite())
                .put(T_ME, new SubDomainParserForTMe())
                .put(PROFI_RU, new SubDomainParserForProfiRu());

        for (YandexDomain value : YandexDomain.values()) {
            SubDomainParserForYandexMaps forYandexMaps = new SubDomainParserForYandexMaps();
            builder.put(MAPS_YANDEX + "." + value.getYandexDomain(), forYandexMaps);

            SubDomainParserForYandexUslugi forYandexUslugi = new SubDomainParserForYandexUslugi();
            builder.put(USLUGI_YANDEX + "." + value.getYandexDomain(), forYandexUslugi);

            SubDomainParserForYandexCollections forYandexCollections = new SubDomainParserForYandexCollections();
            builder.put(COLLECTIONS_YANDEX + "." + value.getYandexDomain(), forYandexCollections);

            SubDomainParserForZen subDomainParserForZen = new SubDomainParserForZen();
            builder.put(ZEN_YANDEX + "." + value.getYandexDomain(), subDomainParserForZen);

            SubDomainParserForPokupki subDomainParserForPokupki = new SubDomainParserForPokupki();
            builder.put(POKUPKI_MARKET_YANDEX + "." + value.getYandexDomain(), subDomainParserForPokupki);
        }
        return builder.build();
    }

    private AggregatorDomainsUtils() {
    }

    /**
     * Возвращает непустой aggregator_domain (вида "test.vk.com") или null
     *
     * @param domain Домен приходит с баннера и должен быть рассчитан заранее, в add или update операциях для баннера
     *               {@link OldBanner#setDomain(String)}
     *               {@link BannerUtils#extractHostFromHrefOrNull(String)}
     */
    @Nullable
    public static String extractAggregatorDomainFromHref(@Nullable String domain, @Nullable String href) {
        if (StringUtil.isBlank(href)) {
            return null;
        }
        href = href.toLowerCase();

        URL url = getURLObjectFromString(href);
        if (url == null || url.getHost() == null) {
            return null;
        }

        Set<String> allLevelDomains = getDomainAllLevelDomains(url.getHost());
        if (SUBDOMAIN_PARSER_BY_DOMAIN.containsKey(domain)) {
            allLevelDomains.add(domain);
        }
        String mainDomain = StreamEx.of(allLevelDomains)
                .findFirst(SUBDOMAIN_PARSER_BY_DOMAIN::containsKey)
                .orElse(null);

        String subDomain = Optional.ofNullable(mainDomain)
                .map(d -> SUBDOMAIN_PARSER_BY_DOMAIN.get(d).apply(url))
                .map(AggregatorDomainsUtils::clearSubDomain)
                .filter(s -> s.length() >= MIN_SUBDOMAIN_LENGTH)
                .orElse(null);

        String result = null;
        if (!StringUtils.isEmpty(mainDomain) && !StringUtils.isEmpty(subDomain)) {
            String tmp = subDomain + "." + mainDomain;
            result = DOMAIN_PATTERN.matcher(tmp).matches() ? tmp : null;
        }
        return result;
    }

    @Nullable
    private static URL getURLObjectFromString(@Nullable String href) {
        if (href == null) {
            return null;
        }
        try {
            try {
                href = URLDecoder.decode(href, StandardCharsets.UTF_8.name());
            } catch (UnsupportedEncodingException e) {
                // not going to happen - value came from JDK's own StandardCharsets
            } catch (IllegalArgumentException e) {
                logger.warn("Bad href found {}", href);
            }
            return new URL(href);
        } catch (MalformedURLException e) {
            logger.error("URI build or parse error for value {}. Href must be already validated. Skip error.", href, e);
            return null;
        }
    }

    @Nullable
    private static String clearSubDomain(@Nullable String subDomain) {
        return SUBDOMAIN_MODIFIER.makeReplacements(subDomain);
    }

    private static Set<String> getDomainAllLevelDomains(String host) {
        if (StringUtils.isEmpty(host)) {
            return emptySet();
        }
        String[] tokens = host.split("\\.");

        Set<String> result = new HashSet<>();
        String currentDomain = tokens[tokens.length - 1];

        for (int i = tokens.length - 2; i >= 0; i--) {
            currentDomain = tokens[i] + "." + currentDomain;
            result.add(currentDomain);
        }
        return result;
    }
}
