package ru.yandex.direct.core.entity.domain;

import java.util.HashMap;
import java.util.Map;
import java.util.regex.Pattern;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.commons.lang3.StringUtils;
import org.apache.http.client.utils.URIBuilder;

@ParametersAreNonnullByDefault
public class DomainUtils {
    private static final String HTTP_PREFIX = "http://";
    private static final String HTTPS_PREFIX = "https://";

    private static final Pattern SOCIAL_NETWORKS_DOMAIN_PATTERN = Pattern.compile(
            "^("
                    + "(([\\w\\-]+\\.)*(ok|odnoklassniki|fb|vk|linkedin|facebook|instagram|twitter|youtube|addthis|livejournal|yandex|ya)\\.\\w+)"
                    + "|t\\.co|youtu\\.be|fb\\.me|vk\\.cc|vk\\.me|t\\.me|instagr\\.am"
                    + ")$");

    @Nullable
    public static String refineDomain(@Nullable String sourceDomain) {
        if (sourceDomain == null) {
            return null;
        }

        if (sourceDomain.startsWith(HTTP_PREFIX)) {
            return sourceDomain.substring(HTTP_PREFIX.length());
        }
        if (sourceDomain.startsWith(HTTPS_PREFIX)) {
            return sourceDomain.substring(HTTPS_PREFIX.length());
        }
        return sourceDomain;
    }

    /**
     * Получить мапу из query парамеров
     * Ключи приводятся к нижнему регистру
     * @param query
     * @return
     */
    public static Map<String, String> getQueryMap(String query) {
        String[] params = query.split("&");
        Map<String, String> map = new HashMap<>();
        for (String param : params) {
            String[] splitParam = param.split("=", 2);
            if (splitParam.length == 2) {
                String name = StringUtils.lowerCase(splitParam[0]);
                String value = splitParam[1];
                map.put(name, value);
            }
        }
        return map;
    }

    public static boolean isSocialNetworkSite(@Nullable String url) {
        if (url == null) {
            return false;
        }
        try {
            URIBuilder uriBuilder = new URIBuilder(url);
            return isSocialNetworkDomain(uriBuilder.getHost());
        } catch (Exception e) {
            return false;
        }
    }

    public static boolean isSocialNetworkDomain(@Nullable String domain) {
        if (domain == null) {
            return false;
        }
        return SOCIAL_NETWORKS_DOMAIN_PATTERN.matcher(domain).find();
    }
}
