package ru.yandex.direct.core.entity.domain.repository;

import java.util.Collection;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.DSLContext;
import org.jooq.InsertValuesStepN;
import org.jooq.util.mysql.MySQLDSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.dbschema.ppc.tables.records.AggregatorDomainsRecord;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;

import static java.util.Arrays.asList;
import static ru.yandex.direct.dbschema.ppc.Tables.AGGREGATOR_DOMAINS;

@Repository
@ParametersAreNonnullByDefault
public class AggregatorDomainsRepository {

    private final DslContextProvider dslContextProvider;

    @Autowired
    public AggregatorDomainsRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
    }

    /**
     * Получение aggregator_domain по списку id баннеров
     * Возвращает мапу: id баннера -> домен,
     * если для баннера нет записи в таблице, то в результирующей мапе его не будет
     * значения в мапе должны быть not null
     */
    public Map<Long, String> getAggregatorDomains(int shard, Collection<Long> bannerIds) {
        return getAggregatorDomains(dslContextProvider.ppc(shard), bannerIds);
    }

    public Map<Long, String> getAggregatorDomains(DSLContext context, Collection<Long> bannerIds) {
        return context
                .select(AGGREGATOR_DOMAINS.BID, AGGREGATOR_DOMAINS.PSEUDO_DOMAIN)
                .from(AGGREGATOR_DOMAINS)
                .where(AGGREGATOR_DOMAINS.BID.in(bannerIds))
                .fetchMap(AGGREGATOR_DOMAINS.BID, AGGREGATOR_DOMAINS.PSEUDO_DOMAIN);
    }

    /**
     * Обновление/добавление записей в aggregator_domains
     * Получает мапу: id баннера -> домен,
     * значения в мапе должны быть not null
     */
    public void updateAggregatorDomains(int shard, Map<Long, String> domainsByBannerIds) {
        updateAggregatorDomains(dslContextProvider.ppc(shard), domainsByBannerIds);
    }

    public void updateAggregatorDomains(DSLContext context, Map<Long, String> domainsByBannerIds) {
        if (domainsByBannerIds.isEmpty()) {
            return;
        }
        InsertValuesStepN<AggregatorDomainsRecord> insertValuesStep = context
                .insertInto(AGGREGATOR_DOMAINS,
                        asList(AGGREGATOR_DOMAINS.BID, AGGREGATOR_DOMAINS.PSEUDO_DOMAIN));
        for (Long bannerId : domainsByBannerIds.keySet()) {
            insertValuesStep = insertValuesStep.values(
                    bannerId, domainsByBannerIds.get(bannerId));
        }
        insertValuesStep.onDuplicateKeyUpdate()
                .set(AGGREGATOR_DOMAINS.PSEUDO_DOMAIN, MySQLDSL.values(AGGREGATOR_DOMAINS.PSEUDO_DOMAIN))
                .execute();
    }

    /**
     * Удаление записей из aggregator_domains по списку id баннеров
     */
    public void deleteAggregatorDomains(int shard, Collection<Long> bannerIds) {
        deleteAggregatorDomains(dslContextProvider.ppc(shard), bannerIds);
    }

    public void deleteAggregatorDomains(DSLContext context, Collection<Long> bannerIds) {
        if (bannerIds.isEmpty()) {
            return;
        }
        context.deleteFrom(AGGREGATOR_DOMAINS)
                .where(AGGREGATOR_DOMAINS.BID.in(bannerIds))
                .execute();
    }

}
