package ru.yandex.direct.core.entity.domain.repository;

import java.sql.ResultSet;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Set;

import com.google.common.collect.Iterables;
import com.google.common.collect.Sets;
import org.jooq.Cursor;
import org.jooq.DSLContext;
import org.jooq.InsertValuesStep1;
import org.jooq.InsertValuesStepN;
import org.jooq.Record2;
import org.jooq.TransactionalRunnable;
import org.jooq.util.mysql.MySQLDSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.common.util.AutoCloseableIterator;
import ru.yandex.direct.core.entity.domain.model.ApiDomainStat;
import ru.yandex.direct.core.entity.domain.model.Domain;
import ru.yandex.direct.core.entity.domain.model.FilterDomain;
import ru.yandex.direct.dbschema.ppc.tables.records.BsDeadDomainsRecord;
import ru.yandex.direct.dbschema.ppc.tables.records.FilterDomainRecord;
import ru.yandex.direct.dbschema.ppcdict.tables.records.DomainsDictRecord;
import ru.yandex.direct.dbutil.QueryWithoutIndex;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;

import static java.util.Arrays.asList;
import static java.util.stream.Collectors.toSet;
import static ru.yandex.direct.dbschema.ppc.tables.BsDeadDomains.BS_DEAD_DOMAINS;
import static ru.yandex.direct.dbschema.ppc.tables.Domains.DOMAINS;
import static ru.yandex.direct.dbschema.ppc.tables.FilterDomain.FILTER_DOMAIN;
import static ru.yandex.direct.dbschema.ppcdict.tables.ApiDomainStat.API_DOMAIN_STAT;
import static ru.yandex.direct.dbschema.ppcdict.tables.DomainsDict.DOMAINS_DICT;
import static ru.yandex.direct.dbschema.ppcdict.tables.Mirrors.MIRRORS;
import static ru.yandex.direct.dbschema.ppcdict.tables.MirrorsCorrection.MIRRORS_CORRECTION;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.misc.lang.StringUtils.reverse;

@Repository
public class DomainRepository {

    private DslContextProvider dslContextProvider;

    private final JooqMapperWithSupplier<Domain> domainMapper;
    private final JooqMapperWithSupplier<Domain> dictDomainMapper;
    private final JooqMapperWithSupplier<ApiDomainStat> apiDomainStatMapper;

    private static final int DOMAINS_COUNT_LIMIT = 1000;

    @Autowired
    public DomainRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
        this.domainMapper = createDomainMapper();
        this.dictDomainMapper = createDictDomainMapper();
        apiDomainStatMapper = createApiDomainStatMapper();
    }

    /**
     * Получение доменов по выбранному шарду
     *
     * @param shard   шард
     * @param domains коллекция доменов в строковом представлении
     * @return список моделей доменов
     */
    public List<Domain> getDomains(int shard, Collection<String> domains) {
        return getDomains(dslContextProvider.ppc(shard), domains);
    }

    /**
     * Получение доменов по domain_id по выбранному шарду
     *
     * @param shard   шард
     * @param domainIds коллекция доменов в строковом представлении
     * @return Карту доменов по domainId
     */
    public Map<Long, String> getDomainByIds(int shard, Collection<Long> domainIds) {
        return getDomainByIds(dslContextProvider.ppc(shard), domainIds);
    }

    /**
     * Получение существюущих ID доменов по выбранному шарду
     *
     * @param shard     шард
     * @param domainIds коллекция ID доменов
     * @return множество ID найденных доменов
     */
    public Set<Long> getExistingDomainIds(int shard, Collection<Long> domainIds) {
        return dslContextProvider.ppc(shard)
                .select(DOMAINS.DOMAIN_ID)
                .from(DOMAINS)
                .where(DOMAINS.DOMAIN_ID.in(domainIds))
                .fetchSet(DOMAINS.DOMAIN_ID);
    }

    /**
     * Получение существюущих ID потухших доменов по выбранному шарду
     *
     * @param shard шард
     * @return множество ID найденных потухших доменов
     */
    @QueryWithoutIndex("Полная выборка из таблицы")
    public Set<Long> getAllDeadDomainIds(int shard) {
        return dslContextProvider.ppc(shard)
                .select(BS_DEAD_DOMAINS.DOMAIN_ID)
                .from(BS_DEAD_DOMAINS)
                .fetchSet(BS_DEAD_DOMAINS.DOMAIN_ID);
    }

    public List<Domain> getDomains(DSLContext dslContext, Collection<String> domains) {
        return dslContext
                .select(domainMapper.getFieldsToRead())
                .from(DOMAINS)
                .where(DOMAINS.DOMAIN.in(domains))
                .fetch(domainMapper::fromDb);
    }

    public Map<Long, String> getDomainByIds(DSLContext dslContext, Collection<Long> domainIds) {
        return dslContext
                .select(DOMAINS.DOMAIN_ID, DOMAINS.DOMAIN)
                .from(DOMAINS)
                .where(DOMAINS.DOMAIN_ID.in(domainIds))
                .fetchMap(DOMAINS.DOMAIN_ID, DOMAINS.DOMAIN);
    }

    /**
     * Получение доменов из словаря {@code PPCDICT} по их ID.
     *
     * @param domainIds коллекция идентификаторов доменов
     */
    public List<Domain> getDomainsByIdsFromDict(Collection<Long> domainIds) {
        return dslContextProvider.ppcdict()
                .select(dictDomainMapper.getFieldsToRead())
                .from(DOMAINS_DICT)
                .where(DOMAINS_DICT.DOMAIN_ID.in(domainIds))
                .fetch(r -> {
                    Domain domain = dictDomainMapper.fromDb(r);
                    return domain.withReverseDomain(reverse(domain.getDomain()));
                });
    }

    /**
     * Получение доменов из словаря {@code PPCDICT} по их имени.
     *
     * @param domains коллекция доменов в строковом представлении
     * @return map домен-id моделей доменов
     */
    public Map<String, Long> getDomainsToIdsFromPpcDict(Collection<String> domains) {
        Map<String, Long> domainsToIdsResult = new HashMap<>();

        List<String> domainsSortedList = new ArrayList<>(domains);
        Collections.sort(domainsSortedList);

        for (List<String> chunk : Iterables.partition(domainsSortedList, DOMAINS_COUNT_LIMIT)) {
            Map<String, Long> domainsToIdsWithOffset =
                    dslContextProvider.ppcdict()
                            .select(DOMAINS_DICT.DOMAIN, DOMAINS_DICT.DOMAIN_ID)
                            .from(DOMAINS_DICT)
                            .where(DOMAINS_DICT.DOMAIN.in(chunk))
                            .fetchMap(DOMAINS_DICT.DOMAIN, DOMAINS_DICT.DOMAIN_ID);

            domainsToIdsResult.putAll(domainsToIdsWithOffset);
        }

        return domainsToIdsResult;
    }

    /**
     * Сохранение коллекции моделей домена:
     * существующие домены игнорируются;
     * если при сохранении записи не было в ppc_dict, сначала добавляется туда, затем сохраняется в текущем шарде
     * под тем же id
     *
     * @param domainModels коллекция доменов
     */
    public Map<String, Long> addDomains(DSLContext dslContext, Collection<Domain> domainModels) {
        Set<String> domains = listToSet(domainModels, Domain::getDomain);

        final Map<String, Long> existDomainsToIds = addDomainsToPpcDict(domains);
        addDomainsToShard(dslContext, domainModels, existDomainsToIds);
        return existDomainsToIds;
    }

    public void addDomainsToShard(
            DSLContext dslContext,
            Collection<Domain> domainModels,
            Map<String, Long> existDomainsToIds) {
        Set<String> domains = listToSet(domainModels, Domain::getDomain);
        Set<String> existDomainsOnShard = listToSet(getDomains(dslContext, domains), Domain::getDomain);
        Set<Domain> newDomainsOnShard = domainModels.stream()
                .filter(d -> !existDomainsOnShard.contains(d.getDomain()))
                .peek(d -> d.setId(existDomainsToIds.get(d.getDomain())))
                .collect(toSet());

        addToPpcDomainsTable(dslContext, newDomainsOnShard);
    }

    public Map<String, Long> addDomains(int shard, Collection<Domain> domainModels) {
        return addDomains(dslContextProvider.ppc(shard), domainModels);
    }

    /**
     * Добавление доменов в ppc_dict
     * Сначала делается сравнение с существующими доменами и затем отсутствующие добавляются
     *
     * @param domains коллекция доменов в строковом представлении
     * @return добавленные и уже существующие домены в одном map
     */
    public Map<String, Long> addDomainsToPpcDict(Collection<String> domains) {
        final Map<String, Long> existDomainsToIds = getDomainsToIdsFromPpcDict(domains);

        if (domains.size() > existDomainsToIds.size()) {
            Set<String> newDomainsOnPpcDict = domains.stream()
                    .filter(d -> !existDomainsToIds.containsKey(d))
                    .collect(toSet());

            Map<String, Long> insertedDomainsToIds = addToPpcDict(newDomainsOnPpcDict);
            existDomainsToIds.putAll(insertedDomainsToIds);
        }
        return existDomainsToIds;
    }

    private Map<String, Long> addToPpcDict(Collection<String> domains) {
        InsertValuesStep1<DomainsDictRecord, String> insert = dslContextProvider.ppcdict()
                .insertInto(DOMAINS_DICT, DOMAINS_DICT.DOMAIN);

        domains.forEach(insert::values);

        return insert.onDuplicateKeyIgnore()
                .returning(DOMAINS_DICT.DOMAIN, DOMAINS_DICT.DOMAIN_ID)
                .fetch()
                .intoMap(DOMAINS_DICT.DOMAIN, DOMAINS_DICT.DOMAIN_ID);
    }

    public void refreshDeadDomains(int shard, List<Domain> deadDomains) {
        Set<Long> deadDomainIds = listToSet(deadDomains, Domain::getId);
        Set<Long> existingDomainIds = getExistingDomainIds(shard, deadDomainIds);
        List<Domain> domainsToAdd = filterList(deadDomains, d -> !existingDomainIds.contains(d.getId()));
        addToPpcDomainsTable(shard, domainsToAdd);
        // на случай дублирования домена в ppcdict.domains_dict и ppc.domains с разными id
        deadDomainIds = listToSet(getDomains(shard, listToSet(deadDomains, Domain::getDomain)), Domain::getId);

        Set<Long> allDeadDomainIds = getAllDeadDomainIds(shard);
        Set<Long> deadDomainIdsToDelete = Sets.difference(allDeadDomainIds, deadDomainIds);
        Set<Long> deadDomainIdsToInsert = Sets.difference(deadDomainIds, allDeadDomainIds);

        TransactionalRunnable tr = conf -> {
            InsertValuesStep1<BsDeadDomainsRecord, Long> insert = conf.dsl()
                    .insertInto(BS_DEAD_DOMAINS, BS_DEAD_DOMAINS.DOMAIN_ID);
            deadDomainIdsToInsert.forEach(insert::values);
            insert.onDuplicateKeyUpdate().set(BS_DEAD_DOMAINS.LAST_CHANGE, LocalDateTime.now()).execute();

            conf.dsl()
                    .delete(BS_DEAD_DOMAINS)
                    .where(BS_DEAD_DOMAINS.DOMAIN_ID.in(deadDomainIdsToDelete))
                    .execute();
        };
        dslContextProvider.ppcTransaction(shard, tr);
    }

    private int addToPpcDomainsTable(int shard, Collection<Domain> domains) {
        return addToPpcDomainsTable(dslContextProvider.ppc(shard), domains);
    }

    private int addToPpcDomainsTable(DSLContext dslContext, Collection<Domain> domains) {
        return new InsertHelper<>(dslContext, DOMAINS)
                .addAll(domainMapper, domains)
                .onDuplicateKeyIgnore()
                .executeIfRecordsAdded();
    }


    private JooqMapperWithSupplier<Domain> createDomainMapper() {
        return JooqMapperWithSupplierBuilder.builder(Domain::new)
                .map(property(Domain.ID, DOMAINS.DOMAIN_ID))
                .map(property(Domain.DOMAIN, DOMAINS.DOMAIN))
                .map(property(Domain.REVERSE_DOMAIN, DOMAINS.REVERSE_DOMAIN))
                .build();
    }

    private JooqMapperWithSupplier<Domain> createDictDomainMapper() {
        return JooqMapperWithSupplierBuilder.builder(Domain::new)
                .map(property(Domain.ID, DOMAINS_DICT.DOMAIN_ID))
                .map(property(Domain.DOMAIN, DOMAINS_DICT.DOMAIN))
                .build();
    }

    private JooqMapperWithSupplier<ApiDomainStat> createApiDomainStatMapper() {
        return JooqMapperWithSupplierBuilder.builder(ApiDomainStat::new)
                .map(property(ApiDomainStat.FILTER_DOMAIN, API_DOMAIN_STAT.FILTER_DOMAIN))
                .map(property(ApiDomainStat.STAT_DATE, API_DOMAIN_STAT.STAT_DATE))
                .map(property(ApiDomainStat.ACCEPTED_ITEMS, API_DOMAIN_STAT.ACCEPTED_ITEMS))
                .map(property(ApiDomainStat.DECLINED_ITEMS, API_DOMAIN_STAT.DECLINED_ITEMS))
                .map(property(ApiDomainStat.BAD_REASONS, API_DOMAIN_STAT.BAD_REASONS))
                .map(property(ApiDomainStat.DECLINED_PHRASES, API_DOMAIN_STAT.DECLINED_PHRASES))
                .build();
    }

    /**
     * Получение доменов c api_domain_stat.show_approx > 0
     *
     * @param domains коллекция доменов в строковом представлении
     * @return множество доменов со статистикой
     */
    public Set<String> getDomainsWithStat(Collection<String> domains) {
        return dslContextProvider.ppcdict()
                .selectDistinct(API_DOMAIN_STAT.FILTER_DOMAIN)
                .from(API_DOMAIN_STAT)
                .where(API_DOMAIN_STAT.FILTER_DOMAIN.in(domains)
                        .and(API_DOMAIN_STAT.SHOWS_APPROX.greaterThan(0L)))
                .fetchSet(API_DOMAIN_STAT.FILTER_DOMAIN);
    }

    /**
     * Получение статистики по доменам.
     * @param domains коллекция доменов, по которым запрашивается статистика
     * @return статистика по доменам
     */
    public List<ApiDomainStat> getDomainsStat(Collection<String> domains) {
        return dslContextProvider.ppcdict()
                .select(apiDomainStatMapper.getFieldsToRead())
                .from(API_DOMAIN_STAT)
                .where(API_DOMAIN_STAT.FILTER_DOMAIN.in(domains))
                .fetch(apiDomainStatMapper::fromDb);
    }

    /**
     * Обновление статитстики по доменам.
     * @param apiDomainStatData стат
     */
    public void updateDomainsStat(Collection<ApiDomainStat> apiDomainStatData) {
        if (apiDomainStatData.isEmpty()) {
            return;
        }

        var today = LocalDate.now();
        var statData = mapList(apiDomainStatData, item -> item.withStatDate(today));
        new InsertHelper<>(dslContextProvider.ppcdict(), API_DOMAIN_STAT)
                .addAll(apiDomainStatMapper, statData)
                .onDuplicateKeyUpdate()
                .set(API_DOMAIN_STAT.ACCEPTED_ITEMS,
                        API_DOMAIN_STAT.ACCEPTED_ITEMS.add(MySQLDSL.values(API_DOMAIN_STAT.ACCEPTED_ITEMS)))
                .set(API_DOMAIN_STAT.DECLINED_ITEMS,
                        API_DOMAIN_STAT.DECLINED_ITEMS.add(MySQLDSL.values(API_DOMAIN_STAT.DECLINED_ITEMS)))
                .set(API_DOMAIN_STAT.BAD_REASONS,
                        API_DOMAIN_STAT.BAD_REASONS.add(MySQLDSL.values(API_DOMAIN_STAT.BAD_REASONS)))
                .set(API_DOMAIN_STAT.DECLINED_PHRASES,
                        API_DOMAIN_STAT.DECLINED_PHRASES.add(MySQLDSL.values(API_DOMAIN_STAT.DECLINED_PHRASES)))
                .execute();
    }

    /**
     * Получение главного зеркала по домену
     * Сначала проверяем домен в таблице ppcdict.mirrors_correction, затем в ppcdict.mirrors
     * Если зеркало для домена не найдено, то домен в мапу не попадает
     *
     * @param domains коллекция доменов в строковом представлении
     * @return мапа домен -> зеркало
     */
    public Map<String, String> getMainMirrors(Collection<String> domains) {
        Map<String, String> mirrorsCorrection = getMirrorsCorrection(domains);
        Collection<String> domainsWithoutCorrections = domains.stream().filter(d ->
                !mirrorsCorrection.containsKey(d)).collect(toSet());
        if (domainsWithoutCorrections.isEmpty()) {
            return mirrorsCorrection;
        }
        Map<String, String> mirrors = getMirrors(domainsWithoutCorrections);
        mirrors.putAll(mirrorsCorrection);
        return mirrors;
    }

    private Map<String, String> getMirrors(Collection<String> domains) {
        return dslContextProvider.ppcdict()
                .select(MIRRORS.DOMAIN, MIRRORS.MIRROR)
                .from(MIRRORS)
                .where(MIRRORS.DOMAIN.in(domains))
                .fetchMap(MIRRORS.DOMAIN, MIRRORS.MIRROR);
    }

    private Map<String, String> getMirrorsCorrection(Collection<String> domains) {
        return dslContextProvider.ppcdict()
                .select(MIRRORS_CORRECTION.DOMAIN, MIRRORS_CORRECTION.REDIRECT_DOMAIN)
                .from(MIRRORS_CORRECTION)
                .where(MIRRORS_CORRECTION.DOMAIN.in(domains))
                .fetchMap(MIRRORS_CORRECTION.DOMAIN, MIRRORS_CORRECTION.REDIRECT_DOMAIN);
    }

    /**
     * Обновление/добавление записей в FilterDomain
     * Если у домена изменилось зеркало - перезаписываем
     *
     * @param dslContext    контекст
     * @param filterDomains список FilterDomain
     */
    public void updateFilterDomain(DSLContext dslContext, Iterable<FilterDomain> filterDomains) {
        InsertValuesStepN<FilterDomainRecord> insertIntoFilterDomen = dslContext
                .insertInto(FILTER_DOMAIN, asList(FILTER_DOMAIN.DOMAIN, FILTER_DOMAIN.FILTER_DOMAIN_));
        for (FilterDomain filterDomain : filterDomains) {
            insertIntoFilterDomen = insertIntoFilterDomen.values(
                    filterDomain.getDomain(), filterDomain.getFilterDomain());
        }
        insertIntoFilterDomen.onDuplicateKeyUpdate()
                .set(FILTER_DOMAIN.FILTER_DOMAIN_, MySQLDSL.values(FILTER_DOMAIN.FILTER_DOMAIN_))
                .execute();
    }

    public AutoCloseableIterator<List<FilterDomain>> allFilterDomains(int shard, int chunkSize) {
        Cursor<Record2<String, String>> cursor = dslContextProvider.ppc(shard)
                .select(FILTER_DOMAIN.DOMAIN, FILTER_DOMAIN.FILTER_DOMAIN_)
                .from(FILTER_DOMAIN)
                .resultSetType(ResultSet.TYPE_FORWARD_ONLY)
                .fetchSize(chunkSize) //Подсказка драйверу MySQL, сколько записей фетчить за раз.
                .fetchLazy();

        return new AutoCloseableIterator<List<FilterDomain>>() {
            @Override
            public void close() {
                cursor.close();
            }

            @Override
            public boolean hasNext() {
                return cursor.hasNext();
            }

            @Override
            public List<FilterDomain> next() {
                if (!cursor.hasNext()) {
                    throw new NoSuchElementException("Underlying cursor doesn't have more elements");
                }

                List<FilterDomain> chunk = new ArrayList<>(chunkSize);

                int i = 0;
                while (i++ < chunkSize && cursor.hasNext()) {
                    Record2<String, String> next = cursor.fetchNext();
                    chunk.add(new FilterDomain()
                            .withDomain(next.component1())
                            .withFilterDomain(next.component2()));
                }

                if (!cursor.hasNext()) {
                    cursor.close();
                }

                return chunk;
            }
        };
    }
}
