package ru.yandex.direct.core.entity.domain.repository;

import java.util.Collection;
import java.util.Map;

import one.util.streamex.StreamEx;
import org.jooq.Record;
import org.jooq.Result;
import org.jooq.util.mysql.MySQLDSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.domain.model.MarketRating;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;

import static java.util.Arrays.asList;
import static ru.yandex.direct.dbschema.ppcdict.tables.DomainsDict.DOMAINS_DICT;
import static ru.yandex.direct.dbschema.ppcdict.tables.MarketRatings.MARKET_RATINGS;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;

@Repository
public class MarketRatingRepository {

    public static final long RATING_NOT_SPECIFIED = -1L;
    private final DslContextProvider dslContextProvider;

    private final JooqMapperWithSupplier<MarketRating> marketRatingMapper;

    @Autowired
    public MarketRatingRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
        this.marketRatingMapper = createMarketRatingMapper();
    }

    private JooqMapperWithSupplier<MarketRating> createMarketRatingMapper() {
        return JooqMapperWithSupplierBuilder.builder(MarketRating::new)
                .map(property(MarketRating.DOMAIN_ID, MARKET_RATINGS.DOMAIN_ID))
                .map(property(MarketRating.RATING, MARKET_RATINGS.RATING))
                .map(property(MarketRating.LAST_CHANGE, MARKET_RATINGS.LAST_CHANGE))
                .build();
    }

    /**
     * Получение актуальных рейтингов (rating <> -1)
     * Возвращается map: код домена - рейтинг
     */
    public Map<Long, MarketRating> getActual() {
        return dslContextProvider.ppcdict()
                .select(marketRatingMapper.getFieldsToRead())
                .from(MARKET_RATINGS)
                .where(MARKET_RATINGS.RATING.notEqual(RATING_NOT_SPECIFIED))
                .fetchMap(MARKET_RATINGS.DOMAIN_ID, marketRatingMapper::fromDb);
    }

    /**
     * Получение всех рейтингов
     * Возвращается map: код домена - объект класса MarketRating
     * <b>Внимание:</b> этот метод используется только для тестов.
     */
    public Map<Long, MarketRating> getAll() {
        return dslContextProvider.ppcdict()
                .select(marketRatingMapper.getFieldsToRead())
                .from(MARKET_RATINGS)
                .fetchMap(MARKET_RATINGS.DOMAIN_ID, marketRatingMapper::fromDb);
    }

    /**
     * Получение всех рейтингов по имени
     * Возвращается map: имя домена - рейтинг
     * <b>Внимание:</b> этот метод используется только для тестов.
     */
    public Map<String, Long> getAllByName() {
        Result<Record> result = dslContextProvider.ppcdict()
                .select(asList(DOMAINS_DICT.DOMAIN, MARKET_RATINGS.RATING))
                .from(MARKET_RATINGS)
                .join(DOMAINS_DICT).on(DOMAINS_DICT.DOMAIN_ID.eq(MARKET_RATINGS.DOMAIN_ID))
                .fetch();

        return StreamEx.of(result).toMap(
                rec -> rec.getValue(DOMAINS_DICT.DOMAIN),
                rec -> rec.getValue(MARKET_RATINGS.RATING));

    }

    /**
     * Обновление рейтингов
     * Добавляются новые записи или, при наличии, изменяется рейтинг
     */
    public int addAll(Collection<MarketRating> ratings) {

        if (ratings.isEmpty()) {
            return 0;
        }

        return new InsertHelper<>(dslContextProvider.ppcdict(), MARKET_RATINGS)
                .addAll(marketRatingMapper, ratings)
                .onDuplicateKeyUpdate()
                .set(MARKET_RATINGS.RATING, MySQLDSL.values(MARKET_RATINGS.RATING))
                .set(MARKET_RATINGS.STATUS_BS_SYNCED, MySQLDSL.values(MARKET_RATINGS.STATUS_BS_SYNCED))
                .set(MARKET_RATINGS.LAST_CHANGE, MySQLDSL.values(MARKET_RATINGS.LAST_CHANGE))
                .execute();
    }
}
