package ru.yandex.direct.core.entity.domain.service;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;

import org.jooq.Configuration;
import org.jooq.DSLContext;
import org.jooq.impl.DSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.banner.model.Banner;
import ru.yandex.direct.core.entity.banner.model.BannerWithAggregatorDomain;
import ru.yandex.direct.core.entity.domain.repository.AggregatorDomainsRepository;
import ru.yandex.direct.model.AppliedChanges;

import static ru.yandex.direct.core.entity.domain.AggregatorDomainsUtils.extractAggregatorDomainFromHref;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * По href расклеивает домены-агрегаторы
 * "http://vk.com/test" -> "test.vk.com"
 * нужно для кластеризации в БК, чтобы можно было показывать одновременно рекламу двух разных рекламодателей,
 * если они размещаются на одном агрегаторе
 */
@Service
public class AggregatorDomainsService {
    private final AggregatorDomainsRepository aggregatorDomainsRepository;

    @Autowired
    public AggregatorDomainsService(AggregatorDomainsRepository aggregatorDomainsRepository) {
        this.aggregatorDomainsRepository = aggregatorDomainsRepository;
    }

    /**
     * Для переданных баннеров вычислить значение {@code aggregator_domain} и обновить его в БД.
     *
     * @param shard   номер шарда
     * @param banners баннеры. {@link BannerWithAggregatorDomain#getHref()} может быть {@code null}
     * @param <B>     тип баннера
     */
    public <B extends BannerWithAggregatorDomain> void updateAggregatorDomains(int shard, Collection<B> banners) {
        UpdateOrDeleteContainer<B> container = new UpdateOrDeleteContainer<>(banners);
        aggregatorDomainsRepository.updateAggregatorDomains(shard, container.getAggregationDomainByIdsToUpdate());
        aggregatorDomainsRepository.deleteAggregatorDomains(shard, container.getIdsToDelete());
    }

    /**
     * Для переданных баннеров вычислить значение {@code aggregator_domain} и обновить его в БД.
     *
     * @param dslContext контекст, в рамках которого выполняется запрос к БД
     * @param banners    баннеры. {@link BannerWithAggregatorDomain#getHref()} может быть {@code null}
     * @param <B>        тип баннера
     */
    public <B extends BannerWithAggregatorDomain> void updateAggregatorDomains(DSLContext dslContext,
                                                                               Collection<B> banners) {
        UpdateOrDeleteContainer<B> container = new UpdateOrDeleteContainer<>(banners);
        aggregatorDomainsRepository.updateAggregatorDomains(dslContext, container.getAggregationDomainByIdsToUpdate());
        aggregatorDomainsRepository.deleteAggregatorDomains(dslContext, container.getIdsToDelete());
    }

    /**
     * Для переданных баннеров вычислить значение @{code aggregator_domain} и обновить его в БД.
     *
     * @param conf           контекст транзакции, в рамках которой выполняется запрос к БД
     * @param appliedChanges изменения баннера. {@link BannerWithAggregatorDomain#getHref()} может быть {@code null}
     * @param <B>            тип баннера
     */
    public <B extends BannerWithAggregatorDomain> void updateAggregatorDomains(Configuration conf,
                                                                               Collection<AppliedChanges<B>> appliedChanges) {
        List<B> banners = mapList(appliedChanges, AppliedChanges::getModel);
        DSLContext dslContext = DSL.using(conf);
        Map<Long, String> aggregatorDomainsById = getAggregatorDomainsFromDb(dslContext, banners);
        UpdateOrDeleteContainer<B> container = new UpdateOrDeleteContainer<>(aggregatorDomainsById, banners);
        aggregatorDomainsRepository.updateAggregatorDomains(dslContext, container.getAggregationDomainByIdsToUpdate());
        aggregatorDomainsRepository.deleteAggregatorDomains(dslContext, container.getIdsToDelete());
    }

    private <B extends BannerWithAggregatorDomain> Map<Long, String> getAggregatorDomainsFromDb(DSLContext dslContext, List<B> banners) {
        List<Long> bannerIds = mapList(banners, Banner::getId);
        return aggregatorDomainsRepository.getAggregatorDomains(dslContext, bannerIds);
    }

    private static class UpdateOrDeleteContainer<B extends BannerWithAggregatorDomain> {
        private final Map<Long, String> aggregationDomainByIdsToUpdate = new HashMap<>();
        private final List<Long> idsToDelete = new ArrayList<>();

        UpdateOrDeleteContainer(Collection<B> banners) {
            this(getAggregatorDomainsById(banners), banners);
        }

        UpdateOrDeleteContainer(Map<Long, String> aggregatorDomainsById, Collection<B> banners) {
            for (B banner : banners) {
                Long bannerId = banner.getId();
                String oldDomain = aggregatorDomainsById.get(bannerId);
                String newDomain = extractAggregatorDomainFromHref(banner.getDomain(), banner.getHref());
                if (!Objects.equals(oldDomain, newDomain)) {
                    if (newDomain == null) {
                        idsToDelete.add(bannerId);
                    } else {
                        aggregationDomainByIdsToUpdate.put(bannerId, newDomain);
                    }
                }
            }
        }

        Map<Long, String> getAggregationDomainByIdsToUpdate() {
            return aggregationDomainByIdsToUpdate;
        }

        List<Long> getIdsToDelete() {
            return idsToDelete;
        }

        private static <B extends BannerWithAggregatorDomain> Map<Long, String> getAggregatorDomainsById(Collection<B> banners) {
            Map<Long, String> aggregatorDomainsById = new HashMap<>();
            for (B banner : banners) {
                if (banner.getAggregatorDomain() != null) {
                    aggregatorDomainsById.put(banner.getId(), banner.getAggregatorDomain());
                }
            }
            return aggregatorDomainsById;
        }
    }
}
