package ru.yandex.direct.core.entity.domain.service;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Predicate;

import com.google.common.base.Preconditions;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.jooq.DSLContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.util.AutoCloseableIterator;
import ru.yandex.direct.core.entity.banner.model.old.OldBanner;
import ru.yandex.direct.core.entity.domain.model.Domain;
import ru.yandex.direct.core.entity.domain.model.FilterDomain;
import ru.yandex.direct.core.entity.domain.model.MarketRating;
import ru.yandex.direct.core.entity.domain.repository.DomainRepository;
import ru.yandex.direct.core.entity.domain.repository.MarketRatingRepository;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.model.AppliedChanges;

import static com.google.common.base.Preconditions.checkState;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;
import static java.util.stream.Collectors.toSet;
import static org.apache.commons.collections4.MapUtils.invertMap;
import static org.apache.commons.lang3.StringUtils.reverse;
import static ru.yandex.direct.core.entity.domain.repository.MarketRatingRepository.RATING_NOT_SPECIFIED;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.utils.FunctionalUtils.mapSet;

@Service
public class DomainService {
    private static final Logger logger = LoggerFactory.getLogger(DomainService.class);
    private final MarketRatingRepository marketRatingRepository;
    private final DomainRepository domainRepository;
    private final ShardHelper shardHelper;
    private final DslContextProvider dslContextProvider;

    @Autowired
    public DomainService(MarketRatingRepository marketRatingRepository,
                         DomainRepository domainRepository,
                         ShardHelper shardHelper,
                         DslContextProvider dslContextProvider) {
        this.marketRatingRepository = marketRatingRepository;
        this.domainRepository = domainRepository;
        this.shardHelper = shardHelper;
        this.dslContextProvider = dslContextProvider;
    }

    /**
     * Обновляет таблицу ppcdict.market_ratings актуальными рейтингами
     * Возвращает количество измененных записей
     */
    public int updateMarketDomainRatings(Map<String, Long> domainsToRatings) {
        // Добавляем в таблицу ppcdict.domains_dict отсутствующие домены
        // Получаем map имя домена-идентификатор
        domainRepository.addDomainsToPpcDict(domainsToRatings.keySet());
        Map<String, Long> domains = domainRepository.getDomainsToIdsFromPpcDict(domainsToRatings.keySet());

        // Загружаем все текущие рейтинги, где рейтинг не равен -1
        Map<Long, MarketRating> oldRatings = marketRatingRepository.getActual();

        // Перебираем рейтинги, сравнивая с текущими, заполняем список ratingsForUpdate в случае расхождения
        // Из списка oldRatings значение удаляем
        List<MarketRating> ratingsForUpdate = new ArrayList<>();
        for (Map.Entry<String, Long> rating : domainsToRatings.entrySet()) {
            String domain = rating.getKey();
            Long id = domains.get(domain);
            Long newRatingValue = rating.getValue();
            MarketRating oldRating = oldRatings.remove(id);

            Long oldRatingValue = oldRating == null ? null : oldRating.getRating();
            if (!Objects.equals(oldRatingValue, newRatingValue)) {
                ratingsForUpdate.add(new MarketRating().withDomainId(id).withRating(newRatingValue)
                        .withLastChange(LocalDateTime.now())
                );

                logger.info("Updated rating for domain {}, id: {}, old rating: {}, new rating: {}", domain, id,
                        oldRatingValue, newRatingValue);
            }
        }

        // Для отсутствующих доменов (которые не были удалены в oldRatings) проставляем рейтинг -1
        // и добавляем в список ratingsForUpdate для изменения
        Map<Long, String> oldDomains = domainRepository.getDomainsByIdsFromDict(oldRatings.keySet()).stream()
                .collect(toMap(Domain::getId, Domain::getDomain));
        for (MarketRating mr : oldRatings.values()) {
            logger.info("Updated rating for domain {}, id: {}, old rating: {}, new rating: {}",
                    oldDomains.get(mr.getDomainId()), mr.getDomainId(), mr.getRating(), RATING_NOT_SPECIFIED);
            mr.setRating(RATING_NOT_SPECIFIED);
            mr.setLastChange(LocalDateTime.now());
            ratingsForUpdate.add(mr);
        }

        // Обновляем рейтинги и возвращаем количество измененных записей
        return marketRatingRepository.addAll(ratingsForUpdate);
    }

    /**
     * Получение доменов из словаря {@code PPCDICT} по их ID.
     *
     * @param domainIds коллекция идентификаторов доменов
     */
    public List<Domain> getDomainsByIdsFromDict(Collection<Long> domainIds) {
        if (domainIds.isEmpty()) {
            return Collections.emptyList();
        }
        return domainRepository.getDomainsByIdsFromDict(domainIds);
    }

    public Map<Long, String> getDomainByIdFromPpc(int shard, Collection<Long> domainIds) {
        if (domainIds.isEmpty()) {
            return Collections.emptyMap();
        }
        return domainRepository.getDomainByIds(shard, domainIds);
    }

    public List<Domain> getDomainsByDomainFromPpc(int shard, Collection<String> domains) {
        if (domains.isEmpty()) {
            return Collections.emptyList();
        }
        return domainRepository.getDomains(shard, domains);
    }

    public Map<String, Long> addDomains(DSLContext dslContext, Collection<String> domains) {
        Set<Domain> newDomainModels = domains.stream().map(this::toDomainModel).collect(toSet());
        return domainRepository.addDomains(dslContext, newDomainModels);
    }

    public Map<String, Long> addDomains(int shard, Collection<String> domains) {
        return addDomains(dslContextProvider.ppc(shard), domains);
    }

    private Domain toDomainModel(String domain) {
        return new Domain().withDomain(domain).withReverseDomain(reverse(domain));
    }

    public void copyDomainsBetweenShards(int sourceShard, int targetShard, List<Long> domainIds) {
        Map<String, Long> existDomainsToIds = invertMap(domainRepository.getDomainByIds(sourceShard, domainIds));
        Set<Domain> newDomainModels = mapSet(existDomainsToIds.keySet(), this::toDomainModel);
        domainRepository.addDomainsToShard(dslContextProvider.ppc(targetShard), newDomainModels, existDomainsToIds);
    }

    /**
     * Добавить несуществующие в базе домены, и получить для существующих и добавленных идентификаторы
     * <p>
     * Внимание! В доменах, перед добавлением (и перед поиском существующих), удаляются краевые пробелы с обоих сторон.
     *
     * @param domainUrls список доменов для добавления и получения идентификаторов
     * @return список идентификаторов в порядке, соотвествующем порядку в domainUrls
     */
    public List<Long> getOrCreate(DSLContext dslContext, List<String> domainUrls) {
        List<String> trimmedDomainUrls = domainUrls.stream()
                .map(String::trim)
                .map(String::toLowerCase)
                .collect(toList());
        Preconditions.checkArgument(trimmedDomainUrls.stream().noneMatch(String::isEmpty));
        Map<String, Long> domainToId = addDomains(dslContext, trimmedDomainUrls);

        return StreamEx.of(trimmedDomainUrls)
                .map(domainToId::get)
                .peek(id -> checkState(id != null))
                .toList();
    }

    public Map<String, Long> getOrCreateDomainIdByDomain(int shard, Collection<String> domains) {
        List<String> orderedDomains = new ArrayList<>(domains);
        List<Long> domainIds = getOrCreate(dslContextProvider.ppc(shard), orderedDomains);
        Map<String, Long> result = new HashMap<>();
        for (int i = 0; i < domainIds.size(); i++) {
            result.put(orderedDomains.get(i), domainIds.get(i));
        }
        return result;
    }

    /**
     * Вычисление главного зеркала домена и запись его в {@code ppc.filter_domain}.
     * Сначала проверяем домен в таблице ppcdict.mirrors_correction, затем в {@code ppcdict.mirrors}.
     * Если зеркала нет, то запись в {@code ppc.filter_domain} не создаём.
     */
    public void updateFilterDomain(int shard, Collection<String> domains) {
        updateFilterDomain(dslContextProvider.ppc(shard), domains);
    }

    public void updateFilterDomain(DSLContext dslContext, Collection<String> domains) {
        Map<String, String> domainToMirror = domainRepository.getMainMirrors(domains);
        Set<FilterDomain> filterDomains = EntryStream.of(domainToMirror).mapKeyValue(this::createFilterDomain).toSet();
        if (!filterDomains.isEmpty()) {
            domainRepository.updateFilterDomain(dslContext, filterDomains);
        }
    }

    /**
     * Возвращает итератор, который читает чанки записей.
     */
    public AutoCloseableIterator<List<FilterDomain>> allFilterDomains(int shard, int chunkSize) {
        return domainRepository.allFilterDomains(shard, chunkSize);
    }

    private FilterDomain createFilterDomain(String domain, String filterDomain) {
        return new FilterDomain().withDomain(domain).withFilterDomain(filterDomain);
    }

    /**
     * Полностью обновляет таблицу недоступных доменов
     * <p>
     * 1. Добавляет переданные домены в ppcdict.domains_dict
     * 2. Добавляет переданные домены ppc.domains на всех шардах
     * 3. Добавляет переданные домены в ppc.bs_dead_domains на всех шардах
     * 4. Удаляет из ppc.bs_dead_domains на всех шардах все домены, кроме переданных
     *
     * @param domains недоступные домены для обновления
     * @return отображение домен -> ID домена для списка переданных доменов
     */
    public Map<String, Long> refreshDeadDomains(Set<String> domains) {
        Map<String, Long> domainToIdMap = domainRepository.addDomainsToPpcDict(domains);
        List<Domain> domainList = EntryStream.of(domainToIdMap)
                .mapKeyValue((domain, domainId) -> new Domain()
                        .withDomain(domain)
                        .withId(domainId)
                        .withReverseDomain(reverse(domain)))
                .collect(toList());
        shardHelper.forEachShard(shard -> domainRepository.refreshDeadDomains(shard, domainList));
        return domainToIdMap;
    }

    /**
     * Передать домены баннеров в {@link DomainRepository} для создания и проставить им ID.
     *
     * @param shard   шард
     * @param banners список баннеров с доменами или без них
     */
    public <B extends OldBanner> void addDomainsAndSetDomainIds(int shard, Collection<B> banners) {
        List<B> bannersWithDomain = banners.stream().filter(b -> b.getDomain() != null).collect(toList());

        Map<String, Long> domainModels = addDomains(shard, mapList(bannersWithDomain, OldBanner::getDomain));

        bannersWithDomain.forEach(b -> b.setDomainId(domainModels.get(b.getDomain())));
    }

    /**
     * Для всех записей, в которых изменился домен добавить в таблицу доменов (если ещё не сохранён)
     * и проставить новый domainId.
     *
     * @param shard          шард
     * @param appliedChanges изменения баннеров
     */
    public <B extends OldBanner> void addNewDomainsAndUpdateDomainIds(int shard,
                                                                      Collection<AppliedChanges<B>> appliedChanges) {
        Predicate<AppliedChanges<B>> domainChangedAndNotNull = ac -> ac.changedAndNotDeleted(B.DOMAIN);

        List<String> domains = appliedChanges.stream()
                .filter(domainChangedAndNotNull)
                .map(item -> item.getNewValue(B.DOMAIN))
                .collect(toList());

        Map<String, Long> domainModels = addDomains(shard, domains);

        appliedChanges.stream()
                .filter(domainChangedAndNotNull)
                .forEach(ac -> ac.modify(B.DOMAIN_ID, domainModels.get(ac.getModel().getDomain())));
    }
}
