package ru.yandex.direct.core.entity.dynamictextadtarget.repository;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableMap;
import one.util.streamex.StreamEx;
import org.jooq.Condition;
import org.jooq.DSLContext;
import org.jooq.Field;
import org.jooq.Record;
import org.jooq.Record2;
import org.jooq.SelectConditionStep;
import org.jooq.SelectJoinStep;
import org.jooq.SelectOnConditionStep;
import org.jooq.impl.DSL;
import org.jooq.types.ULong;
import org.jooq.types.UNumber;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.common.util.RepositoryUtils;
import ru.yandex.direct.core.entity.dynamictextadtarget.container.DynamicTextAdTargetSelectionCriteria;
import ru.yandex.direct.core.entity.dynamictextadtarget.model.DynamicAdTarget;
import ru.yandex.direct.core.entity.dynamictextadtarget.model.DynamicAdTargetBaseStatus;
import ru.yandex.direct.core.entity.dynamictextadtarget.model.DynamicAdTargetTab;
import ru.yandex.direct.core.entity.dynamictextadtarget.model.DynamicAdTargetsQueryFilter;
import ru.yandex.direct.core.entity.dynamictextadtarget.model.DynamicFeedAdTarget;
import ru.yandex.direct.core.entity.dynamictextadtarget.model.DynamicFeedRule;
import ru.yandex.direct.core.entity.dynamictextadtarget.model.DynamicTextAdTarget;
import ru.yandex.direct.core.entity.dynamictextadtarget.model.DynamicTextAdTargetState;
import ru.yandex.direct.core.entity.dynamictextadtarget.utils.DynamicTextAdTargetHashUtils;
import ru.yandex.direct.core.entity.feed.model.BusinessType;
import ru.yandex.direct.core.entity.feed.model.FeedType;
import ru.yandex.direct.core.entity.performancefilter.schema.FilterSchema;
import ru.yandex.direct.core.entity.performancefilter.service.PerformanceFilterConditionDBFormatParser;
import ru.yandex.direct.core.entity.performancefilter.service.PerformanceFilterStorage;
import ru.yandex.direct.dbschema.ppc.enums.BidsDynamicFromTab;
import ru.yandex.direct.dbschema.ppc.tables.records.BidsDynamicRecord;
import ru.yandex.direct.dbutil.model.BusinessIdAndShopId;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapper;
import ru.yandex.direct.jooqmapper.JooqMapperBuilder;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;
import ru.yandex.direct.jooqmapperhelper.JooqUpdateBuilder;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.multitype.entity.LimitOffset;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkState;
import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static org.apache.commons.collections4.CollectionUtils.isNotEmpty;
import static org.jooq.impl.DSL.and;
import static org.jooq.impl.DSL.noCondition;
import static org.jooq.impl.DSL.row;
import static ru.yandex.direct.common.jooqmapperex.ReaderWriterBuildersEx.integerProperty;
import static ru.yandex.direct.core.entity.dynamictextadtarget.repository.DynamicTextAdTargetMapping.SUSPENDED_OPT;
import static ru.yandex.direct.dbschema.ppc.tables.AdgroupsDynamic.ADGROUPS_DYNAMIC;
import static ru.yandex.direct.dbschema.ppc.tables.AdgroupsText.ADGROUPS_TEXT;
import static ru.yandex.direct.dbschema.ppc.tables.BidsDynamic.BIDS_DYNAMIC;
import static ru.yandex.direct.dbschema.ppc.tables.Campaigns.CAMPAIGNS;
import static ru.yandex.direct.dbschema.ppc.tables.DynamicConditions.DYNAMIC_CONDITIONS;
import static ru.yandex.direct.dbschema.ppc.tables.Feeds.FEEDS;
import static ru.yandex.direct.dbschema.ppc.tables.Phrases.PHRASES;
import static ru.yandex.direct.jooqmapper.JooqMapperUtils.makeCaseStatement;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.direct.jooqmapper.read.ReaderBuilders.fromField;
import static ru.yandex.direct.jooqmapper.write.WriterBuilders.fromProperty;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.utils.FunctionalUtils.mapSet;

@Repository
@ParametersAreNonnullByDefault
public class DynamicTextAdTargetRepository {

    private final DslContextProvider dslContextProvider;
    private final ShardHelper shardHelper;
    private final PerformanceFilterStorage performanceFilterStorage;
    private final PerformanceFilterConditionDBFormatParser feedConditionDbParser;

    private final ImmutableMap<DynamicTextAdTargetState, Condition> stateOptions = createStateOptions();

    private final JooqMapper<DynamicAdTarget> baseMapper;
    private final JooqMapperWithSupplier<DynamicTextAdTarget> dynamicTextAdTargetMapper;
    private final JooqMapperWithSupplier<DynamicFeedAdTarget> dynamicFeedAdTargetMapper;

    @Autowired
    public DynamicTextAdTargetRepository(
            DslContextProvider dslContextProvider,
            ShardHelper shardHelper,
            PerformanceFilterStorage performanceFilterStorage) {
        this.dslContextProvider = dslContextProvider;
        this.shardHelper = shardHelper;
        this.performanceFilterStorage = performanceFilterStorage;
        this.feedConditionDbParser = PerformanceFilterConditionDBFormatParser.INSTANCE;
        this.baseMapper = createBaseMapper();
        this.dynamicTextAdTargetMapper = createDynamicTextMapper();
        this.dynamicFeedAdTargetMapper = createDynamicFeedMapper();
    }

    private static JooqMapper<DynamicAdTarget> createBaseMapper() {
        return JooqMapperBuilder.<DynamicAdTarget>builder()
                .map(property(DynamicAdTarget.ID, BIDS_DYNAMIC.DYN_ID))

                .writeField(BIDS_DYNAMIC.DYN_COND_ID, fromProperty(DynamicAdTarget.DYNAMIC_CONDITION_ID))
                .writeField(DYNAMIC_CONDITIONS.DYN_COND_ID, fromProperty(DynamicAdTarget.DYNAMIC_CONDITION_ID))
                .readProperty(DynamicAdTarget.DYNAMIC_CONDITION_ID, fromField(DYNAMIC_CONDITIONS.DYN_COND_ID))

                .writeField(BIDS_DYNAMIC.PID, fromProperty(DynamicAdTarget.AD_GROUP_ID))
                .writeField(DYNAMIC_CONDITIONS.PID, fromProperty(DynamicAdTarget.AD_GROUP_ID))
                .readProperty(DynamicAdTarget.AD_GROUP_ID, fromField(DYNAMIC_CONDITIONS.PID))
                .readProperty(DynamicAdTarget.CAMPAIGN_ID, fromField(CAMPAIGNS.CID))

                .map(convertibleProperty(DynamicAdTarget.PRICE, BIDS_DYNAMIC.PRICE,
                        DynamicTextAdTargetMapping::priceFromDbFormat, DynamicTextAdTargetMapping::priceToDbFormat))
                .map(convertibleProperty(DynamicAdTarget.PRICE_CONTEXT, BIDS_DYNAMIC.PRICE_CONTEXT,
                        DynamicTextAdTargetMapping::priceFromDbFormat, DynamicTextAdTargetMapping::priceToDbFormat))
                .map(integerProperty(DynamicAdTarget.AUTOBUDGET_PRIORITY, BIDS_DYNAMIC.AUTOBUDGET_PRIORITY))
                .map(convertibleProperty(DynamicAdTarget.IS_SUSPENDED, BIDS_DYNAMIC.OPTS,
                        DynamicTextAdTargetMapping::optsFromDb, DynamicTextAdTargetMapping::optsToDb))

                .map(property(DynamicAdTarget.CONDITION_NAME, DYNAMIC_CONDITIONS.CONDITION_NAME))
                .map(convertibleProperty(DynamicAdTarget.CONDITION_HASH, DYNAMIC_CONDITIONS.CONDITION_HASH,
                        UNumber::toBigInteger, ULong::valueOf))
                .map(convertibleProperty(DynamicAdTarget.TAB, BIDS_DYNAMIC.FROM_TAB,
                        DynamicAdTargetTab::fromSource,
                        v -> v == null ? BidsDynamicFromTab.condition : DynamicAdTargetTab.toSource(v)))
                .build();
    }

    private static JooqMapperWithSupplier<DynamicTextAdTarget> createDynamicTextMapper() {
        return JooqMapperWithSupplierBuilder.builder(createBaseMapper(), DynamicTextAdTarget::new)
                .writeField(DYNAMIC_CONDITIONS.CONDITION_JSON,
                        fromProperty(DynamicTextAdTarget.CONDITION).by(DynamicTextAdTargetMapping::webpageRulesToJson))
                .readProperty(DynamicTextAdTarget.CONDITION, fromField(DYNAMIC_CONDITIONS.CONDITION_JSON)
                        .by(DynamicTextAdTargetMapping::webpageRulesFromJson))
                .readProperty(DynamicTextAdTarget.CONDITION_UNIQ_HASH,
                        fromField(DYNAMIC_CONDITIONS.CONDITION_JSON).by(DynamicTextAdTargetHashUtils::getUniqHash))
                .build();
    }

    private static JooqMapperWithSupplier<DynamicFeedAdTarget> createDynamicFeedMapper() {
        return JooqMapperWithSupplierBuilder.builder(createBaseMapper(), DynamicFeedAdTarget::new)
                .readProperty(DynamicFeedAdTarget.FEED_TYPE, fromField(FEEDS.FEED_TYPE)
                        .by(FeedType::fromTypedValue))
                .readProperty(DynamicFeedAdTarget.BUSINESS_TYPE, fromField(FEEDS.BUSINESS_TYPE)
                        .by(BusinessType::fromSource))
                .map(convertibleProperty(DynamicFeedAdTarget.CONDITION, DYNAMIC_CONDITIONS.CONDITION_JSON,
                        //условия не конвертируем в маппере, т.к. нужен контекст из PerformanceFilterStorage
                        v -> null,
                        DynamicTextAdTargetMapping::dynamicFeedRulesToJson))
                .build();
    }

    private static ImmutableMap<DynamicTextAdTargetState, Condition> createStateOptions() {
        Condition deletedCondition = BIDS_DYNAMIC.DYN_ID.isNull();

        Condition suspendedCondition = and(
                BIDS_DYNAMIC.OPTS.eq(SUSPENDED_OPT),
                BIDS_DYNAMIC.DYN_ID.isNotNull()
        );

        // да, эти два условия стали одинаковыми, см DIRECT-155960: сломался статус ON у DynamicAdTextTarget в апи
        // возможно на замену bids_dynamic.statusBsSynced стоит использовать статус из phrases или аггр статусы
        Condition onCondition = and(
                BIDS_DYNAMIC.OPTS.ne(SUSPENDED_OPT),
                BIDS_DYNAMIC.DYN_ID.isNotNull()
        );

        Condition offCondition = and(
                BIDS_DYNAMIC.OPTS.ne(SUSPENDED_OPT),
                BIDS_DYNAMIC.DYN_ID.isNotNull()
        );

        return ImmutableMap.of(DynamicTextAdTargetState.OFF, offCondition,
                DynamicTextAdTargetState.ON, onCondition,
                DynamicTextAdTargetState.SUSPENDED, suspendedCondition,
                DynamicTextAdTargetState.DELETED, deletedCondition);
    }


    /**
     * Получает отображение id кампании в список id условий по заданным условиям отбора
     *
     * @param shard             шард
     * @param selectionCriteria условия отбора
     * @return отображение id кампании в список id условий
     */
    public Map<Long, List<Long>> getCampaignIdToConditionIdsBySelectionCriteria(
            int shard,
            DynamicTextAdTargetSelectionCriteria selectionCriteria) {
        Set<Long> conditionIds = selectionCriteria.getConditionIds();
        Set<Long> adGroupIds = selectionCriteria.getAdGroupIds();
        Set<Long> campaignIds = selectionCriteria.getCampaignIds();
        Set<BusinessIdAndShopId> businessIdsAndShopIds = selectionCriteria.getBusinessIdsAndShopIds();
        Set<DynamicTextAdTargetState> states = selectionCriteria.getStates();

        checkArgument(!conditionIds.isEmpty() || !adGroupIds.isEmpty() || !campaignIds.isEmpty()
                        || !businessIdsAndShopIds.isEmpty() || !states.isEmpty(),
                "ConditionIds, AdGroupIds, CampaignIds, BusinessIdsAndShopIds or States must be specified!");

        boolean joinFeeds = false;
        List<Condition> conditions = new ArrayList<>();

        if (!conditionIds.isEmpty()) {
            conditions.add(DYNAMIC_CONDITIONS.DYN_COND_ID.in(conditionIds));
        }

        if (!adGroupIds.isEmpty()) {
            conditions.add(PHRASES.PID.in(adGroupIds));
        }

        if (!campaignIds.isEmpty()) {
            conditions.add(PHRASES.CID.in(campaignIds));
        }

        if (!businessIdsAndShopIds.isEmpty()) {
            joinFeeds = true;
            conditions.add(row(FEEDS.MARKET_BUSINESS_ID, FEEDS.MARKET_SHOP_ID).in(mapSet(businessIdsAndShopIds,
                    businessIdAndShopId -> row(businessIdAndShopId.getBusinessId(), businessIdAndShopId.getShopId()))));
        }

        if (!states.isEmpty()) {
            conditions.add(DSL.or(mapSet(states, stateOptions::get)));
        }

        SelectJoinStep<Record2<Long, Long>> selectJoinStep = dslContextProvider.ppc(shard)
                .select(DYNAMIC_CONDITIONS.DYN_COND_ID, PHRASES.CID)
                .from(DYNAMIC_CONDITIONS)
                .leftJoin(BIDS_DYNAMIC).on(DYNAMIC_CONDITIONS.DYN_COND_ID.eq(BIDS_DYNAMIC.DYN_COND_ID))
                .join(PHRASES).on(DYNAMIC_CONDITIONS.PID.eq(PHRASES.PID));

        if (joinFeeds) {
            selectJoinStep = selectJoinStep
                    .leftJoin(ADGROUPS_DYNAMIC).on(PHRASES.PID.eq(ADGROUPS_DYNAMIC.PID))
                    .leftJoin(ADGROUPS_TEXT).on(PHRASES.PID.eq(ADGROUPS_TEXT.PID))
                    .join(FEEDS).on(ADGROUPS_DYNAMIC.FEED_ID.eq(FEEDS.FEED_ID)
                            .or(ADGROUPS_TEXT.FEED_ID.eq(FEEDS.FEED_ID)));
        }

        return selectJoinStep
                .where(conditions)
                .orderBy(DYNAMIC_CONDITIONS.DYN_COND_ID)
                .fetchGroups(PHRASES.CID, DYNAMIC_CONDITIONS.DYN_COND_ID);
    }

    /**
     * Получить список условий нацеливания с типом "по домену". ДТО по фиду игнорируются
     *
     * @param shard                  шард
     * @param clientId               идентификатор клиента
     * @param dynamicTextAdTargetIds идентификаторы условий нацеливания
     * @param limitOffset            ограничение количества записей и смещение от начала выборки
     * @return {@link List<DynamicTextAdTarget>} список условий нацеливания
     */
    public List<DynamicTextAdTarget> getDynamicTextAdTargetsWithDomainType(
            int shard, ClientId clientId,
            Collection<Long> dynamicTextAdTargetIds,
            boolean withDeleted,
            LimitOffset limitOffset) {

        DSLContext dslContext = dslContextProvider.ppc(shard);

        SelectConditionStep<Record> selectConditionStep = buildJoinStep(dslContext, withDeleted)
                .where(ADGROUPS_DYNAMIC.MAIN_DOMAIN_ID.isNotNull())
                .and(CAMPAIGNS.CLIENT_ID.eq(clientId.asLong()))
                .and(DYNAMIC_CONDITIONS.DYN_COND_ID.in(dynamicTextAdTargetIds));

        return selectConditionStep
                .orderBy(DYNAMIC_CONDITIONS.DYN_COND_ID)
                .limit(limitOffset.limit())
                .offset(limitOffset.offset())
                .fetch(dynamicTextAdTargetMapper::fromDb);
    }

    public List<DynamicTextAdTarget> getDynamicTextAdTargetsWithDomainTypeByAdGroup(
            int shard, ClientId clientId,
            Collection<Long> adGroupIds,
            boolean withDeleted,
            LimitOffset limitOffset) {

        DSLContext dslContext = dslContextProvider.ppc(shard);

        SelectConditionStep<Record> selectConditionStep = buildJoinStep(dslContext, withDeleted)
                .where(ADGROUPS_DYNAMIC.MAIN_DOMAIN_ID.isNotNull())
                .and(CAMPAIGNS.CLIENT_ID.eq(clientId.asLong()))
                .and(DYNAMIC_CONDITIONS.PID.in(adGroupIds));

        return selectConditionStep
                .orderBy(DYNAMIC_CONDITIONS.DYN_COND_ID)
                .limit(limitOffset.limit())
                .offset(limitOffset.offset())
                .fetch(dynamicTextAdTargetMapper::fromDb);
    }

    public List<DynamicTextAdTarget> getDynamicTextAdTargetsWithDomainTypeByCampaign(
            int shard, ClientId clientId,
            Collection<Long> campaignIds,
            boolean withDeleted,
            LimitOffset limitOffset) {

        DSLContext dslContext = dslContextProvider.ppc(shard);

        SelectConditionStep<Record> selectConditionStep = buildJoinStep(dslContext, withDeleted)
                .where(ADGROUPS_DYNAMIC.MAIN_DOMAIN_ID.isNotNull())
                .and(CAMPAIGNS.CLIENT_ID.eq(clientId.asLong()))
                .and(CAMPAIGNS.CID.in(campaignIds));

        return selectConditionStep
                .orderBy(DYNAMIC_CONDITIONS.DYN_COND_ID)
                .limit(limitOffset.limit())
                .offset(limitOffset.offset())
                .fetch(dynamicTextAdTargetMapper::fromDb);
    }

    private SelectOnConditionStep<Record> buildJoinStep(DSLContext dslContext, boolean withDeleted) {
        SelectJoinStep<Record> selectStep = dslContext
                .select(dynamicTextAdTargetMapper.getFieldsToRead())
                .from(DYNAMIC_CONDITIONS);

        SelectOnConditionStep<Record> joinStep;
        if (withDeleted) {
            joinStep = selectStep
                    .leftJoin(BIDS_DYNAMIC).on(DYNAMIC_CONDITIONS.DYN_COND_ID.eq(BIDS_DYNAMIC.DYN_COND_ID));
        } else {
            joinStep = selectStep
                    .join(BIDS_DYNAMIC).on(DYNAMIC_CONDITIONS.DYN_COND_ID.eq(BIDS_DYNAMIC.DYN_COND_ID));
        }

        joinStep
                .join(PHRASES).on(PHRASES.PID.eq(DYNAMIC_CONDITIONS.PID))
                .join(ADGROUPS_DYNAMIC).on(ADGROUPS_DYNAMIC.PID.eq(PHRASES.PID))
                .join(CAMPAIGNS).on(CAMPAIGNS.CID.eq(PHRASES.CID));

        return joinStep;
    }

    /**
     * Получить список условий нацеливания с типом "по домену". ДТО по фиду игнорируются
     *
     * @param shard      шард
     * @param clientId   идентификатор клиента
     * @param adGroupIds идентификаторы групп
     * @return {@link List<DynamicTextAdTarget>} список условий нацеливания
     */
    public List<DynamicTextAdTarget> getDynamicTextAdTargetsWithDomainTypeByAdGroup(int shard, ClientId clientId,
                                                                                    Collection<Long> adGroupIds) {
        Condition filterCondition = DYNAMIC_CONDITIONS.PID.in(adGroupIds);
        return getDynamicTextAdTargets(shard, clientId, filterCondition);
    }

    public List<DynamicTextAdTarget> getDynamicTextAdTargetsByIds(int shard, ClientId clientId,
                                                                  Collection<Long> ids) {
        Condition filterCondition = BIDS_DYNAMIC.DYN_ID.in(ids);
        return getDynamicTextAdTargets(shard, clientId, filterCondition);
    }

    private List<DynamicTextAdTarget> getDynamicTextAdTargets(int shard, ClientId clientId,
                                                              Condition filterCondition) {
        DSLContext dslContext = dslContextProvider.ppc(shard);
        return getDynamicTextAdTargets(dslContext, clientId, filterCondition);
    }

    private List<DynamicTextAdTarget> getDynamicTextAdTargets(DSLContext dslContext, ClientId clientId,
                                                              Condition filterCondition) {
        SelectOnConditionStep<Record> selectOnConditionStep = dslContext
                .select(dynamicTextAdTargetMapper.getFieldsToRead())
                .from(DYNAMIC_CONDITIONS)
                .leftJoin(BIDS_DYNAMIC).on(DYNAMIC_CONDITIONS.DYN_COND_ID.eq(BIDS_DYNAMIC.DYN_COND_ID))
                .join(PHRASES).on(PHRASES.PID.eq(DYNAMIC_CONDITIONS.PID))
                .join(ADGROUPS_DYNAMIC).on(ADGROUPS_DYNAMIC.PID.eq(PHRASES.PID))
                .join(CAMPAIGNS).on(CAMPAIGNS.CID.eq(PHRASES.CID));

        SelectConditionStep<Record> selectConditionStep = selectOnConditionStep
                .where(ADGROUPS_DYNAMIC.MAIN_DOMAIN_ID.isNotNull())
                .and(CAMPAIGNS.CLIENT_ID.eq(clientId.asLong()))
                .and(filterCondition);

        return selectConditionStep
                .orderBy(DYNAMIC_CONDITIONS.DYN_COND_ID)
                .fetch(dynamicTextAdTargetMapper::fromDb);
    }

    public List<DynamicAdTarget> getDynamicAdTargetsByDynamicConditionIds(int shard, ClientId clientId,
                                                                          Collection<Long> dynamicConditionIds) {
        List<DynamicAdTarget> dynamicAdTargets = new ArrayList<>();
        dynamicAdTargets.addAll(getDynamicTextAdTargetsWithDomainType(shard, clientId, dynamicConditionIds,
                true, LimitOffset.maxLimited()));
        dynamicAdTargets.addAll(getDynamicFeedAdTargets(shard, clientId, dynamicConditionIds));
        return dynamicAdTargets;
    }

    public List<DynamicAdTarget> getDynamicAdTargetsByAdGroupIds(DSLContext dslContext, ClientId clientId,
                                                                 Collection<Long> adGroupIds) {
        List<DynamicAdTarget> dynamicAdTargets = new ArrayList<>();
        dynamicAdTargets.addAll(getDynamicTextAdTargets(dslContext, clientId, DYNAMIC_CONDITIONS.PID.in(adGroupIds)));
        dynamicAdTargets.addAll(getDynamicFeedAdTargets(dslContext, clientId, DYNAMIC_CONDITIONS.PID.in(adGroupIds)));
        return dynamicAdTargets;
    }

    public List<DynamicAdTarget> getDynamicAdTargetsByQueryFilter(int shard, ClientId clientId,
                                                                  DynamicAdTargetsQueryFilter queryFilter) {
        List<DynamicAdTarget> dynamicAdTargets = new ArrayList<>();
        dynamicAdTargets.addAll(getDynamicTextAdTargets(shard, clientId, getFilterCondition(queryFilter)));
        dynamicAdTargets.addAll(getDynamicFeedAdTargets(shard, clientId, getFilterCondition(queryFilter)));
        return dynamicAdTargets;
    }

    /**
     * Получить список условий нацеливания с типом "по фиду"
     */
    public List<DynamicFeedAdTarget> getDynamicFeedAdTargets(int shard, ClientId clientId,
                                                             Collection<Long> dynamicConditionIds) {
        Condition filterCondition = DYNAMIC_CONDITIONS.DYN_COND_ID.in(dynamicConditionIds);
        return getDynamicFeedAdTargets(shard, clientId, filterCondition, true, null);
    }

    public List<DynamicFeedAdTarget> getDynamicFeedAdTargets(int shard, ClientId clientId,
                                                             Collection<Long> dynamicConditionIds,
                                                             boolean withDeleted,
                                                             @Nullable LimitOffset limitOffset) {
        Condition filterCondition = DYNAMIC_CONDITIONS.DYN_COND_ID.in(dynamicConditionIds)
                .and(ADGROUPS_DYNAMIC.FEED_ID.isNotNull()
                        .or(ADGROUPS_TEXT.FEED_ID.isNotNull()));
        return getDynamicFeedAdTargets(shard, clientId, filterCondition, withDeleted, limitOffset);
    }

    public Map<BusinessIdAndShopId, List<DynamicFeedAdTarget>> getDynamicFeedAdTargetsByBusinessIdAndShopId(
            int shard, ClientId clientId, Collection<Long> dynamicConditionIds, boolean withDeleted) {
        Condition condition = DYNAMIC_CONDITIONS.DYN_COND_ID.in(dynamicConditionIds)
                .and(CAMPAIGNS.CLIENT_ID.eq(clientId.asLong()));
        return getDynamicFeedAdTargetsByBusinessIdAndShopId(shard, condition, withDeleted);
    }

    public List<DynamicFeedAdTarget> getDynamicFeedAdTargetsByIds(int shard, ClientId clientId,
                                                                  Collection<Long> ids) {
        Condition filterCondition = BIDS_DYNAMIC.DYN_ID.in(ids);
        return getDynamicFeedAdTargets(shard, clientId, filterCondition);
    }

    public List<DynamicFeedAdTarget> getDynamicFeedAdTargetsByAdGroupIds(int shard, ClientId clientId,
                                                                         Collection<Long> adGroupIds) {
        Condition filterCondition = DYNAMIC_CONDITIONS.PID.in(adGroupIds);
        return getDynamicFeedAdTargets(shard, clientId, filterCondition);
    }

    public List<DynamicFeedAdTarget> getDynamicFeedAdTargetsByAdGroupIds(int shard, ClientId clientId,
                                                                         Collection<Long> adGroupIds,
                                                                         boolean withDeleted,
                                                                         LimitOffset limitOffset) {
        Condition filterCondition = DYNAMIC_CONDITIONS.PID.in(adGroupIds)
                .and(ADGROUPS_DYNAMIC.FEED_ID.isNotNull()
                        .or(ADGROUPS_TEXT.FEED_ID.isNotNull()));
        return getDynamicFeedAdTargets(shard, clientId, filterCondition, withDeleted, limitOffset);
    }

    public List<DynamicFeedAdTarget> getDynamicFeedAdTargetsByCampaignIds(int shard, ClientId clientId,
                                                                          Collection<Long> campaignIds,
                                                                          boolean withDeleted,
                                                                          LimitOffset limitOffset) {
        Condition filterCondition = CAMPAIGNS.CID.in(campaignIds)
                .and(ADGROUPS_DYNAMIC.FEED_ID.isNotNull()
                        .or(ADGROUPS_TEXT.FEED_ID.isNotNull()));
        return getDynamicFeedAdTargets(shard, clientId, filterCondition, withDeleted, limitOffset);
    }

    private List<DynamicFeedAdTarget> getDynamicFeedAdTargets(int shard, ClientId clientId,
                                                              Condition filterCondition) {
        return getDynamicFeedAdTargets(shard, clientId, filterCondition, true, null);
    }

    private List<DynamicFeedAdTarget> getDynamicFeedAdTargets(int shard, ClientId clientId,
                                                              Condition filterCondition,
                                                              boolean withDeleted,
                                                              @Nullable LimitOffset limitOffset) {
        DSLContext dslContext = dslContextProvider.ppc(shard);
        return getDynamicFeedAdTargets(dslContext, clientId, filterCondition, withDeleted, limitOffset);
    }

    private List<DynamicFeedAdTarget> getDynamicFeedAdTargets(DSLContext dslContext, ClientId clientId,
                                                              Condition filterCondition) {
        return getDynamicFeedAdTargets(dslContext, clientId, filterCondition, true, null);
    }

    private List<DynamicFeedAdTarget> getDynamicFeedAdTargets(DSLContext dslContext, ClientId clientId,
                                                              Condition filterCondition,
                                                              boolean withDeleted,
                                                              @Nullable LimitOffset limitOffset) {
        SelectJoinStep<Record> selectStep = joinAllDynamicConditionsRelatedTables(dslContext
                .select(dynamicFeedAdTargetMapper.getFieldsToRead())
                .from(DYNAMIC_CONDITIONS), withDeleted);

        selectStep
                .where(CAMPAIGNS.CLIENT_ID.eq(clientId.asLong()))
                .and(filterCondition)
                .orderBy(DYNAMIC_CONDITIONS.DYN_COND_ID);

        if (limitOffset != null) {
            selectStep
                    .limit(limitOffset.limit())
                    .offset(limitOffset.offset());
        }

        return selectStep
                .fetch(this::convertDynamicFeedAdTargetFromDb);
    }

    private Map<BusinessIdAndShopId, List<DynamicFeedAdTarget>> getDynamicFeedAdTargetsByBusinessIdAndShopId(
            int shard, Condition condition, boolean withDeleted) {
        return joinAllDynamicConditionsRelatedTables(dslContextProvider.ppc(shard)
                .select(dynamicFeedAdTargetMapper.getFieldsToRead())
                .select(FEEDS.MARKET_BUSINESS_ID, FEEDS.MARKET_SHOP_ID)
                .from(DYNAMIC_CONDITIONS), withDeleted)
                .where(condition)
                .fetchGroups(r -> BusinessIdAndShopId.ofNullable(r.get(FEEDS.MARKET_BUSINESS_ID),
                        r.get(FEEDS.MARKET_SHOP_ID)), this::convertDynamicFeedAdTargetFromDb);
    }

    private <R extends Record> SelectJoinStep<R> joinAllDynamicConditionsRelatedTables(SelectJoinStep<R> step,
                                                                                       boolean withDeleted) {
        if (withDeleted) {
            step = step.leftJoin(BIDS_DYNAMIC).on(BIDS_DYNAMIC.DYN_COND_ID.eq(DYNAMIC_CONDITIONS.DYN_COND_ID));
        } else {
            step = step.join(BIDS_DYNAMIC).on(DYNAMIC_CONDITIONS.DYN_COND_ID.eq(BIDS_DYNAMIC.DYN_COND_ID));
        }
        return step
                .join(PHRASES).on(PHRASES.PID.eq(DYNAMIC_CONDITIONS.PID))
                .join(CAMPAIGNS).on(CAMPAIGNS.CID.eq(PHRASES.CID))
                .leftJoin(ADGROUPS_DYNAMIC).on(ADGROUPS_DYNAMIC.PID.eq(PHRASES.PID))
                .leftJoin(ADGROUPS_TEXT).on(ADGROUPS_TEXT.PID.eq(PHRASES.PID))
                .join(FEEDS).on(FEEDS.FEED_ID.eq(ADGROUPS_DYNAMIC.FEED_ID)
                        .or(FEEDS.FEED_ID.eq(ADGROUPS_TEXT.FEED_ID)));
    }

    DynamicFeedAdTarget convertDynamicFeedAdTargetFromDb(Record record) {
        DynamicFeedAdTarget dynamicFeedAdTarget = dynamicFeedAdTargetMapper.fromDb(record);
        String conditionJson = record.get(DYNAMIC_CONDITIONS.CONDITION_JSON);

        FilterSchema filterSchema = performanceFilterStorage.getFilterSchema(
                dynamicFeedAdTarget.getBusinessType(), dynamicFeedAdTarget.getFeedType());
        List<DynamicFeedRule> conditions = feedConditionDbParser.parseDynamic(filterSchema, conditionJson);

        dynamicFeedAdTarget.withCondition(conditions);
        return dynamicFeedAdTarget;
    }

    private static Condition getFilterCondition(DynamicAdTargetsQueryFilter queryFilter) {
        Condition condition = noCondition();
        boolean withDeleted = nvl(queryFilter.getIncludeDeleted(), false);

        if (queryFilter.getIds() != null) {
            condition = condition.and(BIDS_DYNAMIC.DYN_ID.in(queryFilter.getIds()));
        }
        if (queryFilter.getAdGroupIds() != null) {
            condition = condition.and(DYNAMIC_CONDITIONS.PID.in(queryFilter.getAdGroupIds()));
        }
        if (queryFilter.getCampaignIds() != null) {
            condition = condition.and(PHRASES.CID.in(queryFilter.getCampaignIds()));
        }
        if (!withDeleted) {
            condition = condition.and(BIDS_DYNAMIC.DYN_ID.isNotNull());
        }
        if (isNotEmpty(queryFilter.getBaseStatuses())) {
            Condition baseStatusCondition = DSL.or(mapList(queryFilter.getBaseStatuses(),
                    DynamicTextAdTargetRepository::getConditionFromBaseStatus));
            condition = condition.and(baseStatusCondition);
        }
        if (queryFilter.getNameContains() != null) {
            condition = condition.and(DYNAMIC_CONDITIONS.CONDITION_NAME.contains(queryFilter.getNameContains()));
        }
        if (queryFilter.getNameNotContains() != null) {
            condition = condition.and(DYNAMIC_CONDITIONS.CONDITION_NAME.notContains(queryFilter.getNameNotContains()));
        }
        if (queryFilter.getMinPrice() != null) {
            condition = condition.and(BIDS_DYNAMIC.PRICE.ge(queryFilter.getMinPrice()));
        }
        if (queryFilter.getMaxPrice() != null) {
            condition = condition.and(BIDS_DYNAMIC.PRICE.le(queryFilter.getMaxPrice()));
        }
        if (queryFilter.getMinPriceContext() != null) {
            condition = condition.and(BIDS_DYNAMIC.PRICE_CONTEXT.ge(queryFilter.getMinPriceContext()));
        }
        if (queryFilter.getMaxPriceContext() != null) {
            condition = condition.and(BIDS_DYNAMIC.PRICE_CONTEXT.le(queryFilter.getMaxPriceContext()));
        }
        if (queryFilter.getAutobudgetPriorities() != null) {
            condition = condition.and(BIDS_DYNAMIC.AUTOBUDGET_PRIORITY.in(queryFilter.getAutobudgetPriorities()));
        }
        return condition;
    }

    private static Condition getConditionFromBaseStatus(DynamicAdTargetBaseStatus status) {
        switch (status) {
            case ACTIVE:
                return BIDS_DYNAMIC.OPTS.eq(DynamicTextAdTargetMapping.optsToDb(false));
            case SUSPENDED:
                return BIDS_DYNAMIC.OPTS.eq(DynamicTextAdTargetMapping.optsToDb(true));
            default:
                throw new IllegalArgumentException("Unknown status: " + status);
        }
    }

    public Map<Long, Long> getIdByDynamicConditionId(int shard, Collection<Long> dynamicConditionIds) {
        if (dynamicConditionIds.isEmpty()) {
            return emptyMap();
        }
        return dslContextProvider.ppc(shard)
                .select(BIDS_DYNAMIC.DYN_COND_ID, BIDS_DYNAMIC.DYN_ID)
                .from(BIDS_DYNAMIC)
                .where(BIDS_DYNAMIC.DYN_COND_ID.in(dynamicConditionIds))
                .fetchMap(BIDS_DYNAMIC.DYN_COND_ID, BIDS_DYNAMIC.DYN_ID);
    }

    public Map<Long, Long> getDynamicConditionIdById(int shard, Collection<Long> ids) {
        if (ids.isEmpty()) {
            return emptyMap();
        }
        return dslContextProvider.ppc(shard)
                .select(BIDS_DYNAMIC.DYN_ID, BIDS_DYNAMIC.DYN_COND_ID)
                .from(BIDS_DYNAMIC)
                .where(BIDS_DYNAMIC.DYN_ID.in(ids))
                .fetchMap(BIDS_DYNAMIC.DYN_ID, BIDS_DYNAMIC.DYN_COND_ID);
    }

    /**
     * Удаление объявлений по Id из таблицы bids_dynamic.
     *
     * @param dynCondIds список id баннеров
     * @param shard      шард
     */
    public void deleteDynamicTextAdTargets(int shard, Collection<Long> dynCondIds) {
        deleteDynamicTextAdTargets(dslContextProvider.ppc(shard), dynCondIds);
    }

    /**
     * Удаление объявлений по Id из таблицы bids_dynamic.
     *
     * @param dynCondIds список id баннеров
     * @param context    jooq контекст
     */
    public void deleteDynamicTextAdTargets(DSLContext context, Collection<Long> dynCondIds) {
        context
                .deleteFrom(BIDS_DYNAMIC)
                .where(BIDS_DYNAMIC.DYN_COND_ID.in(dynCondIds))
                .execute();
    }


    /**
     * Использовать с оcторожностью. Есть контракт, что данные в этой таблице неизменяемы.
     * Нужно для переиспользования существующих записей в таблице
     *
     * @param context    jooq контекст
     * @param dynCondIds id для удаления из таблицы ppc.dynamic_conditions
     */
    public void deleteFromDynamicConditionTable(DSLContext context, Collection<Long> dynCondIds) {
        if (dynCondIds.isEmpty()) {
            return;
        }

        context
                .deleteFrom(DYNAMIC_CONDITIONS)
                .where(DYNAMIC_CONDITIONS.DYN_COND_ID.in(dynCondIds))
                .execute();
    }

    public void updateSuspended(int shard, List<Long> ids, boolean isSuspended) {
        updateSuspended(dslContextProvider.ppc(shard), ids, isSuspended);
    }

    public void updateSuspended(DSLContext context, List<Long> ids, boolean isSuspended) {
        context
                .update(BIDS_DYNAMIC)
                .set(BIDS_DYNAMIC.OPTS, DynamicTextAdTargetMapping.optsToDb(isSuspended))
                .where(BIDS_DYNAMIC.DYN_COND_ID.in(ids))
                .execute();
    }

    public List<Long> add(int shard, List<? extends DynamicAdTarget> conditions) {
        List<Long> dynamicConditionIds = new ArrayList<>();

        dslContextProvider.ppcTransaction(shard, config ->
                dynamicConditionIds.addAll(
                        addWithoutTransaction(config.dsl(), conditions))
        );
        return dynamicConditionIds;
    }

    public List<Long> addWithoutTransaction(DSLContext context, List<? extends DynamicAdTarget> conditions) {
        if (conditions.isEmpty()) {
            return emptyList();
        }

        generateDynamicAdTargetIdsIfAbsent(conditions);

        addToDynamicConditionsTable(context, conditions);
        addToBidsDynamicTable(context, conditions);

        return mapList(conditions, DynamicAdTarget::getDynamicConditionId);
    }

    private void addToDynamicConditionsTable(DSLContext context, List<? extends DynamicAdTarget> dynamicAdTargets) {
        List<DynamicTextAdTarget> dynamicTextAdTargets = StreamEx.of(dynamicAdTargets)
                .select(DynamicTextAdTarget.class)
                .toList();
        List<DynamicFeedAdTarget> dynamicFeedAdTargets = StreamEx.of(dynamicAdTargets)
                .select(DynamicFeedAdTarget.class)
                .toList();
        checkState(dynamicTextAdTargets.size() + dynamicFeedAdTargets.size() == dynamicAdTargets.size());

        addDynamicTextAdTargetsToDynamicConditionsTable(context, dynamicTextAdTargets);
        addDynamicFeedAdTargetsToDynamicConditionsTable(context, dynamicFeedAdTargets);
    }

    public void setBids(int shard, Collection<? extends AppliedChanges<? extends DynamicAdTarget>> changes) {
        updateBidsDynamic(dslContextProvider.ppc(shard), changes);
    }

    public void updateBidsDynamic(DSLContext context,
                                  Collection<? extends AppliedChanges<? extends DynamicAdTarget>> appliedChanges) {
        JooqUpdateBuilder<BidsDynamicRecord, DynamicAdTarget> ub =
                new JooqUpdateBuilder<>(BIDS_DYNAMIC.DYN_ID, appliedChanges);

        ub.processProperty(DynamicAdTarget.PRICE, BIDS_DYNAMIC.PRICE, DynamicTextAdTargetMapping::priceToDbFormat);
        ub.processProperty(DynamicAdTarget.PRICE_CONTEXT, BIDS_DYNAMIC.PRICE_CONTEXT,
                DynamicTextAdTargetMapping::priceToDbFormat);
        ub.processProperty(DynamicAdTarget.AUTOBUDGET_PRIORITY, BIDS_DYNAMIC.AUTOBUDGET_PRIORITY,
                RepositoryUtils::intToLong);
        ub.processProperty(DynamicAdTarget.IS_SUSPENDED, BIDS_DYNAMIC.OPTS, DynamicTextAdTargetMapping::optsToDb);
        ub.processProperty(DynamicAdTarget.TAB, BIDS_DYNAMIC.FROM_TAB, DynamicAdTargetTab::toSource);

        context
                .update(BIDS_DYNAMIC)
                .set(ub.getValues())
                .where(BIDS_DYNAMIC.DYN_ID.in(ub.getChangedIds()))
                .execute();
    }

    public void updateDynamicConditions(
            DSLContext context,
            Collection<? extends AppliedChanges<? extends DynamicAdTarget>> appliedChanges
    ) {
        Map<Long, String> nameByDynamicConditionId = StreamEx.of(appliedChanges)
                .filter(ac -> ac.changed(DynamicAdTarget.CONDITION_NAME))
                .map(AppliedChanges::getModel)
                .mapToEntry(DynamicAdTarget::getDynamicConditionId, DynamicAdTarget::getConditionName)
                .distinctKeys()
                .toMap();

        if (nameByDynamicConditionId.isEmpty()) {
            return;
        }
        Field<String> conditionNameCase = makeCaseStatement(DYNAMIC_CONDITIONS.DYN_COND_ID,
                DYNAMIC_CONDITIONS.CONDITION_NAME, nameByDynamicConditionId);

        context
                .update(DYNAMIC_CONDITIONS)
                .set(DYNAMIC_CONDITIONS.CONDITION_NAME, conditionNameCase)
                .where(DYNAMIC_CONDITIONS.DYN_COND_ID.in(nameByDynamicConditionId.keySet()))
                .execute();
    }

    private void generateDynamicAdTargetIdsIfAbsent(List<? extends DynamicAdTarget> dynamicAdTargets) {
        int countBidsDynamic = (int) dynamicAdTargets.stream()
                .filter(d -> d.getId() == null)
                .count();

        int countDynamicCondition = (int) dynamicAdTargets.stream()
                .filter(d -> d.getDynamicConditionId() == null)
                .count();

        Iterator<Long> bidsDynamicIds = shardHelper.generateDynamicIds(countBidsDynamic).iterator();
        Iterator<Long> dynamicConditionIds = shardHelper.generateDynamicConditionIds(countDynamicCondition).iterator();

        dynamicAdTargets.forEach(dynamicAdTarget -> {
            if (dynamicAdTarget.getId() == null) {
                dynamicAdTarget
                        .withId(bidsDynamicIds.next());
            }
            if (dynamicAdTarget.getDynamicConditionId() == null) {
                dynamicAdTarget
                        .withDynamicConditionId(dynamicConditionIds.next());
            }
        });
    }

    private void addToBidsDynamicTable(DSLContext context, List<? extends DynamicAdTarget> conditions) {
        InsertHelper.saveModelObjectsToDbTable(context, BIDS_DYNAMIC, baseMapper, conditions);
    }

    private void addDynamicTextAdTargetsToDynamicConditionsTable(DSLContext context,
                                                                 List<DynamicTextAdTarget> conditions) {
        InsertHelper.saveModelObjectsToDbTable(context, DYNAMIC_CONDITIONS, dynamicTextAdTargetMapper, conditions);
    }

    private void addDynamicFeedAdTargetsToDynamicConditionsTable(DSLContext context,
                                                                 List<DynamicFeedAdTarget> conditions) {
        InsertHelper.saveModelObjectsToDbTable(context, DYNAMIC_CONDITIONS, dynamicFeedAdTargetMapper, conditions);
    }
}
