package ru.yandex.direct.core.entity.dynamictextadtarget.service;

import java.math.BigDecimal;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.DSLContext;
import org.jooq.TransactionalRunnable;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.common.log.container.LogPriceData;
import ru.yandex.direct.common.log.service.LogPriceService;
import ru.yandex.direct.core.entity.StatusBsSynced;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupSimple;
import ru.yandex.direct.core.entity.adgroup.model.StatusModerate;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.autobudget.service.AutobudgetAlertService;
import ru.yandex.direct.core.entity.banner.repository.BannerCommonRepository;
import ru.yandex.direct.core.entity.campaign.model.CampaignSimple;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.dynamictextadtarget.model.DynamicAdTarget;
import ru.yandex.direct.core.entity.dynamictextadtarget.service.validation.DynamicTextAdTargetConstants;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.dbutil.model.ClientId;

import static com.google.common.base.Preconditions.checkState;
import static org.apache.commons.lang3.BooleanUtils.isNotTrue;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.filterAndMapToSet;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Хэлпер для {@link DynamicTextAdTargetsAddOperation} и {@link DynamicFeedAdTargetsAddOperation}.
 * Содержит общие методы.
 */
@Component
@ParametersAreNonnullByDefault
public class DynamicAdTargetsAddOperationHelper {

    private final AdGroupRepository adGroupRepository;
    private final BannerCommonRepository bannerCommonRepository;
    private final CampaignRepository campaignRepository;
    private final LogPriceService logPriceService;
    private final DynamicAdTargetsAddUpdateHelper dynamicAdTargetsAddUpdateHelper;
    private final AutobudgetAlertService autobudgetAlertService;

    @Autowired
    public DynamicAdTargetsAddOperationHelper(
            AdGroupRepository adGroupRepository,
            BannerCommonRepository bannerCommonRepository,
            CampaignRepository campaignRepository,
            LogPriceService logPriceService,
            DynamicAdTargetsAddUpdateHelper dynamicAdTargetsAddUpdateHelper,
            AutobudgetAlertService autobudgetAlertService) {
        this.adGroupRepository = adGroupRepository;
        this.bannerCommonRepository = bannerCommonRepository;
        this.campaignRepository = campaignRepository;
        this.logPriceService = logPriceService;
        this.dynamicAdTargetsAddUpdateHelper = dynamicAdTargetsAddUpdateHelper;
        this.autobudgetAlertService = autobudgetAlertService;
    }

    void prepareSystemFields(int shard, Collection<? extends DynamicAdTarget> dynamicAdTargets) {
        fillInCampaignIds(shard, dynamicAdTargets);
    }

    private void fillInCampaignIds(int shard, Collection<? extends DynamicAdTarget> dynamicAdTargets) {
        List<Long> adGroupIds = mapList(dynamicAdTargets, DynamicAdTarget::getAdGroupId);
        Map<Long, CampaignSimple> campaignsByAdGroupIds =
                campaignRepository.getCampaignsSimpleByAdGroupIds(shard, adGroupIds);

        dynamicAdTargets.forEach(dynamicAdTarget -> {
            Long adGroupId = dynamicAdTarget.getAdGroupId();
            CampaignSimple campaign = campaignsByAdGroupIds.get(adGroupId);
            dynamicAdTarget.withCampaignId(campaign == null ? null : campaign.getId());
        });
    }

    void setMissingPrices(List<? extends DynamicAdTarget> dynamicAdTargets, Currency currency) {
        BigDecimal minPrice = currency.getMinPrice();
        dynamicAdTargets.forEach(dynamicAdTarget -> {
            if (dynamicAdTarget.getPrice() == null) {
                dynamicAdTarget.setPrice(minPrice);
            }
            if (dynamicAdTarget.getPriceContext() == null) {
                dynamicAdTarget.setPriceContext(minPrice);
            }
            if (dynamicAdTarget.getAutobudgetPriority() == null) {
                dynamicAdTarget.setAutobudgetPriority(DynamicTextAdTargetConstants.NORMAL_AUTOBUDGET_PRIORITY);
            }
        });
    }

    Runnable computeLogPriceTask(Collection<? extends DynamicAdTarget> dynamicAdTargets,
                                 Long operatorUid, Currency clientCurrency) {
        return () -> {
            List<LogPriceData> priceDataList = computeLogPriceDataList(dynamicAdTargets, clientCurrency);
            logPriceService.logPrice(priceDataList, operatorUid);
        };
    }

    private List<LogPriceData> computeLogPriceDataList(Collection<? extends DynamicAdTarget> dynamicAdTargets,
                                                       Currency clientCurrency) {
        return mapList(dynamicAdTargets, dynamicAdTarget -> {
            checkState(dynamicAdTarget.getId() != null,
                    "attempt to log dynamicAdTarget without id (may be repository does not set id to added " +
                            "dynamicAdTarget or computing log records is called before dynamicAdTarget is added.");
            return toLogPriceData(dynamicAdTarget, clientCurrency, LogPriceData.OperationType.DYN_COND_ADD);
        });
    }

    static LogPriceData toLogPriceData(DynamicAdTarget dynamicAdTarget, Currency clientCurrency,
                                       LogPriceData.OperationType operationType) {
        return new LogPriceData(
                dynamicAdTarget.getCampaignId(),
                dynamicAdTarget.getAdGroupId(),
                dynamicAdTarget.getId(),
                nvl(dynamicAdTarget.getPriceContext(), BigDecimal.ZERO).doubleValue(),
                nvl(dynamicAdTarget.getPrice(), BigDecimal.ZERO).doubleValue(),
                clientCurrency.getCode(),
                operationType);
    }

    TransactionalRunnable computeUpdateStatusesTask(int shard, ClientId clientId,
                                                    Collection<? extends DynamicAdTarget> dynamicAdTargets) {
        return conf -> {
            Set<Long> adGroupIds = listToSet(dynamicAdTargets, DynamicAdTarget::getAdGroupId);

            if (!adGroupIds.isEmpty()) {
                Map<Long, AdGroupSimple> adGroups = adGroupRepository.getAdGroupSimple(shard, clientId, adGroupIds);

                Set<Long> adGroupIdsModerateStatusNotNew = adGroupIds
                        .stream()
                        .filter(adGroupId -> adGroups.get(adGroupId).getStatusModerate() != StatusModerate.NEW)
                        .collect(Collectors.toSet());
                Set<Long> adGroupIdsModerateStatusIsNew = adGroupIds
                        .stream()
                        .filter(adGroupId -> adGroups.get(adGroupId).getStatusModerate() == StatusModerate.NEW)
                        .collect(Collectors.toSet());

                // Обновим статус генерации фраз в BannerLand только если условие не выключено
                Set<Long> adGroupIdsToUpdateStatusBlGenerated = dynamicAdTargets.stream()
                        .filter(dynamicAdTarget -> isNotTrue(dynamicAdTarget.getIsSuspended()))
                        .map(DynamicAdTarget::getAdGroupId)
                        .filter(adGroupIdsModerateStatusNotNew::contains)
                        .collect(Collectors.toSet());

                adGroupRepository
                        .updateStatusBsSyncedWithLastChange(conf, adGroupIdsModerateStatusNotNew, StatusBsSynced.NO);
                adGroupRepository.updateLastChange(conf, adGroupIdsModerateStatusIsNew);

                adGroupRepository.setStatusBlGeneratedForDynamicAdGroupsToProcessingIfNo(conf.dsl(),
                        adGroupIdsToUpdateStatusBlGenerated);
                bannerCommonRepository
                        .updateStatusBsSyncedByAdgroupId(conf.dsl(), adGroupIdsModerateStatusNotNew, StatusBsSynced.NO);

                var campaignIdsToFreezeAlerts = filterAndMapToSet(dynamicAdTargets,
                        dynCond -> adGroupIdsModerateStatusNotNew.contains(dynCond.getAdGroupId()),
                        DynamicAdTarget::getCampaignId);
                autobudgetAlertService.freezeAlertsOnKeywordsChange(clientId, campaignIdsToFreezeAlerts);
            }
        };
    }

    List<Long> addDynamicAdTargets(DSLContext dslContext, ClientId clientId,
                                   List<? extends DynamicAdTarget> dynamicAdTargets) {
        return dynamicAdTargetsAddUpdateHelper.addDynamicAdTargets(dslContext, clientId, dynamicAdTargets);
    }
}
