package ru.yandex.direct.core.entity.dynamictextadtarget.service;

import java.math.BigInteger;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.DSLContext;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.dynamictextadtarget.model.DynamicAdTarget;
import ru.yandex.direct.core.entity.dynamictextadtarget.repository.DynamicTextAdTargetRepository;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.model.AppliedChanges;

import static java.util.stream.Collectors.groupingBy;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Хэлпер для AddOperation и UpdateOperation.
 * Содержит методы для создания и обновления динамических условий в БД.
 */
@Component
@ParametersAreNonnullByDefault
public class DynamicAdTargetsAddUpdateHelper {

    private final DynamicTextAdTargetRepository dynamicTextAdTargetRepository;
    private final DslContextProvider dslContextProvider;
    private final FeatureService featureService;

    @Autowired
    public DynamicAdTargetsAddUpdateHelper(
            DynamicTextAdTargetRepository dynamicTextAdTargetRepository,
            DslContextProvider dslContextProvider,
            FeatureService featureService) {
        this.dynamicTextAdTargetRepository = dynamicTextAdTargetRepository;
        this.dslContextProvider = dslContextProvider;
        this.featureService = featureService;
    }

    <T extends DynamicAdTarget> void updateDynamicAdTargets(int shard, ClientId clientId,
                                                            List<AppliedChanges<T>> applicableAppliedChanges) {
        dslContextProvider.ppcTransaction(shard, conf ->
                updateDynamicAdTargets(conf.dsl(), clientId, applicableAppliedChanges)
        );
    }

    <T extends DynamicAdTarget> void updateDynamicAdTargets(DSLContext dslContext, ClientId clientId,
                                                            List<AppliedChanges<T>> applicableAppliedChanges) {
        List<T> dynamicAdTargetsToReCreate = new ArrayList<>();
        List<AppliedChanges<T>> dynamicAdTargetsToUpdate = new ArrayList<>();

        boolean updateAllowed = featureService.isEnabledForClientId(clientId, FeatureName.UPDATE_DYNAMIC_CONDITIONS_ALLOWED);
        for (AppliedChanges<T> change : applicableAppliedChanges) {
            if (isConditionChanged(change)) {
                dynamicAdTargetsToReCreate.add(change.getModel());
            } else {
                dynamicAdTargetsToUpdate.add(change);
            }
        }

        dynamicTextAdTargetRepository.updateBidsDynamic(dslContext, dynamicAdTargetsToUpdate);
        dynamicTextAdTargetRepository.updateDynamicConditions(dslContext, dynamicAdTargetsToUpdate);

        reCreateDynamicAdTargets(dslContext, clientId, dynamicAdTargetsToReCreate, updateAllowed);
    }

    static <T extends DynamicAdTarget> boolean isConditionChanged(AppliedChanges<T> change) {
        return change.changed(DynamicAdTarget.CONDITION_HASH);
    }

    /**
     * Обновление динамических условий
     * <p>
     * При изменении условия (condition_json) добавляется новая запись в dynamic_conditions
     * и ее dyn_cond_id проставляется в bids_dynamic (так как dynamic_conditions не изменяемая таблица)
     * В bids_dynamic есть ограничение UNIQUE KEY для dyn_cond_id,
     * поэтому не можем просто обновить bids_dynamic.dyn_cond_id в случае (d1, d2) -> (d2, d1)
     * <p>
     * поэтому вначале удаляем условия, а потом добавляем новые с таким же id (bids_dynamic.dyn_id)
     */
    private <T extends DynamicAdTarget> void reCreateDynamicAdTargets(DSLContext dslContext, ClientId clientId,
                                                                      List<T> dynamicAdTargets, boolean updateAllowed) {
        if (dynamicAdTargets.isEmpty()) {
            return;
        }
        List<Long> dynamicConditionIds = mapList(dynamicAdTargets, DynamicAdTarget::getDynamicConditionId);
        dynamicTextAdTargetRepository.deleteDynamicTextAdTargets(dslContext, dynamicConditionIds);

        if (!updateAllowed) {
            // dynamicConditionId соответствующий новому условию проставится при добавлении
            dynamicAdTargets.forEach(d -> d.setDynamicConditionId(null));
        } else { // фича, разрешающая обновлять условия, включена (DIRECT-142282)
            // Вследствие того, что мы не можем обновить записи в таблице dynamic_conditions, не нарушая констрейнтов,
            // мы будем создавать новые записи с такими же dynamicConditionId, для чего нам нужно их сначала освободить.
            // Ссылок из bids_dynamic на них уже нет - записи, ссылающиеся на эти условия, удалены парой строк кода выше.
            dynamicTextAdTargetRepository.deleteFromDynamicConditionTable(dslContext, dynamicConditionIds);
        }

        addDynamicAdTargets(dslContext, clientId, dynamicAdTargets);
    }

    List<Long> addDynamicAdTargets(DSLContext dslContext, ClientId clientId,
                                   List<? extends DynamicAdTarget> dynamicAdTargets) {
        // удаляем переиспользуемое условие из dynamic_condition
        // оно сразу добавится методом dynamicTextAdTargetRepository.addWithoutTransaction
        // который добавляет сразу в обе таблице dynamic_condition и bids_dynamic
        List<Long> dynamicAdTargetIdsToDeleteWhenReuse =
                processReuseDeletedCondition(dslContext, clientId, dynamicAdTargets);
        dynamicTextAdTargetRepository.deleteFromDynamicConditionTable(dslContext,
                dynamicAdTargetIdsToDeleteWhenReuse);

        return dynamicTextAdTargetRepository.addWithoutTransaction(dslContext, dynamicAdTargets);
    }

    /**
     * Переиспользование удаленных условий в ppc.dynamic_conditions
     * <p>
     * Есть две таблицы ppc.bids_dynamic и ppc.dynamic_conditions
     * <p>
     * bids_dynamic изменяемая, в ней цена, ссылка на условие из таблицы dynamic_conditions и другие данные.
     * А dynamic_conditions неизменяемая - в нее можно только добавлять
     * <p>
     * Если создали запись в bids_dynamic с условием в dynamic_conditions, а потом удалили его, то само условие
     * останется. И в следующей раз, если создаем bids_dynamic с таким же условием, это условие нужно
     * переиспользовать, указав его id в качестве dyn_cond_id
     * <p>
     * Поэтому ищем удаленные условия, с условием равным добавляемому.
     * Если нашли, его id проставляем в bids_dynamic.
     * Если не нашли, создаем новое и проставляем id нового (делается в
     * {@link DynamicTextAdTargetRepository#addWithoutTransaction}
     *
     * @param dynamicAdTargets добавляемые условия
     * @return список id условий из dynamic_conditions, которые удалось переиспользовать (нужно будет удалить)
     */
    private List<Long> processReuseDeletedCondition(DSLContext dslContext, ClientId clientId,
                                                    List<? extends DynamicAdTarget> dynamicAdTargets) {
        Set<Long> adGroupIds = listToSet(dynamicAdTargets, DynamicAdTarget::getAdGroupId);

        List<DynamicAdTarget> existedDynamicAdTargets = dynamicTextAdTargetRepository
                .getDynamicAdTargetsByAdGroupIds(dslContext, clientId, adGroupIds);

        Map<Long, List<DynamicAdTarget>> deletedDynamicAdTargetsByAdGroupId =
                existedDynamicAdTargets.stream()
                        .filter(dynamicAdTarget -> dynamicAdTarget.getId() == null)
                        .collect(groupingBy(DynamicAdTarget::getAdGroupId));

        // мапа удаленные условия в таблице dynamic_conditions: adGroupId -> hash -> [condition]
        Map<Long, Map<BigInteger, List<DynamicAdTarget>>> deletedDynamicAdTargetsByAdGroupIdByHash =
                deletedDynamicAdTargetsByAdGroupId.entrySet().stream()
                        .collect(Collectors.toMap(Map.Entry::getKey, v -> v.getValue().stream()
                                .collect(groupingBy(DynamicAdTarget::getConditionHash))));

        List<Long> dynamicAdTargetIdsReused = new ArrayList<>();

        // данные в таблице dynamic_conditions неизменяемы и если осталось условие от удаленного bids_dynamic,
        // его нужно переиспользовать. Есть констрейнт, что пара pid + condition_hash в таблице уникальна
        dynamicAdTargets.forEach(dynamicAdTarget -> {
            Map<BigInteger, List<DynamicAdTarget>> dynamicAdTargetsByHash =
                    deletedDynamicAdTargetsByAdGroupIdByHash.get(dynamicAdTarget.getAdGroupId());

            if (dynamicAdTargetsByHash != null) {
                // в таблице dynamic_conditions в текущей adGroup есть удаленные условия

                List<DynamicAdTarget> existedDynamicAdTargetsInAdGroup =
                        dynamicAdTargetsByHash.get(dynamicAdTarget.getConditionHash());

                if (existedDynamicAdTargetsInAdGroup != null && !existedDynamicAdTargetsInAdGroup.isEmpty()) {
                    // в таблице dynamic_conditions есть удаленное условие с совпадающем condition_hash для добавляемого
                    // переиспользуем его
                    DynamicAdTarget dynamicAdTargetToReuse = existedDynamicAdTargetsInAdGroup.get(0);
                    // если dynamicConditionId уже проставлен, это значит, что условие собираются обновить,
                    // сохраняя его id - DIRECT-142282. Удалить найденное условие при этом будет нужно,
                    // чтобы в таблице не было условий с совпадающим pid + condition_hash
                    if (dynamicAdTarget.getDynamicConditionId() == null) {
                        dynamicAdTarget.setDynamicConditionId(dynamicAdTargetToReuse.getDynamicConditionId());
                    }
                    dynamicAdTargetIdsReused.add(dynamicAdTargetToReuse.getDynamicConditionId());
                }
            }
        });

        return dynamicAdTargetIdsReused;
    }
}
