package ru.yandex.direct.core.entity.dynamictextadtarget.service;

import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.jooq.DSLContext;
import org.jooq.TransactionalRunnable;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.common.log.container.LogPriceData;
import ru.yandex.direct.common.log.service.LogPriceService;
import ru.yandex.direct.core.entity.StatusBsSynced;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupSimple;
import ru.yandex.direct.core.entity.adgroup.model.StatusModerate;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.banner.repository.BannerCommonRepository;
import ru.yandex.direct.core.entity.dynamictextadtarget.model.DynamicAdTarget;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.model.AppliedChanges;

import static ru.yandex.direct.core.entity.dynamictextadtarget.service.DynamicAdTargetsAddOperationHelper.toLogPriceData;
import static ru.yandex.direct.core.entity.dynamictextadtarget.service.DynamicAdTargetsAddUpdateHelper.isConditionChanged;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;

/**
 * Хэлпер для {@link DynamicTextAdTargetsUpdateOperation} и {@link DynamicFeedAdTargetsUpdateOperation}.
 * Содержит общие методы.
 */
@Component
@ParametersAreNonnullByDefault
public class DynamicAdTargetsUpdateOperationHelper {

    private final AdGroupRepository adGroupRepository;
    private final BannerCommonRepository bannerCommonRepository;
    private final LogPriceService logPriceService;
    private final DynamicAdTargetsAddUpdateHelper dynamicAdTargetsAddUpdateHelper;

    @Autowired
    public DynamicAdTargetsUpdateOperationHelper(
            AdGroupRepository adGroupRepository,
            BannerCommonRepository bannerCommonRepository,
            LogPriceService logPriceService,
            DynamicAdTargetsAddUpdateHelper dynamicAdTargetsAddUpdateHelper) {
        this.adGroupRepository = adGroupRepository;
        this.bannerCommonRepository = bannerCommonRepository;
        this.logPriceService = logPriceService;
        this.dynamicAdTargetsAddUpdateHelper = dynamicAdTargetsAddUpdateHelper;
    }

    <T extends DynamicAdTarget> Runnable computeAdditionalTask(Collection<AppliedChanges<T>> validAppliedChanges,
                                                               long operatorUid, Currency clientCurrency) {
        List<LogPriceData> priceDataList = StreamEx.of(validAppliedChanges)
                .filter(ac -> ac.changed(DynamicAdTarget.PRICE) || ac.changed(DynamicAdTarget.PRICE_CONTEXT))
                .map(AppliedChanges::getModel)
                .map(d -> toLogPriceData(d, clientCurrency, LogPriceData.OperationType.DYN_COND_UPDATE))
                .toList();

        return () -> {
            if (!priceDataList.isEmpty()) {
                logPriceService.logPrice(priceDataList, operatorUid);
            }
        };
    }

    <T extends DynamicAdTarget> TransactionalRunnable computeTransactionalTask(
            int shard, ClientId clientId, Collection<AppliedChanges<T>> validAppliedChanges) {

        Set<Long> adGroupIds = listToSet(validAppliedChanges, c -> c.getModel().getAdGroupId());
        Map<Long, AdGroupSimple> adGroupById = adGroupRepository.getAdGroupSimple(shard, clientId, adGroupIds);

        Set<Long> adGroupIdsModerateStatusNotNew = adGroupById.values().stream()
                .filter(adGroup -> adGroup.getStatusModerate() != StatusModerate.NEW)
                .map(AdGroupSimple::getId)
                .collect(Collectors.toSet());

        Set<Long> adGroupIdsToBsSync = new HashSet<>();
        Set<Long> adGroupIdsToUpdateStatusBlGenerated = new HashSet<>();
        Set<Long> adGroupIdsToUpdateLastChange = new HashSet<>();

        for (AppliedChanges<T> change : validAppliedChanges) {
            Long adGroupId = change.getModel().getAdGroupId();
            boolean isSuspended = nvl(change.getModel().getIsSuspended(), false);

            boolean isConditionChanged = isConditionChanged(change);
            boolean isSuspendedChanged = change.changed(DynamicAdTarget.IS_SUSPENDED);
            boolean isNameChanged = change.changed(DynamicAdTarget.CONDITION_NAME);

            // Если группа не черновик и изменилось содержимое/приостановка условия,
            // то сбросим статусы синхронизации группы/баннеров
            if ((isConditionChanged || isSuspendedChanged) && adGroupIdsModerateStatusNotNew.contains(adGroupId)) {
                adGroupIdsToBsSync.add(adGroupId);

                // Обновим статус генерации фраз в BannerLand только если условие не выключено
                if (!isSuspended) {
                    adGroupIdsToUpdateStatusBlGenerated.add(adGroupId);
                }
            }
            // При изменении имени/содержимого/приостановки условия -- обновим время изменения группы
            if (isNameChanged || isConditionChanged || isSuspendedChanged) {
                adGroupIdsToUpdateLastChange.add(adGroupId);
            }
        }

        return conf -> {
            adGroupRepository.updateStatusBsSynced(conf, adGroupIdsToBsSync, StatusBsSynced.NO);
            bannerCommonRepository.updateStatusBsSyncedByAdgroupId(conf.dsl(), adGroupIdsToBsSync, StatusBsSynced.NO);

            adGroupRepository.updateLastChange(conf, adGroupIdsToUpdateLastChange);
            adGroupRepository.setStatusBlGeneratedForDynamicAdGroupsToProcessingIfNo(conf.dsl(),
                    adGroupIdsToUpdateStatusBlGenerated);
        };
    }

    <T extends DynamicAdTarget> void updateDynamicAdTargets(DSLContext dslContext, ClientId clientId,
                                                            List<AppliedChanges<T>> applicableAppliedChanges) {
        dynamicAdTargetsAddUpdateHelper.updateDynamicAdTargets(dslContext, clientId, applicableAppliedChanges);
    }
}
