package ru.yandex.direct.core.entity.dynamictextadtarget.service;

import java.math.BigDecimal;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.common.log.container.LogPriceData;
import ru.yandex.direct.common.log.service.LogPriceService;
import ru.yandex.direct.core.entity.bids.container.SetBidItem;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.dynamictextadtarget.model.DynamicFeedAdTarget;
import ru.yandex.direct.core.entity.dynamictextadtarget.repository.DynamicTextAdTargetRepository;
import ru.yandex.direct.core.entity.dynamictextadtarget.service.validation.DynamicTextAdTargetSetBidsValidationService;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.multitype.entity.LimitOffset;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.util.stream.Collectors.toList;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.result.ValidationResult.getValidItems;

@Service
@ParametersAreNonnullByDefault
public class DynamicFeedAdTargetSetBidsService {
    private final ShardHelper shardHelper;
    private final DynamicTextAdTargetRepository dynamicAdTargetRepository;

    private final DynamicTextAdTargetSetBidsValidationService setBidsValidationService;
    private final ClientService clientService;
    private final LogPriceService logPriceService;

    @Autowired
    public DynamicFeedAdTargetSetBidsService(ShardHelper shardHelper,
                                             DynamicTextAdTargetRepository dynamicAdTargetRepository,
                                             DynamicTextAdTargetSetBidsValidationService setBidsValidationService,
                                             ClientService clientService, LogPriceService logPriceService) {
        this.shardHelper = shardHelper;
        this.dynamicAdTargetRepository = dynamicAdTargetRepository;
        this.setBidsValidationService = setBidsValidationService;
        this.clientService = clientService;
        this.logPriceService = logPriceService;
    }

    public MassResult<SetBidItem> setBids(ClientId clientId, Long operatorUid, List<SetBidItem> setBidItemList) {
        if (setBidItemList.isEmpty()) {
            return MassResult.emptyMassAction();
        }

        int shard = shardHelper.getShardByClientIdStrictly(clientId);

        RequestSetBidType requestType = getRequestSetBidType(setBidItemList);

        List<DynamicFeedAdTarget> dynamicFeedAdTargets =
                getDynamicFeedAdTargetsByBid(shard, clientId, setBidItemList, requestType);

        ValidationResult<List<SetBidItem>, Defect> validation = setBidsValidationService
                .validateForFeedAdTargets(shard, operatorUid, clientId, setBidItemList, requestType,
                        dynamicFeedAdTargets);

        List<SetBidItem> filteredSetBids = getValidItems(validation);
        if (validation.hasErrors() || filteredSetBids.isEmpty()) {
            return MassResult.brokenMassAction(validation.getValue(), validation);
        }

        List<AppliedChanges<DynamicFeedAdTarget>> appliedChanges =
                collectDynamicFeedAdTargetChanges(requestType, filteredSetBids, dynamicFeedAdTargets);

        dynamicAdTargetRepository.setBids(shard, appliedChanges);

        logPriceChanges(clientId, operatorUid, appliedChanges);

        return MassResult.successfulMassAction(validation.getValue(), validation);
    }

    public List<DynamicFeedAdTarget> getDynamicFeedAdTargetsByBid(int shard, ClientId clientId,
                                                                  List<SetBidItem> filteredSetBidItems,
                                                                  RequestSetBidType requestType) {
        List<Long> ids = mapList(filteredSetBidItems, requestType::getId);

        switch (requestType) {
            case ID:
                return dynamicAdTargetRepository
                        .getDynamicFeedAdTargets(shard, clientId, ids, true, LimitOffset.maxLimited());
            case ADGROUP_ID:
                return dynamicAdTargetRepository
                        .getDynamicFeedAdTargetsByAdGroupIds(shard, clientId, ids, false, LimitOffset.maxLimited());
            case CAMPAIGN_ID:
                return dynamicAdTargetRepository
                        .getDynamicFeedAdTargetsByCampaignIds(shard, clientId, ids, false, LimitOffset.maxLimited());
            default:
                throw new IllegalStateException("Неизвестный тип для RequestSetBidType");
        }
    }

    private List<AppliedChanges<DynamicFeedAdTarget>> collectDynamicFeedAdTargetChanges(
            RequestSetBidType requestType,
            List<SetBidItem> validBidsToSet,
            List<DynamicFeedAdTarget> dynamicFeedAdTargets) {
        Map<Long, List<DynamicFeedAdTarget>> dynamicFeedAdTargetsById = dynamicFeedAdTargets.stream()
                .collect(Collectors.groupingBy(requestType::getId));

        return StreamEx.of(validBidsToSet)
                .mapToEntry(setBidItem -> getDynamicFeedAdTargetsBySelection(setBidItem, dynamicFeedAdTargetsById,
                        requestType))
                .flatMapValues(Collection::stream)
                .mapKeyValue((setBidItem, dynamicFeedAdTarget) -> applyChanges(setBidItem, dynamicFeedAdTarget))
                .filter(AppliedChanges::hasActuallyChangedProps)
                .collect(toList());
    }

    @Nonnull
    private List<DynamicFeedAdTarget> getDynamicFeedAdTargetsBySelection(
            SetBidItem setBidItem,
            Map<Long, List<DynamicFeedAdTarget>> dynamicFeedAdTargetsById,
            RequestSetBidType requestType) {
        Long id = requestType.getId(setBidItem);
        return dynamicFeedAdTargetsById.get(id);
    }

    private AppliedChanges<DynamicFeedAdTarget> applyChanges(SetBidItem setBidItem,
                                                             DynamicFeedAdTarget dynamicFeedAdTarget) {
        ModelChanges<DynamicFeedAdTarget> changes =
                new ModelChanges<>(dynamicFeedAdTarget.getId(), DynamicFeedAdTarget.class);
        changes.processNotNull(setBidItem.getPriceSearch(), DynamicFeedAdTarget.PRICE);
        changes.processNotNull(setBidItem.getPriceContext(), DynamicFeedAdTarget.PRICE_CONTEXT);
        changes.processNotNull(setBidItem.getAutobudgetPriority(), DynamicFeedAdTarget.AUTOBUDGET_PRIORITY);
        return changes.applyTo(dynamicFeedAdTarget);
    }

    private void logPriceChanges(ClientId clientId, Long operatorUid,
                                 Collection<AppliedChanges<DynamicFeedAdTarget>> appliedChanges) {
        Currency currency = clientService.getWorkCurrency(clientId);

        Function<DynamicFeedAdTarget, LogPriceData> dynamicFeedAdTargetToLog = r -> new LogPriceData(
                r.getCampaignId(),
                r.getAdGroupId(),
                r.getId(),
                nvl(r.getPriceContext(), BigDecimal.ZERO).doubleValue(),
                nvl(r.getPrice(), BigDecimal.ZERO).doubleValue(),
                currency.getCode(),
                LogPriceData.OperationType.DYN_COND_UPDATE);

        List<LogPriceData> logPriceList = StreamEx.of(appliedChanges)
                .filter(ac -> ac.changed(DynamicFeedAdTarget.PRICE) || ac.changed(DynamicFeedAdTarget.PRICE_CONTEXT))
                .map(AppliedChanges::getModel)
                .map(dynamicFeedAdTargetToLog)
                .toList();

        logPriceService.logPrice(logPriceList, operatorUid);
    }

    private RequestSetBidType getRequestSetBidType(List<SetBidItem> bids) {
        //requestType всегда найдется, потому что на это есть проверка в DynamicFeedAdTargetsSetBidsValidationService
        //noinspection OptionalGetWithoutIsPresent
        return Arrays.stream(RequestSetBidType.values())
                .filter(requestType -> bids.stream().allMatch(bid -> requestType.getId(bid) != null))
                .findFirst()
                .get();
    }
}
