package ru.yandex.direct.core.entity.dynamictextadtarget.service;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.jooq.DSLContext;
import org.jooq.TransactionalRunnable;

import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.dynamictextadtarget.model.DynamicFeedAdTarget;
import ru.yandex.direct.core.entity.dynamictextadtarget.service.validation.AddDynamicFeedAdTargetValidationService;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.operation.add.ModelsPreValidatedStep;
import ru.yandex.direct.operation.add.ModelsValidatedStep;
import ru.yandex.direct.operation.add.SimpleAbstractAddOperation;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceProfile;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static com.google.common.base.Preconditions.checkNotNull;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;

public class DynamicFeedAdTargetsAddOperation extends SimpleAbstractAddOperation<DynamicFeedAdTarget, Long> {

    private final ClientId clientId;
    private final Long operatorUid;
    private final int shard;
    private Runnable additionalTask;
    private TransactionalRunnable transactionalAdditionalTask;
    private final Currency clientCurrency;
    private final AddDynamicFeedAdTargetValidationService addValidationService;
    private final AdGroupRepository adGroupRepository;
    private final DslContextProvider dslContextProvider;
    private final DynamicAdTargetsAddOperationHelper helper;


    public DynamicFeedAdTargetsAddOperation(Applicability applicability,
                                            List<DynamicFeedAdTarget> models, ClientId clientId,
                                            Long operatorUid, int shard,
                                            AddDynamicFeedAdTargetValidationService addValidationService,
                                            AdGroupRepository adGroupRepository,
                                            DslContextProvider dslContextProvider,
                                            DynamicAdTargetsAddOperationHelper helper,
                                            ClientService clientService) {
        super(applicability, models);

        models.forEach(mc -> checkNotNull(mc.getAdGroupId(), "AdGroupId must be specified"));

        this.clientId = clientId;
        this.operatorUid = operatorUid;
        this.shard = shard;
        this.addValidationService = addValidationService;
        this.adGroupRepository = adGroupRepository;
        this.dslContextProvider = dslContextProvider;
        this.helper = helper;

        clientCurrency = clientService.getWorkCurrency(clientId);
    }

    @Override
    protected void onPreValidated(ModelsPreValidatedStep<DynamicFeedAdTarget> modelsPreValidatedStep) {
        Collection<DynamicFeedAdTarget> dynamicFeedAdTargets = modelsPreValidatedStep.getPreValidModelsMap().values();

        // перед валидацией необходимо вычислить campaignId по adGroupId
        helper.prepareSystemFields(shard, dynamicFeedAdTargets);
    }

    @Override
    protected void validate(ValidationResult<List<DynamicFeedAdTarget>, Defect> preValidationResult) {
        List<DynamicFeedAdTarget> dynamicFeedAdTargets = preValidationResult.getValue();
        validationResult = addValidationService.validateAdd(shard, operatorUid, clientId, dynamicFeedAdTargets);
    }

    @Override
    protected void onModelsValidated(ModelsValidatedStep<DynamicFeedAdTarget> modelsValidatedStep) {
        List<DynamicFeedAdTarget> dynamicFeedAdTargets = modelsValidatedStep.getModels();
        helper.setMissingPrices(dynamicFeedAdTargets, clientCurrency);
    }

    @Override
    protected void beforeExecution(Map<Integer, DynamicFeedAdTarget> validModelsMapToApply) {
        Collection<DynamicFeedAdTarget> dynamicFeedAdTargets = validModelsMapToApply.values();

        additionalTask = helper.computeLogPriceTask(dynamicFeedAdTargets, operatorUid, clientCurrency);
        transactionalAdditionalTask = helper.computeUpdateStatusesTask(shard, clientId, dynamicFeedAdTargets);
    }

    @Override
    protected List<Long> execute(List<DynamicFeedAdTarget> dynamicFeedAdTargets) {
        List<Long> dynamicFeedAdTargetIds = new ArrayList<>();
        TransactionalRunnable saveFn = conf -> {
            DSLContext dslContext = conf.dsl();

            Set<Long> adGroupIds = listToSet(dynamicFeedAdTargets, DynamicFeedAdTarget::getAdGroupId);
            adGroupRepository.getLockOnAdGroups(conf, adGroupIds);

            List<Long> addedDynamicAdTargetIds = helper.addDynamicAdTargets(
                    dslContext, clientId, dynamicFeedAdTargets);
            dynamicFeedAdTargetIds.addAll(addedDynamicAdTargetIds);

            transactionalAdditionalTask.run(conf);
        };

        try (TraceProfile profile = Trace.current()
                .profile("dynamicFeedAdTargetsAdd:write", "", dynamicFeedAdTargets.size())) {
            dslContextProvider.ppcTransaction(shard, saveFn);
            additionalTask.run();
        }

        return dynamicFeedAdTargetIds;
    }
}
