package ru.yandex.direct.core.entity.dynamictextadtarget.service;

import java.util.Collection;
import java.util.List;
import java.util.Set;

import org.jooq.TransactionalRunnable;

import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.dynamictextadtarget.model.DynamicFeedAdTarget;
import ru.yandex.direct.core.entity.dynamictextadtarget.repository.DynamicTextAdTargetRepository;
import ru.yandex.direct.core.entity.dynamictextadtarget.service.validation.AddDynamicFeedAdTargetValidationService;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.operation.update.ExecutionStep;
import ru.yandex.direct.operation.update.SimpleAbstractUpdateOperation;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceProfile;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

public class DynamicFeedAdTargetsUpdateOperation extends SimpleAbstractUpdateOperation<DynamicFeedAdTarget, Long> {

    private final DynamicAdTargetsUpdateOperationHelper helper;
    private final AddDynamicFeedAdTargetValidationService validationService;
    private final DynamicTextAdTargetRepository dynamicTextAdTargetRepository;
    private final AdGroupRepository adGroupRepository;
    private final DslContextProvider dslContextProvider;

    private final int shard;
    private final ClientId clientId;
    private final Currency clientCurrency;
    private final long operatorUid;

    private Runnable additionalTask;
    private TransactionalRunnable transactionalAdditionalTask;

    public DynamicFeedAdTargetsUpdateOperation(
            Applicability applicability,
            List<ModelChanges<DynamicFeedAdTarget>> modelChangesList,
            DynamicAdTargetsUpdateOperationHelper helper,
            AddDynamicFeedAdTargetValidationService validationService,
            DynamicTextAdTargetRepository dynamicTextAdTargetRepository,
            AdGroupRepository adGroupRepository,
            DslContextProvider dslContextProvider,
            int shard, ClientId clientId, Currency clientCurrency,
            long operatorUid) {
        super(applicability, modelChangesList, id -> new DynamicFeedAdTarget().withId(id));

        this.shard = shard;
        this.clientId = clientId;
        this.clientCurrency = clientCurrency;
        this.operatorUid = operatorUid;

        this.helper = helper;
        this.validationService = validationService;
        this.dynamicTextAdTargetRepository = dynamicTextAdTargetRepository;
        this.adGroupRepository = adGroupRepository;
        this.dslContextProvider = dslContextProvider;
    }

    @Override
    protected ValidationResult<List<ModelChanges<DynamicFeedAdTarget>>, Defect> validateModelChanges(
            List<ModelChanges<DynamicFeedAdTarget>> modelChanges) {
        return validationService.validateModelChanges(shard, clientId, modelChanges);
    }

    @Override
    protected Collection<DynamicFeedAdTarget> getModels(Collection<Long> ids) {
        return dynamicTextAdTargetRepository.getDynamicFeedAdTargetsByIds(shard, clientId, ids);
    }

    @Override
    protected ValidationResult<List<DynamicFeedAdTarget>, Defect> validateAppliedChanges(
            ValidationResult<List<DynamicFeedAdTarget>, Defect> validationResult) {
        return validationService.validateDynamicFeedAdTargets(shard, operatorUid, clientId, validationResult);
    }

    @Override
    protected void beforeExecution(ExecutionStep<DynamicFeedAdTarget> executionStep) {
        Collection<AppliedChanges<DynamicFeedAdTarget>> validAppliedChanges = executionStep.getValidAppliedChanges();

        transactionalAdditionalTask = helper.computeTransactionalTask(shard, clientId, validAppliedChanges);
        additionalTask = helper.computeAdditionalTask(validAppliedChanges, operatorUid, clientCurrency);
    }

    @Override
    protected List<Long> execute(List<AppliedChanges<DynamicFeedAdTarget>> applicableAppliedChanges) {

        TransactionalRunnable saveFn = conf -> {
            Set<Long> affectedAdGroupIds = listToSet(applicableAppliedChanges, ac -> ac.getModel().getAdGroupId());
            adGroupRepository.getLockOnAdGroups(conf, affectedAdGroupIds);

            helper.updateDynamicAdTargets(conf.dsl(), clientId, applicableAppliedChanges);
            transactionalAdditionalTask.run(conf);
        };

        try (TraceProfile profile = Trace.current()
                .profile("dynamicFeedAdTargetsUpdate:write", "", applicableAppliedChanges.size())) {
            dslContextProvider.ppcTransaction(shard, saveFn);
            additionalTask.run();
        }
        return mapList(applicableAppliedChanges, ac -> ac.getModel().getId());
    }
}
