package ru.yandex.direct.core.entity.dynamictextadtarget.service;

import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.copyentity.CopyOperationContainer;
import ru.yandex.direct.core.copyentity.EntityService;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignSubObjectAccessChecker;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignSubObjectAccessCheckerFactory;
import ru.yandex.direct.core.entity.dynamictextadtarget.container.DynamicTextAdTargetSelectionCriteria;
import ru.yandex.direct.core.entity.dynamictextadtarget.model.DynamicAdTarget;
import ru.yandex.direct.core.entity.dynamictextadtarget.model.DynamicAdTargetsQueryFilter;
import ru.yandex.direct.core.entity.dynamictextadtarget.model.DynamicFeedAdTarget;
import ru.yandex.direct.core.entity.dynamictextadtarget.model.DynamicTextAdTarget;
import ru.yandex.direct.core.entity.dynamictextadtarget.repository.DynamicTextAdTargetRepository;
import ru.yandex.direct.core.entity.performancefilter.model.Operator;
import ru.yandex.direct.dbutil.model.BusinessIdAndShopId;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.multitype.entity.LimitOffset;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.operation.Operation;
import ru.yandex.direct.operation.aggregator.SplitAndMergeOperationAggregator;
import ru.yandex.direct.operation.creator.OperationCreator;
import ru.yandex.direct.result.MassResult;

import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static java.util.stream.Collectors.toSet;
import static ru.yandex.direct.operation.aggregator.SplitAndMergeOperationAggregator.getMassResultWithAggregatedValidationResult;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Service
@ParametersAreNonnullByDefault
public class DynamicTextAdTargetService implements EntityService<DynamicAdTarget, Long> {

    private final ShardHelper shardHelper;
    private final DynamicTextAdTargetRepository dynamicTextAdTargetRepository;
    private final CampaignSubObjectAccessCheckerFactory campaignSubObjectAccessCheckerFactory;
    private final DynamicTextAdTargetsDeleteOperationFactory dynamicTextAdTargetsDeleteOperationFactory;
    private final DynamicAdTargetsAddOperationFactory dynamicAdTargetsAddOperationFactory;

    @Autowired
    public DynamicTextAdTargetService(ShardHelper shardHelper,
                                      DynamicTextAdTargetRepository dynamicTextAdTargetRepository,
                                      CampaignSubObjectAccessCheckerFactory campaignSubObjectAccessCheckerFactory,
                                      DynamicTextAdTargetsDeleteOperationFactory dynamicTextAdTargetsDeleteOperationFactory,
                                      DynamicAdTargetsAddOperationFactory dynamicAdTargetsAddOperationFactory) {
        this.shardHelper = shardHelper;
        this.dynamicTextAdTargetRepository = dynamicTextAdTargetRepository;
        this.campaignSubObjectAccessCheckerFactory = campaignSubObjectAccessCheckerFactory;
        this.dynamicTextAdTargetsDeleteOperationFactory = dynamicTextAdTargetsDeleteOperationFactory;
        this.dynamicAdTargetsAddOperationFactory = dynamicAdTargetsAddOperationFactory;
    }

    public List<DynamicTextAdTarget> getDynamicTextAdTargets(ClientId clientId, Long operatorUid,
                                                             DynamicTextAdTargetSelectionCriteria selectionCriteria,
                                                             LimitOffset limitOffset) {
        if (isSelectionCriteriaEmpty(selectionCriteria)) {
            return emptyList();
        }

        int shard = shardHelper.getShardByClientIdStrictly(clientId);

        Set<Long> conditionIds = getConditionIds(shard, clientId, operatorUid, selectionCriteria);

        return dynamicTextAdTargetRepository
                .getDynamicTextAdTargetsWithDomainType(shard, clientId, conditionIds, true, limitOffset);
    }

    public List<DynamicFeedAdTarget> getDynamicFeedAdTargets(ClientId clientId, Long operatorUid,
                                                             DynamicTextAdTargetSelectionCriteria selectionCriteria,
                                                             LimitOffset limitOffset) {
        if (isSelectionCriteriaEmpty(selectionCriteria)) {
            return emptyList();
        }
        int shard = shardHelper.getShardByClientIdStrictly(clientId);

        Set<Long> conditionIds = getConditionIds(shard, clientId, operatorUid, selectionCriteria);

        List<DynamicFeedAdTarget> dynamicAdTargets = dynamicTextAdTargetRepository
                .getDynamicFeedAdTargets(shard, clientId, conditionIds, true, limitOffset);

        return mapList(dynamicAdTargets, this::removeDefaultFilterCondition);
    }

    public Map<BusinessIdAndShopId, List<DynamicFeedAdTarget>> getDynamicFeedAdTargetsByBusinessIdAndShopId(
            ClientId clientId, Long operatorUid, DynamicTextAdTargetSelectionCriteria selectionCriteria) {
        if (isSelectionCriteriaEmpty(selectionCriteria)) {
            return emptyMap();
        }

        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        Set<Long> conditionIds = getConditionIds(shard, clientId, operatorUid, selectionCriteria);
        return dynamicTextAdTargetRepository
                .getDynamicFeedAdTargetsByBusinessIdAndShopId(shard, clientId, conditionIds, true);
    }

    private boolean isSelectionCriteriaEmpty(DynamicTextAdTargetSelectionCriteria selectionCriteria) {
        return selectionCriteria.getAdGroupIds().isEmpty()
                && selectionCriteria.getCampaignIds().isEmpty()
                && selectionCriteria.getBusinessIdsAndShopIds().isEmpty()
                && selectionCriteria.getStates().isEmpty()
                && selectionCriteria.getConditionIds().isEmpty();
    }

    private Set<Long> getConditionIds(int shard, ClientId clientId, Long operatorUid,
                                      DynamicTextAdTargetSelectionCriteria selectionCriteria) {
        Map<Long, List<Long>> conditionIdsByCampaignIds =
                dynamicTextAdTargetRepository.getCampaignIdToConditionIdsBySelectionCriteria(shard, selectionCriteria);

        Set<Long> campaignIds = conditionIdsByCampaignIds.keySet();

        CampaignSubObjectAccessChecker<?> accessChecker = campaignSubObjectAccessCheckerFactory
                .newCampaignChecker(operatorUid, clientId, campaignIds);

        Set<Long> filteredCampaignIds = campaignIds.stream()
                .filter(accessChecker::objectInVisibleCampaign)
                .collect(toSet());

        return EntryStream.of(conditionIdsByCampaignIds)
                .filterKeys(filteredCampaignIds::contains)
                .values()
                .toFlatCollection(Function.identity(), HashSet::new);
    }

    public List<DynamicAdTarget> getDynamicAdTargets(ClientId clientId, DynamicAdTargetsQueryFilter queryFilter) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        List<DynamicAdTarget> dynamicAdTargets =
                dynamicTextAdTargetRepository.getDynamicAdTargetsByQueryFilter(shard, clientId, queryFilter);

        return mapList(dynamicAdTargets, target -> target instanceof DynamicFeedAdTarget
                ? removeDefaultFilterCondition((DynamicFeedAdTarget) target)
                : target);
    }

    public List<DynamicFeedAdTarget> getDynamicFeedAdTargetsByIds(ClientId clientId, Collection<Long> ids) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        List<DynamicFeedAdTarget> dynamicAdTargets =
                dynamicTextAdTargetRepository.getDynamicFeedAdTargetsByIds(shard, clientId, ids);

        return mapList(dynamicAdTargets, this::removeDefaultFilterCondition);
    }

    // DIRECT-166949: Убираем из ответа дефолтное условие для фильтров ТК
    private DynamicFeedAdTarget removeDefaultFilterCondition(DynamicFeedAdTarget adTarget) {
        return adTarget.withCondition(filterList(adTarget.getCondition(),
                c -> !(c.getFieldName().equals("available") && c.getOperator() == Operator.NOT_EQUALS)));
    }

    public Map<Long, Long> getIdByDynamicConditionId(ClientId clientId, Collection<Long> dynamicConditionIds) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        return dynamicTextAdTargetRepository.getIdByDynamicConditionId(shard, dynamicConditionIds);
    }

    public Map<Long, Long> getDynamicConditionIdById(ClientId clientId, Collection<Long> ids) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);
        return dynamicTextAdTargetRepository.getDynamicConditionIdById(shard, ids);
    }

    /**
     * Массовое удаление условий для клиента.
     * Считается успешным, если удалено хотя бы одно объявление.
     *
     * @param clientId           id клиента
     * @param dynamicAdTargetIds удаляемые id условий
     * @return результат удаления условий
     */
    public MassResult<Long> deleteDynamicAdTargets(long operatorUid, ClientId clientId,
                                                   List<Long> dynamicAdTargetIds) {

        DynamicTextAdTargetsDeleteOperation deleteOperation = dynamicTextAdTargetsDeleteOperationFactory
                .createDeleteOperation(operatorUid, clientId, dynamicAdTargetIds);

        return deleteOperation.prepareAndApply();
    }

    public MassResult<Long> addDynamicTextAdTargets(ClientId clientId, Long operatorUid,
                                                    List<DynamicTextAdTarget> dynamicTextAdTargets) {
        return dynamicAdTargetsAddOperationFactory
                .createDynamicTextAdTargetsAddOperation(clientId, operatorUid, dynamicTextAdTargets)
                .prepareAndApply();
    }

    public MassResult<Long> addDynamicFeedAdTargets(ClientId clientId, Long operatorUid,
                                                    List<DynamicFeedAdTarget> dynamicFeedAdTargets) {
        return dynamicAdTargetsAddOperationFactory
                .createDynamicFeedAdTargetsAddOperation(clientId, operatorUid, dynamicFeedAdTargets)
                .prepareAndApply();
    }

    public MassResult<Long> updateDynamicTextAdTargets(ClientId clientId, long operatorUid,
                                                       List<ModelChanges<DynamicTextAdTarget>> modelChangesList) {
        return dynamicAdTargetsAddOperationFactory
                .createDynamicTextAdTargetsUpdateOperation(clientId, operatorUid, modelChangesList)
                .prepareAndApply();
    }

    public MassResult<Long> updateDynamicFeedAdTargets(ClientId clientId, long operatorUid,
                                                       List<ModelChanges<DynamicFeedAdTarget>> modelChangesList) {
        return dynamicAdTargetsAddOperationFactory
                .createDynamicFeedAdTargetsUpdateOperation(clientId, operatorUid, modelChangesList)
                .prepareAndApply();
    }

    @Override
    public List<DynamicAdTarget> get(ClientId clientId, Long operatorUid, Collection<Long> ids) {
        DynamicAdTargetsQueryFilter queryFilter = new DynamicAdTargetsQueryFilter()
                .withIds(new HashSet<>(ids))
                .withIncludeDeleted(false);
        return getDynamicAdTargets(clientId, queryFilter);
    }

    @Override
    public MassResult<Long> add(
            ClientId clientId, Long operatorUid, List<DynamicAdTarget> entities, Applicability applicability) {

        OperationCreator<DynamicAdTarget, Operation<Long>> addDynamicTextAdTargetsOperationCreator = inputItems -> {
            List<DynamicTextAdTarget> dynamicTextAdTargets = mapList(inputItems, d -> (DynamicTextAdTarget) d);
            return dynamicAdTargetsAddOperationFactory
                    .createDynamicTextAdTargetsAddOperation(clientId, operatorUid, dynamicTextAdTargets);
        };

        OperationCreator<DynamicAdTarget, Operation<Long>> addDynamicFeedAdTargetsOperationCreator = inputItems -> {
            List<DynamicFeedAdTarget> dynamicFeedAdTargets = mapList(inputItems, d -> (DynamicFeedAdTarget) d);
            return dynamicAdTargetsAddOperationFactory
                    .createDynamicFeedAdTargetsAddOperation(clientId, operatorUid, dynamicFeedAdTargets);
        };

        SplitAndMergeOperationAggregator<DynamicAdTarget, Long> addOperation =
                SplitAndMergeOperationAggregator.builderForPartialOperations()
                        .addSubOperation(
                                d -> d instanceof DynamicTextAdTarget,
                                addDynamicTextAdTargetsOperationCreator)
                        .addSubOperation(
                                d -> d instanceof DynamicFeedAdTarget,
                                addDynamicFeedAdTargetsOperationCreator)
                        .build();

        MassResult<Long> aggregateOperationResult = addOperation.execute(entities);
        return getMassResultWithAggregatedValidationResult(aggregateOperationResult);
    }

    @Override
    public MassResult<Long> copy(CopyOperationContainer copyContainer,
                                 List<DynamicAdTarget> entities, Applicability applicability) {
        return add(copyContainer.getClientIdTo(), copyContainer.getOperatorUid(), entities, applicability);
    }
}
