package ru.yandex.direct.core.entity.dynamictextadtarget.service;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.jooq.DSLContext;
import org.jooq.TransactionalRunnable;

import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.dynamictextadtarget.model.DynamicTextAdTarget;
import ru.yandex.direct.core.entity.dynamictextadtarget.service.validation.AddDynamicTextAdTargetValidationService;
import ru.yandex.direct.core.entity.dynamictextadtarget.service.validation.DynamicTextAdTargetConstants;
import ru.yandex.direct.core.entity.dynamictextadtarget.utils.DynamicTextAdTargetHashUtils;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.operation.add.ModelsPreValidatedStep;
import ru.yandex.direct.operation.add.ModelsValidatedStep;
import ru.yandex.direct.operation.add.SimpleAbstractAddOperation;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceProfile;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static com.google.common.base.Preconditions.checkNotNull;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;

public class DynamicTextAdTargetsAddOperation extends SimpleAbstractAddOperation<DynamicTextAdTarget, Long> {

    private final AddDynamicTextAdTargetValidationService addValidationService;
    private final ClientId clientId;
    private final Long operatorUid;
    private final int shard;
    private Runnable additionalTask;
    private TransactionalRunnable transactionalAdditionalTask;
    private final Currency clientCurrency;
    private final AdGroupRepository adGroupRepository;
    private final DslContextProvider dslContextProvider;
    private final DynamicAdTargetsAddOperationHelper helper;


    public DynamicTextAdTargetsAddOperation(Applicability applicability,
                                            List<DynamicTextAdTarget> models, ClientId clientId,
                                            Long operatorUid, int shard,
                                            AddDynamicTextAdTargetValidationService addValidationService,
                                            AdGroupRepository adGroupRepository,
                                            DslContextProvider dslContextProvider,
                                            DynamicAdTargetsAddOperationHelper helper,
                                            ClientService clientService) {
        super(applicability, models);

        // при использовании из api5 выполняется всегда, т.к. там поле NonNull
        // для других потребителей такой гарантии нет, сделаем проверку
        models.forEach(mc -> {
            checkNotNull(mc.getAdGroupId(), "AdGroupId must be specified");
        });

        this.clientId = clientId;
        this.operatorUid = operatorUid;
        this.shard = shard;
        this.addValidationService = addValidationService;
        this.adGroupRepository = adGroupRepository;
        this.dslContextProvider = dslContextProvider;
        this.helper = helper;

        clientCurrency = clientService.getWorkCurrency(clientId);
    }

    @Override
    protected void onPreValidated(ModelsPreValidatedStep<DynamicTextAdTarget> modelsPreValidatedStep) {
        Collection<DynamicTextAdTarget> dynamicTextAdTargets = modelsPreValidatedStep.getPreValidModelsMap().values();

        // перед валидацией необходимо вычислить campaignId по adGroupId
        helper.prepareSystemFields(shard, dynamicTextAdTargets);
        // и проставить дефолтное условие ALL_PAGE_CONDITION, если не задано никакое
        fillInDefaultValues(dynamicTextAdTargets);
    }

    public void fillInDefaultValues(Collection<DynamicTextAdTarget> dynamicTextAdTargets) {
        dynamicTextAdTargets.forEach(dynamicTextAdTarget -> {
            if (dynamicTextAdTarget.getCondition().isEmpty()) {
                dynamicTextAdTarget
                        .withCondition(DynamicTextAdTargetConstants.ALL_PAGE_CONDITION)
                        .withConditionHash(DynamicTextAdTargetHashUtils.getHash(
                                DynamicTextAdTargetConstants.ALL_PAGE_CONDITION))
                        .withConditionUniqHash(DynamicTextAdTargetConstants.ALL_PAGE_CONDITION_UNIQ_HASH);
            }
        });
    }

    @Override
    protected void validate(ValidationResult<List<DynamicTextAdTarget>, Defect> preValidationResult) {
        List<DynamicTextAdTarget> dynamicTextAdTargetsList = preValidationResult.getValue();
        validationResult = addValidationService.validateAdd(shard, operatorUid, clientId, dynamicTextAdTargetsList);
    }


    @Override
    protected void onModelsValidated(ModelsValidatedStep<DynamicTextAdTarget> modelsValidatedStep) {
        List<DynamicTextAdTarget> dynamicTextAdTargets = modelsValidatedStep.getModels();
        helper.setMissingPrices(dynamicTextAdTargets, clientCurrency);
    }

    @Override
    protected void beforeExecution(Map<Integer, DynamicTextAdTarget> validModelsMapToApply) {
        Collection<DynamicTextAdTarget> dynamicTextAdTargets = validModelsMapToApply.values();

        additionalTask = helper.computeLogPriceTask(dynamicTextAdTargets, operatorUid, clientCurrency);
        transactionalAdditionalTask = helper.computeUpdateStatusesTask(shard, clientId, dynamicTextAdTargets);
    }

    @Override
    protected List<Long> execute(List<DynamicTextAdTarget> dynamicTextAdTargets) {
        List<Long> dynamicTextAdTargetIds = new ArrayList<>();
        TransactionalRunnable saveFn = conf -> {
            DSLContext dslContext = conf.dsl();

            Set<Long> adGroupIds = listToSet(dynamicTextAdTargets, DynamicTextAdTarget::getAdGroupId);
            adGroupRepository.getLockOnAdGroups(conf, adGroupIds);

            List<Long> addedDynamicAdTargetIds = helper.addDynamicAdTargets(
                    dslContext, clientId, dynamicTextAdTargets);
            dynamicTextAdTargetIds.addAll(addedDynamicAdTargetIds);

            transactionalAdditionalTask.run(conf);
        };

        try (TraceProfile profile = Trace.current()
                .profile("dynamicTextAdTargetsAdd:write", "", dynamicTextAdTargets.size())) {
            dslContextProvider.ppcTransaction(shard, saveFn);
            additionalTask.run();
        }

        return dynamicTextAdTargetIds;
    }
}
