package ru.yandex.direct.core.entity.dynamictextadtarget.service;

import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.jooq.DSLContext;

import ru.yandex.direct.core.entity.StatusBsSynced;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupSimple;
import ru.yandex.direct.core.entity.adgroup.model.StatusModerate;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.banner.repository.BannerCommonRepository;
import ru.yandex.direct.core.entity.dynamictextadtarget.model.DynamicAdTarget;
import ru.yandex.direct.core.entity.dynamictextadtarget.repository.DynamicTextAdTargetRepository;
import ru.yandex.direct.core.entity.dynamictextadtarget.service.validation.DeleteDynamicTextAdTargetValidationService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.operation.operationwithid.AbstractOperationWithId;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.utils.FunctionalUtils.mapList;

public class DynamicTextAdTargetsDeleteOperation extends AbstractOperationWithId {

    private final DslContextProvider ppcDslContextProvider;
    private final DeleteDynamicTextAdTargetValidationService deleteDynamicTextAdTargetValidationService;
    private final DynamicTextAdTargetRepository dynamicTextAdTargetRepository;
    private final AdGroupRepository adGroupRepository;
    private final BannerCommonRepository bannerCommonRepository;
    private final int shard;
    private final ClientId clientId;
    private final long operatorUid;

    private List<Long> adGroupIdsModerateStatusNotNew;
    private List<Long> adGroupIdsModerateStatusIsNew;

    public DynamicTextAdTargetsDeleteOperation(int shard, ClientId clientId,
                                               long operatorUid, Applicability applicability, List<Long> modelIds,
                                               DslContextProvider ppcDslContextProvider,
                                               DeleteDynamicTextAdTargetValidationService deleteDynamicTextAdTargetValidationService,
                                               DynamicTextAdTargetRepository dynamicTextAdTargetRepository,
                                               AdGroupRepository adGroupRepository,
                                               BannerCommonRepository bannerCommonRepository) {
        super(applicability, modelIds);
        this.shard = shard;
        this.clientId = clientId;
        this.operatorUid = operatorUid;
        this.ppcDslContextProvider = ppcDslContextProvider;
        this.deleteDynamicTextAdTargetValidationService = deleteDynamicTextAdTargetValidationService;
        this.dynamicTextAdTargetRepository = dynamicTextAdTargetRepository;
        this.adGroupRepository = adGroupRepository;
        this.bannerCommonRepository = bannerCommonRepository;
    }

    @Override
    protected ValidationResult<List<Long>, Defect> validate(List<Long> dynamicTextAdTargetIds) {
        return deleteDynamicTextAdTargetValidationService
                .validateDelete(operatorUid, clientId, dynamicTextAdTargetIds);
    }

    @Override
    protected void beforeExecution(List<Long> dynamicTextAdTargetIds) {
        List<DynamicAdTarget> dynamicAdTargets = dynamicTextAdTargetRepository
                .getDynamicAdTargetsByDynamicConditionIds(shard, clientId, dynamicTextAdTargetIds);

        Map<Long, AdGroupSimple> adGroups = adGroupRepository
                .getAdGroupSimple(shard, clientId, mapList(dynamicAdTargets, DynamicAdTarget::getAdGroupId));


        List<Long> adGroupIds = adGroups.values()
                .stream()
                .map(AdGroupSimple::getId)
                .collect(Collectors.toList());

        this.adGroupIdsModerateStatusNotNew = adGroupIds
                .stream()
                .filter(adGroupId -> adGroups.get(adGroupId).getStatusModerate() != StatusModerate.NEW)
                .collect(Collectors.toList());

        this.adGroupIdsModerateStatusIsNew = adGroupIds
                .stream()
                .filter(adGroupId -> adGroups.get(adGroupId).getStatusModerate() == StatusModerate.NEW)
                .collect(Collectors.toList());
    }

    @Override
    protected void execute(List<Long> dynamicTextAdTargetIds) {
        if (dynamicTextAdTargetIds.isEmpty()) {
            return;
        }

        ppcDslContextProvider.ppcTransaction(shard, config -> {
            DSLContext context = config.dsl();

            dynamicTextAdTargetRepository.deleteDynamicTextAdTargets(context, dynamicTextAdTargetIds);

            adGroupRepository
                    .updateStatusBsSyncedWithLastChange(config, adGroupIdsModerateStatusNotNew, StatusBsSynced.NO);
            adGroupRepository.updateLastChange(config, adGroupIdsModerateStatusIsNew);

            bannerCommonRepository
                    .updateStatusBsSyncedByAdgroupId(context, adGroupIdsModerateStatusNotNew, StatusBsSynced.NO);
        });
    }
}
