package ru.yandex.direct.core.entity.dynamictextadtarget.service;

import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.jooq.DSLContext;

import ru.yandex.direct.core.entity.StatusBsSynced;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupSimple;
import ru.yandex.direct.core.entity.adgroup.model.StatusModerate;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.banner.repository.BannerCommonRepository;
import ru.yandex.direct.core.entity.dynamictextadtarget.model.DynamicTextAdTarget;
import ru.yandex.direct.core.entity.dynamictextadtarget.repository.DynamicTextAdTargetRepository;
import ru.yandex.direct.core.entity.dynamictextadtarget.service.validation.SuspendResumeDynamicTextAdTargetValidationService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.multitype.entity.LimitOffset;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.operation.operationwithid.AbstractOperationWithId;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.utils.FunctionalUtils.mapList;

public class DynamicTextAdTargetsSuspendResumeOperation extends AbstractOperationWithId {

    private final int shard;
    private final ClientId clientId;
    private final long operatorUid;
    private final boolean isResumeOperation;

    private final SuspendResumeDynamicTextAdTargetValidationService validationService;
    private final DynamicTextAdTargetRepository dynamicTextAdTargetRepository;
    private final AdGroupRepository adGroupRepository;
    private final BannerCommonRepository bannerCommonRepository;
    private final DslContextProvider ppcDslContextProvider;

    private List<Long> adGroupIds;
    private List<Long> adGroupIdsModerateStatusNotNew;

    public DynamicTextAdTargetsSuspendResumeOperation(int shard, ClientId clientId,
                                                      long operatorUid, List<Long> ids,
                                                      boolean isResumeOperation,
                                                      SuspendResumeDynamicTextAdTargetValidationService validationService,
                                                      DynamicTextAdTargetRepository dynamicTextAdTargetRepository,
                                                      AdGroupRepository adGroupRepository,
                                                      BannerCommonRepository bannerCommonRepository,
                                                      DslContextProvider ppcDslContextProvider) {
        super(Applicability.PARTIAL, ids);
        this.shard = shard;
        this.clientId = clientId;
        this.operatorUid = operatorUid;
        this.isResumeOperation = isResumeOperation;
        this.validationService = validationService;
        this.dynamicTextAdTargetRepository = dynamicTextAdTargetRepository;
        this.adGroupRepository = adGroupRepository;
        this.bannerCommonRepository = bannerCommonRepository;
        this.ppcDslContextProvider = ppcDslContextProvider;
    }

    @Override
    protected ValidationResult<List<Long>, Defect> validate(List<Long> ids) {
        return validationService.validate(shard, clientId, operatorUid, ids, isResumeOperation);
    }

    @Override
    protected void beforeExecution(List<Long> ids) {
        List<DynamicTextAdTarget> dynamicTextAdTargets = dynamicTextAdTargetRepository
                .getDynamicTextAdTargetsWithDomainType(shard, clientId, ids, true, LimitOffset.maxLimited());

        Map<Long, AdGroupSimple> adGroups = adGroupRepository
                .getAdGroupSimple(shard, clientId, mapList(dynamicTextAdTargets, DynamicTextAdTarget::getAdGroupId));

        this.adGroupIds = mapList(adGroups.values(), AdGroupSimple::getId);

        this.adGroupIdsModerateStatusNotNew = adGroupIds
                .stream()
                .filter(adGroupId -> adGroups.get(adGroupId).getStatusModerate() != StatusModerate.NEW)
                .collect(Collectors.toList());
    }

    @Override
    protected void execute(List<Long> ids) {
        ppcDslContextProvider.ppcTransaction(shard, config -> {
            DSLContext context = config.dsl();

            dynamicTextAdTargetRepository.updateSuspended(context, ids, !isResumeOperation);

            adGroupRepository
                    .updateStatusBsSyncedWithLastChange(config, adGroupIdsModerateStatusNotNew, StatusBsSynced.NO);
            adGroupRepository.updateLastChange(config, adGroupIds);

            bannerCommonRepository
                    .updateStatusBsSyncedByAdgroupId(context, adGroupIdsModerateStatusNotNew, StatusBsSynced.NO);

            if (isResumeOperation) {
                adGroupRepository.setStatusBlGeneratedForDynamicAdGroupsToProcessingIfNo(context,
                        adGroupIdsModerateStatusNotNew);
            }
        });
    }
}
