package ru.yandex.direct.core.entity.dynamictextadtarget.service;

import java.util.Collection;
import java.util.List;
import java.util.Set;

import org.jooq.TransactionalRunnable;

import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.dynamictextadtarget.model.DynamicTextAdTarget;
import ru.yandex.direct.core.entity.dynamictextadtarget.repository.DynamicTextAdTargetRepository;
import ru.yandex.direct.core.entity.dynamictextadtarget.service.validation.AddDynamicTextAdTargetValidationService;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.operation.Applicability;
import ru.yandex.direct.operation.update.ExecutionStep;
import ru.yandex.direct.operation.update.SimpleAbstractUpdateOperation;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceProfile;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

public class DynamicTextAdTargetsUpdateOperation extends SimpleAbstractUpdateOperation<DynamicTextAdTarget, Long> {

    private final DynamicAdTargetsUpdateOperationHelper helper;
    private final AddDynamicTextAdTargetValidationService validationService;
    private final DynamicTextAdTargetRepository dynamicTextAdTargetRepository;
    private final AdGroupRepository adGroupRepository;
    private final DslContextProvider dslContextProvider;

    private final int shard;
    private final ClientId clientId;
    private final Currency clientCurrency;
    private final long operatorUid;

    private Runnable additionalTask;
    private TransactionalRunnable transactionalAdditionalTask;

    public DynamicTextAdTargetsUpdateOperation(
            Applicability applicability,
            List<ModelChanges<DynamicTextAdTarget>> modelChangesList,
            DynamicAdTargetsUpdateOperationHelper helper,
            AddDynamicTextAdTargetValidationService validationService,
            DynamicTextAdTargetRepository dynamicTextAdTargetRepository,
            AdGroupRepository adGroupRepository,
            DslContextProvider dslContextProvider,
            int shard, ClientId clientId, Currency clientCurrency,
            long operatorUid) {
        super(applicability, modelChangesList, id -> new DynamicTextAdTarget().withId(id));

        this.shard = shard;
        this.clientId = clientId;
        this.clientCurrency = clientCurrency;
        this.operatorUid = operatorUid;

        this.helper = helper;
        this.validationService = validationService;
        this.dynamicTextAdTargetRepository = dynamicTextAdTargetRepository;
        this.adGroupRepository = adGroupRepository;
        this.dslContextProvider = dslContextProvider;
    }

    @Override
    protected ValidationResult<List<ModelChanges<DynamicTextAdTarget>>, Defect> validateModelChanges(
            List<ModelChanges<DynamicTextAdTarget>> modelChanges) {
        return validationService.validateModelChanges(shard, clientId, modelChanges);
    }

    @Override
    protected Collection<DynamicTextAdTarget> getModels(Collection<Long> ids) {
        return dynamicTextAdTargetRepository.getDynamicTextAdTargetsByIds(shard, clientId, ids);
    }

    @Override
    protected ValidationResult<List<DynamicTextAdTarget>, Defect> validateAppliedChanges(
            ValidationResult<List<DynamicTextAdTarget>, Defect> validationResult) {
        return validationService.validateDynamicTextAdTargets(shard, operatorUid, clientId, validationResult);
    }

    @Override
    protected void beforeExecution(ExecutionStep<DynamicTextAdTarget> executionStep) {
        Collection<AppliedChanges<DynamicTextAdTarget>> validAppliedChanges = executionStep.getValidAppliedChanges();

        transactionalAdditionalTask = helper.computeTransactionalTask(shard, clientId, validAppliedChanges);
        additionalTask = helper.computeAdditionalTask(validAppliedChanges, operatorUid, clientCurrency);
    }

    @Override
    protected List<Long> execute(List<AppliedChanges<DynamicTextAdTarget>> applicableAppliedChanges) {

        TransactionalRunnable saveFn = conf -> {
            Set<Long> affectedAdGroupIds = listToSet(applicableAppliedChanges, ac -> ac.getModel().getAdGroupId());
            adGroupRepository.getLockOnAdGroups(conf, affectedAdGroupIds);

            helper.updateDynamicAdTargets(conf.dsl(), clientId, applicableAppliedChanges);
            transactionalAdditionalTask.run(conf);
        };

        try (TraceProfile profile = Trace.current()
                .profile("dynamicTextAdTargetsUpdate:write", "", applicableAppliedChanges.size())) {
            dslContextProvider.ppcTransaction(shard, saveFn);
            additionalTask.run();
        }
        return mapList(applicableAppliedChanges, ac -> ac.getModel().getId());
    }
}
