package ru.yandex.direct.core.entity.dynamictextadtarget.service.validation;

import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignSubObjectAccessCheckerFactory;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignSubObjectAccessConstraint;
import ru.yandex.direct.core.entity.campaign.service.validation.CampaignAccessType;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.dynamictextadtarget.model.DynamicAdTarget;
import ru.yandex.direct.core.entity.dynamictextadtarget.model.DynamicAdTargetTab;
import ru.yandex.direct.core.entity.dynamictextadtarget.model.DynamicFeedAdTarget;
import ru.yandex.direct.core.entity.dynamictextadtarget.model.DynamicFeedRule;
import ru.yandex.direct.core.entity.dynamictextadtarget.repository.DynamicTextAdTargetRepository;
import ru.yandex.direct.core.entity.feed.model.Feed;
import ru.yandex.direct.core.entity.feed.service.FeedService;
import ru.yandex.direct.core.entity.performancefilter.schema.FilterSchema;
import ru.yandex.direct.core.entity.performancefilter.service.PerformanceFilterStorage;
import ru.yandex.direct.core.entity.performancefilter.utils.PerformanceFilterUtils;
import ru.yandex.direct.core.entity.performancefilter.validation.FilterConditionsValidator;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.util.ModelChangesValidationTool;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static java.util.stream.Collectors.toSet;
import static ru.yandex.direct.core.entity.dynamictextadtarget.service.validation.AddDynamicTextAdTargetValidationService.ACCESS_DEFECTS;
import static ru.yandex.direct.core.entity.dynamictextadtarget.service.validation.DynamicAdTargetValidationHelper.adGroupIdValidator;
import static ru.yandex.direct.core.entity.dynamictextadtarget.service.validation.DynamicAdTargetValidationHelper.bidsValidator;
import static ru.yandex.direct.core.entity.dynamictextadtarget.service.validation.DynamicAdTargetValidationHelper.dynamicAdTargetNoMoreThanMax;
import static ru.yandex.direct.core.entity.dynamictextadtarget.service.validation.DynamicAdTargetValidationHelper.nameValidator;
import static ru.yandex.direct.core.entity.dynamictextadtarget.service.validation.DynamicTextAdTargetConstants.ALLOWED_DYNAMIC_AD_GROUP_TYPES;
import static ru.yandex.direct.core.entity.dynamictextadtarget.service.validation.DynamicTextAdTargetDefects.notAcceptableAdGroupType;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.constraint.CommonConstraints.inSet;
import static ru.yandex.direct.validation.defect.CollectionDefects.duplicatedObject;
import static ru.yandex.direct.validation.defect.CommonDefects.inconsistentState;


@Service
@ParametersAreNonnullByDefault
public class AddDynamicFeedAdTargetValidationService {

    private final CampaignSubObjectAccessCheckerFactory campaignSubObjectAccessCheckerFactory;
    private final ClientService clientService;
    private final CampaignRepository campaignRepository;
    private final AdGroupRepository adGroupRepository;
    private final DynamicTextAdTargetRepository dynamicTextAdTargetRepository;
    private final FeedService feedService;
    private final PerformanceFilterStorage filterSchemaStorage;
    private final ModelChangesValidationTool updateValidationTool;

    @Autowired
    public AddDynamicFeedAdTargetValidationService(CampaignSubObjectAccessCheckerFactory campaignSubObjectAccessCheckerFactory,
                                                   ClientService clientService,
                                                   CampaignRepository campaignRepository,
                                                   AdGroupRepository adGroupRepository,
                                                   DynamicTextAdTargetRepository dynamicTextAdTargetRepository,
                                                   FeedService feedService,
                                                   PerformanceFilterStorage filterSchemaStorage) {
        this.campaignSubObjectAccessCheckerFactory = campaignSubObjectAccessCheckerFactory;
        this.clientService = clientService;
        this.campaignRepository = campaignRepository;
        this.adGroupRepository = adGroupRepository;
        this.dynamicTextAdTargetRepository = dynamicTextAdTargetRepository;
        this.feedService = feedService;
        this.filterSchemaStorage = filterSchemaStorage;
        this.updateValidationTool = ModelChangesValidationTool.builder().build();
    }

    public ValidationResult<List<ModelChanges<DynamicFeedAdTarget>>, Defect> validateModelChanges(
            int shard, ClientId clientId, List<ModelChanges<DynamicFeedAdTarget>> changes) {
        Set<Long> ids = listToSet(changes, ModelChanges::getId);

        List<DynamicFeedAdTarget> existingDynamicAdTargets =
                dynamicTextAdTargetRepository.getDynamicFeedAdTargetsByIds(shard, clientId, ids);
        Set<Long> existingIds = listToSet(existingDynamicAdTargets, DynamicAdTarget::getId);

        return updateValidationTool.validateModelChangesList(changes, existingIds);
    }

    public ValidationResult<List<DynamicFeedAdTarget>, Defect> validateAdd(
            int shard, Long operatorUid, ClientId clientId, List<DynamicFeedAdTarget> dynamicAdTargets) {
        ValidationResult<List<DynamicFeedAdTarget>, Defect> validationResult = new ValidationResult<>(dynamicAdTargets);
        return validateDynamicFeedAdTargets(shard, operatorUid, clientId, validationResult);
    }

    public ValidationResult<List<DynamicFeedAdTarget>, Defect> validateDynamicFeedAdTargets(
            int shard, Long operatorUid, ClientId clientId,
            ValidationResult<List<DynamicFeedAdTarget>, Defect> validationResult) {

        List<DynamicFeedAdTarget> dynamicAdTargets = filterList(validationResult.getValue(),
                d -> d.getAdGroupId() != null);

        Currency currency = clientService.getWorkCurrency(clientId);

        List<Long> campaignIds = mapList(dynamicAdTargets, DynamicAdTarget::getCampaignId);
        List<Long> adGroupIds = mapList(dynamicAdTargets, DynamicAdTarget::getAdGroupId);

        List<Campaign> campaigns = campaignRepository.getCampaigns(shard, campaignIds);
        Map<Long, Campaign> campaignById = listToMap(campaigns, Campaign::getId);

        List<AdGroup> adGroups = adGroupRepository.getAdGroups(shard, adGroupIds);
        Set<Long> dynamicAdGroupsIds = adGroups.stream()
                .filter(adGroup -> ALLOWED_DYNAMIC_AD_GROUP_TYPES.contains(adGroup.getType()))
                .map(AdGroup::getId)
                .collect(toSet());

        Map<Long, Feed> feedByAdGroupId = feedService.getFeedByDynamicAdGroupId(clientId, adGroupIds);

        List<DynamicFeedAdTarget> existedDynamicAdTargets = dynamicTextAdTargetRepository
                .getDynamicFeedAdTargetsByAdGroupIds(shard, clientId, dynamicAdGroupsIds).stream()
                .filter(d -> d.getId() != null)
                .collect(Collectors.toList());

        CampaignSubObjectAccessConstraint accessConstraint = campaignSubObjectAccessCheckerFactory
                .newAdGroupChecker(operatorUid, clientId, adGroupIds)
                .createValidator(CampaignAccessType.READ_WRITE, ACCESS_DEFECTS)
                .getAccessConstraint();

        ListValidationBuilder<DynamicFeedAdTarget, Defect> lvb = new ListValidationBuilder<>(validationResult);
        lvb
                .checkEachBy(adGroupIdValidator(), When.isValid())
                .checkEach((Constraint<DynamicFeedAdTarget, Defect>)
                        d -> accessConstraint.apply(d.getAdGroupId()), When.isValid())
                .checkEach(dynamicAdTargetNoMoreThanMax(dynamicAdTargets, existedDynamicAdTargets),
                        When.isValid())
                .checkEachBy(checkAdGroupType(dynamicAdGroupsIds, feedByAdGroupId), When.isValid());

        lvb.checkEachBy(bidsValidator(currency, campaignById), When.isValid());
        lvb.checkEachBy(nameValidator(), When.isValid());

        Map<Long, List<DynamicFeedAdTarget>> dynamicAdTargetsByAdGroupId = StreamEx.of(dynamicAdTargets)
                .groupingBy(DynamicAdTarget::getAdGroupId);

        Map<Long, List<DynamicFeedAdTarget>> existedDynamicAdTargetsByAdGroupId = StreamEx.of(existedDynamicAdTargets)
                .groupingBy(DynamicAdTarget::getAdGroupId);

        lvb.checkEachBy(this::validateCondition, When.isValid());
        lvb.checkEachBy(d -> checkForDuplicateConditionInAdding(d, dynamicAdTargetsByAdGroupId), When.isValid());
        lvb.checkEachBy(d -> checkForDuplicateConditionInExisting(d, existedDynamicAdTargetsByAdGroupId),
                When.isValid());

        return lvb.getResult();
    }

    private Validator<DynamicFeedAdTarget, Defect> checkAdGroupType(Set<Long> dynamicAdGroupsIds,
                                                                    Map<Long, Feed> feedByAdGroupId) {
        return dynamicAdTarget -> {
            ModelItemValidationBuilder<DynamicFeedAdTarget> v = ModelItemValidationBuilder.of(dynamicAdTarget);

            Feed feed = feedByAdGroupId.get(dynamicAdTarget.getAdGroupId());

            v.item(DynamicFeedAdTarget.AD_GROUP_ID)
                    .check(inSet(dynamicAdGroupsIds), inconsistentState())
                    .check(Constraint.fromPredicate(d -> feed != null, notAcceptableAdGroupType()),
                            When.isValid());

            return v.getResult();
        };
    }

    private ValidationResult<DynamicFeedAdTarget, Defect> checkForDuplicateConditionInExisting(
            DynamicFeedAdTarget dynamicAdTarget, Map<Long, List<DynamicFeedAdTarget>> dynamicAdTargetsByAdGroupId) {

        Long adGroupId = dynamicAdTarget.getAdGroupId();
        List<DynamicFeedAdTarget> dynamicAdTargets = dynamicAdTargetsByAdGroupId.getOrDefault(adGroupId, List.of());
        List<DynamicFeedAdTarget> otherDynamicAdTargets = filterList(dynamicAdTargets,
                d -> !Objects.equals(d.getId(), dynamicAdTarget.getId()));

        return checkForDuplicateCondition(dynamicAdTarget, otherDynamicAdTargets);
    }

    private ValidationResult<DynamicFeedAdTarget, Defect> checkForDuplicateConditionInAdding(
            DynamicFeedAdTarget dynamicAdTarget, Map<Long, List<DynamicFeedAdTarget>> dynamicAdTargetsByAdGroupId) {

        Long adGroupId = dynamicAdTarget.getAdGroupId();
        List<DynamicFeedAdTarget> dynamicAdTargets = dynamicAdTargetsByAdGroupId.getOrDefault(adGroupId, List.of());
        List<DynamicFeedAdTarget> otherDynamicAdTargets = filterList(dynamicAdTargets, d -> d != dynamicAdTarget);

        return checkForDuplicateCondition(dynamicAdTarget, otherDynamicAdTargets);
    }

    private ValidationResult<DynamicFeedAdTarget, Defect> checkForDuplicateCondition(
            DynamicFeedAdTarget dynamicAdTarget, List<DynamicFeedAdTarget> otherDynamicAdTargets) {
        // нельзя добавлять условие-дубликаты
        boolean isDuplicate = otherDynamicAdTargets.stream()
                .anyMatch(d -> PerformanceFilterUtils.haveEqualConditions(d, dynamicAdTarget));

        if (isDuplicate) {
            return ValidationResult.failed(dynamicAdTarget, duplicatedObject());
        }
        return ValidationResult.success(dynamicAdTarget);
    }

    private ValidationResult<DynamicFeedAdTarget, Defect> validateCondition(DynamicFeedAdTarget dynamicFeedAdTarget) {
        FilterSchema filterSchema = filterSchemaStorage.getFilterSchema(dynamicFeedAdTarget.getBusinessType(),
                dynamicFeedAdTarget.getFeedType());
        DynamicAdTargetTab filterTab = dynamicFeedAdTarget.getTab();

        ModelItemValidationBuilder<DynamicFeedAdTarget> vb = ModelItemValidationBuilder.of(dynamicFeedAdTarget);

        vb.list(DynamicFeedAdTarget.CONDITION)
                .checkBy(dynamicFeedRulesValidator(filterSchema, filterTab));

        return vb.getResult();
    }

    private Validator<List<DynamicFeedRule>, Defect> dynamicFeedRulesValidator(FilterSchema filterSchema,
                                                                               DynamicAdTargetTab filterTab) {
        // FilterConditionsValidator работает с родительским классом PerformanceFilterCondition.
        // Приводим к валидатору для списка DynamicFeedRule
        Validator<?, Defect> validator = new FilterConditionsValidator(filterSchema, filterTab);
        return (Validator<List<DynamicFeedRule>, Defect>) validator;
    }
}

