package ru.yandex.direct.core.entity.dynamictextadtarget.service.validation;

import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.adgroup.model.AdGroup;
import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.adgroup.repository.AdGroupRepository;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignAccessDefects;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignSubObjectAccessCheckerFactory;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignSubObjectAccessConstraint;
import ru.yandex.direct.core.entity.campaign.service.validation.CampaignAccessType;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.dynamictextadtarget.model.DynamicAdTarget;
import ru.yandex.direct.core.entity.dynamictextadtarget.model.DynamicTextAdTarget;
import ru.yandex.direct.core.entity.dynamictextadtarget.model.WebpageRule;
import ru.yandex.direct.core.entity.dynamictextadtarget.model.WebpageRuleType;
import ru.yandex.direct.core.entity.dynamictextadtarget.repository.DynamicTextAdTargetRepository;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.ListItemValidator;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.constraint.CollectionConstraints;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.util.ModelChangesValidationTool;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static java.util.stream.Collectors.groupingBy;
import static java.util.stream.Collectors.toSet;
import static ru.yandex.direct.core.entity.campaign.service.accesschecker.AccessDefectPresets.DEFAULT_DEFECTS;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignDefects.archivedCampaignModification;
import static ru.yandex.direct.core.entity.dynamictextadtarget.service.validation.DynamicAdTargetValidationHelper.adGroupIdValidator;
import static ru.yandex.direct.core.entity.dynamictextadtarget.service.validation.DynamicAdTargetValidationHelper.bidsValidator;
import static ru.yandex.direct.core.entity.dynamictextadtarget.service.validation.DynamicAdTargetValidationHelper.dynamicAdTargetNoMoreThanMax;
import static ru.yandex.direct.core.entity.dynamictextadtarget.service.validation.DynamicAdTargetValidationHelper.nameValidator;
import static ru.yandex.direct.core.entity.dynamictextadtarget.service.validation.DynamicTextAdTargetConstants.ALL_PAGE_CONDITION_UNIQ_HASH;
import static ru.yandex.direct.core.entity.dynamictextadtarget.service.validation.DynamicTextAdTargetConstants.MAX_ARGUMENTS;
import static ru.yandex.direct.core.entity.dynamictextadtarget.service.validation.DynamicTextAdTargetConstants.MAX_CONDITIONS_PER_WEBPAGE;
import static ru.yandex.direct.core.entity.dynamictextadtarget.service.validation.DynamicTextAdTargetConstants.MIN_ARGUMENTS;
import static ru.yandex.direct.core.entity.dynamictextadtarget.service.validation.DynamicTextAdTargetConstants.MIN_CONDITIONS_PER_WEBPAGE;
import static ru.yandex.direct.core.entity.dynamictextadtarget.service.validation.DynamicTextAdTargetConstants.URL_RULE;
import static ru.yandex.direct.core.entity.dynamictextadtarget.service.validation.DynamicTextAdTargetConstants.WEBPAGE_RULE_AVAILABLE_ARGUMENT_LENGTH;
import static ru.yandex.direct.core.entity.dynamictextadtarget.service.validation.DynamicTextAdTargetConstants.WEBPAGE_RULE_AVAILABLE_OPERATIONS;
import static ru.yandex.direct.core.entity.dynamictextadtarget.service.validation.DynamicTextAdTargetConstraints.allowedChars;
import static ru.yandex.direct.core.entity.dynamictextadtarget.service.validation.DynamicTextAdTargetConstraints.isNotBlankUrl;
import static ru.yandex.direct.core.entity.dynamictextadtarget.service.validation.DynamicTextAdTargetDefects.adGroupNotFound;
import static ru.yandex.direct.core.entity.dynamictextadtarget.service.validation.DynamicTextAdTargetDefects.allPageConditionNotWithAnother;
import static ru.yandex.direct.core.entity.dynamictextadtarget.service.validation.DynamicTextAdTargetDefects.duplicateRulesInCondition;
import static ru.yandex.direct.core.entity.dynamictextadtarget.service.validation.DynamicTextAdTargetDefects.exceededMaxLengthInArguments;
import static ru.yandex.direct.core.entity.dynamictextadtarget.service.validation.DynamicTextAdTargetDefects.invalidFormatWebpageCondition;
import static ru.yandex.direct.core.entity.dynamictextadtarget.service.validation.DynamicTextAdTargetDefects.invalidLettersInRule;
import static ru.yandex.direct.core.entity.dynamictextadtarget.service.validation.DynamicTextAdTargetDefects.invalidUrlFormat;
import static ru.yandex.direct.core.entity.dynamictextadtarget.service.validation.DynamicTextAdTargetDefects.notAcceptableAdGroupType;
import static ru.yandex.direct.core.entity.dynamictextadtarget.service.validation.DynamicTextAdTargetDefects.numberArgumentsMustBeFromTo;
import static ru.yandex.direct.core.entity.dynamictextadtarget.service.validation.DynamicTextAdTargetDefects.numberOfRulesInDynamicTextAdTarget;
import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.unique;
import static ru.yandex.direct.validation.constraint.CommonConstraints.inSet;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.StringConstraints.maxStringLength;
import static ru.yandex.direct.validation.constraint.StringConstraints.validHref;
import static ru.yandex.direct.validation.defect.CollectionDefects.duplicatedElement;
import static ru.yandex.direct.validation.defect.CollectionDefects.duplicatedObject;
import static ru.yandex.direct.validation.defect.CommonDefects.inconsistentState;
import static ru.yandex.direct.validation.defect.CommonDefects.inconsistentStateAlreadyExists;

@Service
public class AddDynamicTextAdTargetValidationService {

    static final CampaignAccessDefects ACCESS_DEFECTS = DEFAULT_DEFECTS.toBuilder()
            .withNotVisible(adGroupNotFound())
            .withTypeNotSupported(inconsistentState())
            .withArchivedModification(archivedCampaignModification())
            .build();

    private final CampaignSubObjectAccessCheckerFactory campaignSubObjectAccessCheckerFactory;
    private final ClientService clientService;
    private final CampaignRepository campaignRepository;
    private final AdGroupRepository adGroupRepository;
    private final DynamicTextAdTargetRepository dynamicTextAdTargetRepository;
    private final ModelChangesValidationTool updateValidationTool;

    @Autowired
    public AddDynamicTextAdTargetValidationService(CampaignSubObjectAccessCheckerFactory campaignSubObjectAccessCheckerFactory,
                                                   ClientService clientService,
                                                   CampaignRepository campaignRepository,
                                                   AdGroupRepository adGroupRepository,
                                                   DynamicTextAdTargetRepository dynamicTextAdTargetRepository) {
        this.campaignSubObjectAccessCheckerFactory = campaignSubObjectAccessCheckerFactory;
        this.clientService = clientService;
        this.campaignRepository = campaignRepository;
        this.adGroupRepository = adGroupRepository;
        this.dynamicTextAdTargetRepository = dynamicTextAdTargetRepository;
        this.updateValidationTool = ModelChangesValidationTool.builder().build();
    }

    public ValidationResult<List<ModelChanges<DynamicTextAdTarget>>, Defect> validateModelChanges(
            int shard, ClientId clientId, List<ModelChanges<DynamicTextAdTarget>> changes) {
        Set<Long> ids = listToSet(changes, ModelChanges::getId);

        List<DynamicTextAdTarget> existingDynamicAdTargets =
                dynamicTextAdTargetRepository.getDynamicTextAdTargetsByIds(shard, clientId, ids);
        Set<Long> existingIds = listToSet(existingDynamicAdTargets, DynamicAdTarget::getId);

        return updateValidationTool.validateModelChangesList(changes, existingIds);
    }

    public ValidationResult<List<DynamicTextAdTarget>, Defect> validateAdd(
            int shard, Long operatorUid, ClientId clientId, List<DynamicTextAdTarget> dynamicTextAdTargets) {
        ValidationResult<List<DynamicTextAdTarget>, Defect> validationResult =
                new ValidationResult<>(dynamicTextAdTargets);
        return validateDynamicTextAdTargets(shard, operatorUid, clientId, validationResult);
    }

    public ValidationResult<List<DynamicTextAdTarget>, Defect> validateDynamicTextAdTargets(
            int shard, Long operatorUid, ClientId clientId,
            ValidationResult<List<DynamicTextAdTarget>, Defect> validationResult) {

        List<DynamicTextAdTarget> dynamicTextAdTargets = filterList(validationResult.getValue(),
                d -> d.getAdGroupId() != null);

        Currency currency = clientService.getWorkCurrency(clientId);

        List<Long> campaignIds = mapList(dynamicTextAdTargets, DynamicTextAdTarget::getCampaignId);
        List<Long> adGroupIds = mapList(dynamicTextAdTargets, DynamicTextAdTarget::getAdGroupId);

        List<Campaign> campaigns = campaignRepository.getCampaigns(shard, campaignIds);
        Map<Long, Campaign> campaignById = listToMap(campaigns, Campaign::getId);

        Set<Long> dynamicAdGroupsIds = adGroupRepository.getAdGroups(shard, adGroupIds).stream()
                .filter(adGroup -> adGroup.getType() == AdGroupType.DYNAMIC)
                .map(AdGroup::getId)
                .collect(toSet());

        Map<Long, String> dynamicMainDomains = adGroupRepository
                .getDynamicMainDomains(shard, dynamicAdGroupsIds);

        List<DynamicTextAdTarget> existedDynamicTextAdTargets = dynamicTextAdTargetRepository
                .getDynamicTextAdTargetsWithDomainTypeByAdGroup(shard, clientId, dynamicAdGroupsIds).stream()
                .filter(d -> d.getId() != null)
                .collect(Collectors.toList());

        Map<Long, List<DynamicTextAdTarget>> existedDynamicTextAdTargetsByAdGroupId =
                existedDynamicTextAdTargets.stream()
                        .collect(groupingBy(DynamicTextAdTarget::getAdGroupId));

        Map<Long, List<DynamicTextAdTarget>> addingDynamicTextAdTargetsConditionByAdGroupId =
                dynamicTextAdTargets.stream()
                        .collect(groupingBy(DynamicTextAdTarget::getAdGroupId));

        CampaignSubObjectAccessConstraint accessConstraint = campaignSubObjectAccessCheckerFactory
                .newAdGroupChecker(operatorUid, clientId, adGroupIds)
                .createValidator(CampaignAccessType.READ_WRITE, ACCESS_DEFECTS)
                .getAccessConstraint();


        ListValidationBuilder<DynamicTextAdTarget, Defect> lvb = new ListValidationBuilder<>(validationResult);
        lvb.check(notNull())
                .checkEach(notNull(), When.isValid())
                .checkEach(unique(), duplicatedObject(), When.isValid())
                .checkEachBy(adGroupIdValidator(), When.isValid())
                .checkEach(
                        (Constraint<DynamicTextAdTarget, Defect>) dynamicTextAdTarget -> accessConstraint
                                .apply(dynamicTextAdTarget.getAdGroupId()), When.isValid())
                .checkEach(dynamicAdTargetNoMoreThanMax(dynamicTextAdTargets, existedDynamicTextAdTargets),
                        When.isValid())
                .checkEachBy(checkAdGroupType(dynamicAdGroupsIds, dynamicMainDomains), When.isValid())
                .checkEachBy(checkForDuplicateConditionInAdding(addingDynamicTextAdTargetsConditionByAdGroupId),
                        When.isValid())
                .checkEachBy(checkForDuplicateConditionInExisting(existedDynamicTextAdTargetsByAdGroupId),
                        When.isValid());

        lvb.checkEachBy(bidsValidator(currency, campaignById), When.isValid());

        lvb.checkEachBy(conditionValidator(), When.isValid());
        lvb.checkEachBy(webpageRulesValidator(), When.isValid());
        lvb.checkEachBy(nameValidator(), When.isValid());

        return lvb.getResult();
    }

    private Validator<DynamicTextAdTarget, Defect> checkForDuplicateConditionInExisting(
            Map<Long, List<DynamicTextAdTarget>> existedDynamicTextAdTargetsByAdGroupId) {
        return dynamicTextAdTarget -> {
            ModelItemValidationBuilder<DynamicTextAdTarget> v = ModelItemValidationBuilder.of(dynamicTextAdTarget);

            Long adGroupId = dynamicTextAdTarget.getAdGroupId();

            List<DynamicTextAdTarget> existedDynamicTextAdTargets =
                    existedDynamicTextAdTargetsByAdGroupId.getOrDefault(adGroupId, List.of());

            Set<Integer> existedConditionHashes = existedDynamicTextAdTargets.stream()
                    .filter(d -> !Objects.equals(d.getId(), dynamicTextAdTarget.getId()))
                    .map(DynamicTextAdTarget::getConditionUniqHash)
                    .collect(toSet());

            if (existedConditionHashes.isEmpty()) {
                return v.getResult();
            }

            Integer conditionHash = dynamicTextAdTarget.getConditionUniqHash();

            v.item(DynamicTextAdTarget.CONDITION)
                    // нельзя добавлять условие-дубликаты
                    .check(Constraint.fromPredicate(c -> !existedConditionHashes.contains(conditionHash),
                            inconsistentStateAlreadyExists()))
                    // вместе с условием "все страницы" нельзя добавлять какие-либо еще.
                    .check(allPageConditionNotWithAnotherConditions(existedConditionHashes, conditionHash),
                            When.isValid());

            return v.getResult();
        };
    }

    private Validator<DynamicTextAdTarget, Defect> checkForDuplicateConditionInAdding(
            Map<Long, List<DynamicTextAdTarget>> addingDynamicTextAdTargetsByAdGroupId) {
        return dynamicTextAdTarget -> {
            ModelItemValidationBuilder<DynamicTextAdTarget> v = ModelItemValidationBuilder.of(dynamicTextAdTarget);

            Long adGroupId = dynamicTextAdTarget.getAdGroupId();

            List<DynamicTextAdTarget> addingDynamicTextAdTargets =
                    addingDynamicTextAdTargetsByAdGroupId.get(adGroupId);

            Set<Integer> addingConditionHashes = addingDynamicTextAdTargets.stream()
                    .filter(d -> d != dynamicTextAdTarget)
                    .map(DynamicTextAdTarget::getConditionUniqHash)
                    .collect(toSet());

            if (addingConditionHashes.isEmpty()) {
                return v.getResult();
            }

            Integer conditionHash = dynamicTextAdTarget.getConditionUniqHash();

            v.item(DynamicTextAdTarget.CONDITION)
                    // нельзя добавлять условие-дубликаты
                    .check(Constraint
                            .fromPredicate(c -> !addingConditionHashes.contains(conditionHash), duplicatedElement()))
                    // вместе с условием "все страницы" нельзя добавлять какие-либо еще.
                    .check(allPageConditionNotWithAnotherConditions(addingConditionHashes, conditionHash),
                            When.isValid());

            return v.getResult();
        };
    }

    /**
     * Существует условие ALL_PAGE_CONDITIONS, несовместимое с другими условиями.
     * Если в списке условий есть ALL_PAGE_CONDITIONS, других быть не может
     * <p>
     * Поэтому, если добавляем ALL_PAGE_CONDITIONS и другие условия уже существует
     * Или, если среди других условий есть ALL_PAGE_CONDITIONS, констрейнт не выполняется
     *
     * @param conditionHashes    - другие добавляемые или уже записанные в базе условия
     * @param conditionHashToAdd - добавлямое условие
     */
    private Constraint<List<WebpageRule>, Defect> allPageConditionNotWithAnotherConditions(Set<Integer> conditionHashes,
                                                                                           Integer conditionHashToAdd) {
        return Constraint.fromPredicate(condition -> {
            if (conditionHashes.isEmpty()) {
                return true;
            }

            if (ALL_PAGE_CONDITION_UNIQ_HASH
                    .equals(conditionHashToAdd) || conditionHashes.contains(ALL_PAGE_CONDITION_UNIQ_HASH)) {
                return false;
            }

            return true;

        }, allPageConditionNotWithAnother());
    }

    private Validator<DynamicTextAdTarget, Defect> checkAdGroupType(Set<Long> dynamicAdGroupsIds,
                                                                    Map<Long, String> dynamicMainDomains) {
        return dynamicTextAdTarget -> {
            ModelItemValidationBuilder<DynamicTextAdTarget> v = ModelItemValidationBuilder.of(dynamicTextAdTarget);

            String mainDomain = dynamicMainDomains.get(dynamicTextAdTarget.getAdGroupId());

            v.item(DynamicTextAdTarget.AD_GROUP_ID)
                    .check(inSet(dynamicAdGroupsIds), inconsistentState())
                    .check(Constraint.fromPredicate(d -> mainDomain != null, notAcceptableAdGroupType()),
                            When.isValid());

            return v.getResult();
        };
    }

    private static Validator<DynamicTextAdTarget, Defect> conditionValidator() {
        return dynamicTextAdTarget -> {
            ModelItemValidationBuilder<DynamicTextAdTarget> vb = ModelItemValidationBuilder.of(dynamicTextAdTarget);

            vb.item(dynamicTextAdTarget.getCondition(), DynamicTextAdTarget.CONDITION.name())
                    .check(CollectionConstraints.listSize(MIN_CONDITIONS_PER_WEBPAGE, MAX_CONDITIONS_PER_WEBPAGE),
                            numberOfRulesInDynamicTextAdTarget(MIN_CONDITIONS_PER_WEBPAGE, MAX_CONDITIONS_PER_WEBPAGE),
                            When.isValid())
                    .check(fromPredicate(o -> StreamEx.of(o).distinct(2).count() == 0,
                            duplicateRulesInCondition()), When.isValid());

            return vb.getResult();
        };
    }

    static Validator<DynamicTextAdTarget, Defect> webpageRulesValidator() {
        return dynamicTextAdTarget -> {
            ModelItemValidationBuilder<DynamicTextAdTarget> vb = ModelItemValidationBuilder.of(dynamicTextAdTarget);
            vb.list(dynamicTextAdTarget.getCondition(), DynamicTextAdTarget.CONDITION.name())
                    .checkEachBy(webpageRuleValidator(), When.isValid());
            return vb.getResult();
        };
    }

    private static ListItemValidator<WebpageRule, Defect> webpageRuleValidator() {
        return (ruleIndex, rule) -> {
            ItemValidationBuilder<WebpageRule, Defect> vb = ItemValidationBuilder.of(rule);
            vb.list(rule.getValue(), WebpageRule.VALUE.name())
                    .check(CollectionConstraints.listSize(MIN_ARGUMENTS, MAX_ARGUMENTS),
                            numberArgumentsMustBeFromTo(ruleIndex + 1, MIN_ARGUMENTS, MAX_ARGUMENTS), When.isValid())
                    .checkEachBy((argumentIndex, argument) -> {
                                ItemValidationBuilder<String, Defect> v = ItemValidationBuilder.of(argument);
                                v
                                        .check(maxStringLength(WEBPAGE_RULE_AVAILABLE_ARGUMENT_LENGTH.get(rule.getType())),
                                                exceededMaxLengthInArguments(argumentIndex + 1, ruleIndex + 1,
                                                        WEBPAGE_RULE_AVAILABLE_ARGUMENT_LENGTH.get(rule.getType())),
                                                When.isValid())
                                        .check(validHref(), invalidUrlFormat(argumentIndex + 1, ruleIndex + 1),
                                                When.isTrue(rule.getType() == WebpageRuleType.URL_PRODLIST))
                                        .check(isNotBlankUrl(argumentIndex + 1, ruleIndex + 1),
                                                When.isValidAnd(When.isTrue(URL_RULE.contains(rule.getType()))))
                                        .check(allowedChars(), invalidLettersInRule(argumentIndex + 1, ruleIndex + 1),
                                                When.isValidAnd(When.isTrue(URL_RULE.contains(rule.getType()))));

                                return v.getResult();
                            }, When.isTrue(!vb.getResult().hasAnyErrors())
                    );

            vb.item(rule.getKind(), WebpageRule.KIND.name())
                    .check(inSet(WEBPAGE_RULE_AVAILABLE_OPERATIONS.get(rule.getType())),
                            invalidFormatWebpageCondition(ruleIndex + 1), When.isValid());

            return vb.getResult();
        };
    }
}

