package ru.yandex.direct.core.entity.dynamictextadtarget.service.validation;

import java.math.BigDecimal;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.bids.validation.PriceValidator;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.model.DbStrategy;
import ru.yandex.direct.core.entity.dynamictextadtarget.model.DynamicAdTarget;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.core.entity.bids.validation.BidsConstraints.priceContextIsAcceptedForStrategy;
import static ru.yandex.direct.core.entity.bids.validation.BidsConstraints.priceSearchIsAcceptedForStrategy;
import static ru.yandex.direct.core.entity.bids.validation.BidsValidator.autoBudgetPriorityIsAcceptedForStrategy;
import static ru.yandex.direct.core.entity.dynamictextadtarget.service.validation.DynamicTextAdTargetConstants.MAX_DYNAMIC_TEXT_AD_TARGETS_IN_GROUP;
import static ru.yandex.direct.core.entity.dynamictextadtarget.service.validation.DynamicTextAdTargetConstants.MAX_NAME_LENGTH;
import static ru.yandex.direct.core.entity.dynamictextadtarget.service.validation.DynamicTextAdTargetConstraints.allowedChars;
import static ru.yandex.direct.core.entity.dynamictextadtarget.service.validation.DynamicTextAdTargetConstraints.isNotBlankName;
import static ru.yandex.direct.core.entity.dynamictextadtarget.service.validation.DynamicTextAdTargetDefects.exceededMaxLengthInName;
import static ru.yandex.direct.core.entity.dynamictextadtarget.service.validation.DynamicTextAdTargetDefects.invalidLettersInName;
import static ru.yandex.direct.core.entity.dynamictextadtarget.service.validation.DynamicTextAdTargetDefects.maxDynamicTextAdTargetsInAdGroup;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notNull;
import static ru.yandex.direct.validation.constraint.CommonConstraints.validId;
import static ru.yandex.direct.validation.constraint.StringConstraints.maxStringLength;

public class DynamicAdTargetValidationHelper {

    private DynamicAdTargetValidationHelper() {
    }

    static <T extends DynamicAdTarget> Constraint<T, Defect> dynamicAdTargetNoMoreThanMax(
            List<T> addingDynamicAdTargets, List<T> existedDynamicAdTargets) {

        Map<Long, Long> addingCountByAdGroupId = StreamEx.of(addingDynamicAdTargets)
                .filter(d -> d.getId() == null)
                .groupingBy(DynamicAdTarget::getAdGroupId, Collectors.counting());

        Map<Long, Long> existedCountByAdGroupId = StreamEx.of(existedDynamicAdTargets)
                .groupingBy(DynamicAdTarget::getAdGroupId, Collectors.counting());

        return dynamicAdTarget -> {
            Long adGroupId = dynamicAdTarget.getAdGroupId();

            Long addingCount = addingCountByAdGroupId.getOrDefault(adGroupId, 0L);
            Long existedCount = existedCountByAdGroupId.getOrDefault(adGroupId, 0L);

            if (addingCount + existedCount <= MAX_DYNAMIC_TEXT_AD_TARGETS_IN_GROUP) {
                return null;
            }

            return maxDynamicTextAdTargetsInAdGroup(MAX_DYNAMIC_TEXT_AD_TARGETS_IN_GROUP);
        };
    }

    static <T extends DynamicAdTarget> Validator<T, Defect> bidsValidator(
            Currency currency, Map<Long, Campaign> campaignById) {

        return dynamicAdTarget -> {
            Campaign campaign = campaignById.get(dynamicAdTarget.getCampaignId());
            DbStrategy strategy = campaign.getStrategy();

            ModelItemValidationBuilder<T> v = ModelItemValidationBuilder.of(dynamicAdTarget);

            v.item(DynamicAdTarget.PRICE)
                    .checkBy(priceValidator(currency))
                    .weakCheck(priceSearchIsAcceptedForStrategy(strategy),
                            When.isValidAnd(When.isTrue(strategy != null)));

            v.item(DynamicAdTarget.PRICE_CONTEXT)
                    .checkBy(priceValidator(currency))
                    .weakCheck(priceContextIsAcceptedForStrategy(strategy),
                            When.isValidAnd(When.isTrue(strategy != null)));

            v.item(DynamicAdTarget.AUTOBUDGET_PRIORITY)
                    .weakCheck(autoBudgetPriorityIsAcceptedForStrategy(strategy),
                            When.isValidAnd(When.isTrue(strategy != null)));

            return v.getResult();
        };
    }

    private static Validator<BigDecimal, Defect> priceValidator(Currency currency) {
        return price -> {
            ItemValidationBuilder<BigDecimal, Defect> v = ItemValidationBuilder.of(price);
            v.checkBy(
                    new PriceValidator(currency, AdGroupType.DYNAMIC),
                    When.notNull());
            return v.getResult();
        };
    }

    static <T extends DynamicAdTarget> Validator<T, Defect> nameValidator() {
        return dynamicAdTarget -> {
            ModelItemValidationBuilder<T> v = ModelItemValidationBuilder.of(dynamicAdTarget);
            v.item(dynamicAdTarget.getConditionName(), DynamicAdTarget.CONDITION_NAME.name())
                    .check(maxStringLength(MAX_NAME_LENGTH), exceededMaxLengthInName(MAX_NAME_LENGTH), When.isValid())
                    .check(isNotBlankName(), When.isValid())
                    .check(allowedChars(), invalidLettersInName(), When.isValid());
            return v.getResult();
        };
    }

    static <T extends DynamicAdTarget> Validator<T, Defect> adGroupIdValidator() {
        return dynamicAdTarget -> {
            ModelItemValidationBuilder<T> v = ModelItemValidationBuilder.of(dynamicAdTarget);

            v.item(DynamicAdTarget.AD_GROUP_ID)
                    .check(notNull())
                    .check(validId(), When.notNull());
            return v.getResult();
        };
    }
}
