package ru.yandex.direct.core.entity.dynamictextadtarget.service.validation;

import java.math.BigDecimal;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.bids.container.SetBidItem;
import ru.yandex.direct.core.entity.bids.service.CommonSetBidsValidationService;
import ru.yandex.direct.core.entity.bids.validation.PriceValidator;
import ru.yandex.direct.core.entity.bids.validation.SetBidConstraints;
import ru.yandex.direct.core.entity.campaign.model.Campaign;
import ru.yandex.direct.core.entity.campaign.model.DbStrategy;
import ru.yandex.direct.core.entity.campaign.repository.CampaignRepository;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignAccessDefects;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignSubObjectAccessChecker;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignSubObjectAccessCheckerFactory;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignSubObjectAccessConstraint;
import ru.yandex.direct.core.entity.campaign.service.validation.CampaignAccessType;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.dynamictextadtarget.model.DynamicFeedAdTarget;
import ru.yandex.direct.core.entity.dynamictextadtarget.model.DynamicTextAdTarget;
import ru.yandex.direct.core.entity.dynamictextadtarget.service.RequestSetBidType;
import ru.yandex.direct.currency.Currency;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.model.ModelProperty;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.Validator;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static java.util.Arrays.asList;
import static java.util.stream.Collectors.toMap;
import static ru.yandex.direct.core.entity.banner.service.validation.defects.BannerDefects.adGroupNotFound;
import static ru.yandex.direct.core.entity.bids.validation.BidsConstraints.priceContextIsAcceptedForStrategy;
import static ru.yandex.direct.core.entity.bids.validation.BidsConstraints.priceSearchIsAcceptedForStrategy;
import static ru.yandex.direct.core.entity.bids.validation.BidsDefects.oneOfFieldsShouldBeSpecified;
import static ru.yandex.direct.core.entity.bids.validation.BidsValidator.autoBudgetPriorityIsAcceptedForStrategy;
import static ru.yandex.direct.core.entity.campaign.service.accesschecker.AccessDefectPresets.DEFAULT_DEFECTS;
import static ru.yandex.direct.core.entity.campaign.service.validation.CampaignDefects.campaignNotFound;
import static ru.yandex.direct.core.entity.dynamictextadtarget.service.validation.DynamicTextAdTargetDefects.dynamicTextAdTargetNotFoundInAdGroup;
import static ru.yandex.direct.core.entity.dynamictextadtarget.service.validation.DynamicTextAdTargetDefects.dynamicTextAdTargetNotFoundInCampaign;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.builder.Constraint.fromPredicate;
import static ru.yandex.direct.validation.defect.CommonDefects.objectNotFound;

@Service
@ParametersAreNonnullByDefault
public class DynamicTextAdTargetSetBidsValidationService {
    private final CommonSetBidsValidationService commonSetBidsValidationService;
    private final CampaignSubObjectAccessCheckerFactory campaignSubObjectAccessCheckerFactory;
    private final ClientService clientService;
    private final CampaignRepository campaignRepository;

    public static final List<ModelProperty<?, ?>> BIDS_FIELDS =
            asList(SetBidItem.PRICE_SEARCH, SetBidItem.PRICE_CONTEXT, SetBidItem.AUTOBUDGET_PRIORITY);

    @Autowired
    public DynamicTextAdTargetSetBidsValidationService(
            CommonSetBidsValidationService commonSetBidsValidationService,
            CampaignSubObjectAccessCheckerFactory campaignSubObjectAccessCheckerFactory,
            ClientService clientService,
            CampaignRepository campaignRepository) {
        this.commonSetBidsValidationService = commonSetBidsValidationService;
        this.campaignSubObjectAccessCheckerFactory = campaignSubObjectAccessCheckerFactory;
        this.clientService = clientService;
        this.campaignRepository = campaignRepository;
    }

    public ValidationResult<List<SetBidItem>, Defect> validateForFeedAdTargets(int shard, Long operatorUid,
                                                                               ClientId clientId,
                                                                               List<SetBidItem> setBids,
                                                                               RequestSetBidType requestType,
                                                                               List<DynamicFeedAdTarget> dynamicFeedAdTargets) {
        Function<Collection<Long>, CampaignSubObjectAccessChecker> getChecker;
        Defect<Void> notFoundDefect;

        if (requestType == RequestSetBidType.ID) {
            getChecker = ids -> campaignSubObjectAccessCheckerFactory.newDynamicFeedAdTargetChecker(operatorUid,
                    clientId, ids);
            notFoundDefect = objectNotFound();
        } else if (requestType == RequestSetBidType.ADGROUP_ID) {
            getChecker = ids -> campaignSubObjectAccessCheckerFactory.newAdGroupChecker(operatorUid, clientId, ids);
            notFoundDefect = adGroupNotFound();
        } else {
            getChecker = ids -> campaignSubObjectAccessCheckerFactory.newCampaignChecker(operatorUid, clientId, ids);
            notFoundDefect = campaignNotFound();
        }

        Set<Long> existedIds = listToSet(dynamicFeedAdTargets, requestType::getId);

        Map<Long, Long> campaignIdByDynamicAdTargetId = dynamicFeedAdTargets.stream()
                .collect(toMap(DynamicFeedAdTarget::getDynamicConditionId, DynamicFeedAdTarget::getCampaignId,
                        (id1, id2) -> id1));
        Map<Long, Long> campaignIdByAdGroupId = dynamicFeedAdTargets.stream()
                .collect(toMap(DynamicFeedAdTarget::getAdGroupId, DynamicFeedAdTarget::getCampaignId,
                        (id1, id2) -> id1));
        Map<Long, Campaign> visibleCampaignsById =
                listToMap(campaignRepository.getCampaigns(shard, campaignIdByDynamicAdTargetId.values()),
                        Campaign::getId);

        return validate(clientId, setBids, requestType, getChecker, notFoundDefect, existedIds,
                campaignIdByDynamicAdTargetId, campaignIdByAdGroupId, visibleCampaignsById);
    }

    public ValidationResult<List<SetBidItem>, Defect> validateForTextAdTargets(int shard, Long operatorUid,
                                                                               ClientId clientId,
                                                                               List<SetBidItem> setBids,
                                                                               RequestSetBidType requestType,
                                                                               List<DynamicTextAdTarget> dynamicTextAdTargets) {
        Function<Collection<Long>, CampaignSubObjectAccessChecker> getChecker;
        Defect<Void> notFoundDefect;

        if (requestType == RequestSetBidType.ID) {
            getChecker = ids -> campaignSubObjectAccessCheckerFactory.newDynamicTextAdTargetChecker(operatorUid, clientId, ids);
            notFoundDefect = objectNotFound();
        } else if (requestType == RequestSetBidType.ADGROUP_ID) {
            getChecker = ids -> campaignSubObjectAccessCheckerFactory.newAdGroupChecker(operatorUid, clientId, ids);
            notFoundDefect = adGroupNotFound();
        } else {
            getChecker = ids -> campaignSubObjectAccessCheckerFactory.newCampaignChecker(operatorUid, clientId, ids);
            notFoundDefect = campaignNotFound();
        }

        Set<Long> existedIds = listToSet(dynamicTextAdTargets, requestType::getId);

        Map<Long, Long> campaignIdByDynamicAdTargetId = dynamicTextAdTargets.stream()
                .collect(toMap(DynamicTextAdTarget::getDynamicConditionId, DynamicTextAdTarget::getCampaignId,
                        (id1, id2) -> id1));
        Map<Long, Long> campaignIdByAdGroupId = dynamicTextAdTargets.stream()
                .collect(toMap(DynamicTextAdTarget::getAdGroupId, DynamicTextAdTarget::getCampaignId,
                        (id1, id2) -> id1));
        Map<Long, Campaign> visibleCampaignsById =
                listToMap(campaignRepository.getCampaigns(shard, campaignIdByDynamicAdTargetId.values()),
                        Campaign::getId);

        return validate(clientId, setBids, requestType, getChecker, notFoundDefect, existedIds,
                campaignIdByDynamicAdTargetId, campaignIdByAdGroupId, visibleCampaignsById);
    }

    private ValidationResult<List<SetBidItem>, Defect> validate(
            ClientId clientId,
            List<SetBidItem> setBids,
            RequestSetBidType requestType,
            Function<Collection<Long>, CampaignSubObjectAccessChecker> getChecker,
            Defect<Void> notFoundDefect,
            Set<Long> existedIds,
            Map<Long, Long> campaignIdByDynamicAdTargetId,
            Map<Long, Long> campaignIdByAdGroupId,
            Map<Long, Campaign> visibleCampaignsById) {
        CampaignAccessDefects accessDefects = DEFAULT_DEFECTS.toBuilder()
                .withNotVisible(notFoundDefect)
                .withTypeNotAllowable(DynamicTextAdTargetDefects::dynamicTextAdTargetNotFoundInCampaign)
                .build();

        CampaignSubObjectAccessChecker checker = getChecker.apply(mapList(setBids, requestType::getId));

        CampaignSubObjectAccessConstraint accessConstraint = checker
                .createValidator(CampaignAccessType.READ_WRITE, accessDefects)
                .getAccessConstraint();

        Currency currency = clientService.getWorkCurrency(clientId);

        ListValidationBuilder<SetBidItem, Defect> lvb = ListValidationBuilder.of(setBids, Defect.class);

        lvb
                .checkBy(commonSetBidsValidationService::preValidate)
                .checkEachBy(SetBidConstraints.positiveIdsValidator(), When.isValid())
                .checkEachBy(checkBidAndStrategyPriority(campaignIdByDynamicAdTargetId, campaignIdByAdGroupId,
                        visibleCampaignsById, requestType), When.isValid());

        lvb.checkEach(
                (Constraint<SetBidItem, Defect>) setBidItem -> accessConstraint.apply(requestType.getId(setBidItem)),
                When.isValid());

        lvb.checkEachBy(checkExistense(existedIds, requestType), When.isValid());

        lvb.checkEachBy(bidsValidator(currency), When.isValid());

        return lvb.getResult();
    }

    private Validator<SetBidItem, Defect> checkExistense(Set<Long> existedIds, RequestSetBidType requestType) {
        return bid -> {
            ModelItemValidationBuilder<SetBidItem> v = ModelItemValidationBuilder.of(bid);

            Long id = requestType.getId(bid);

            Defect defect;
            if (requestType == RequestSetBidType.CAMPAIGN_ID) {
                defect = dynamicTextAdTargetNotFoundInCampaign(id);
            } else if (requestType == RequestSetBidType.ADGROUP_ID) {
                defect = dynamicTextAdTargetNotFoundInAdGroup(id);
            } else {
                defect = objectNotFound();
            }

            v.check(fromPredicate(t -> existedIds.contains(id), defect));
            return v.getResult();
        };
    }

    private Validator<SetBidItem, Defect> checkBidAndStrategyPriority(
            Map<Long, Long> campaignIdByDynamicAdTargetId,
            Map<Long, Long> campaignIdByAdGroupId,
            Map<Long, Campaign> visibleCampaignsById,
            RequestSetBidType requestType) {
        return bid -> {

            Long campaignId = getCampaignId(requestType, bid, campaignIdByAdGroupId, campaignIdByDynamicAdTargetId);
            Campaign campaign = visibleCampaignsById.getOrDefault(campaignId, new Campaign());

            DbStrategy strategy = campaign.getStrategy();

            ModelItemValidationBuilder<SetBidItem> v = ModelItemValidationBuilder.of(bid);
            v.check(fromPredicate(
                    t -> t.getPriceSearch() != null || t.getPriceContext() != null || t.getAutobudgetPriority() != null,
                    oneOfFieldsShouldBeSpecified(BIDS_FIELDS)));

            v.item(bid.getPriceSearch(), SetBidItem.PRICE_SEARCH.name())
                    .weakCheck(priceSearchIsAcceptedForStrategy(strategy),
                            When.isValidAnd(When.isTrue(strategy != null)));

            v.item(bid.getAutobudgetPriority(), SetBidItem.AUTOBUDGET_PRIORITY.name())
                    .weakCheck(autoBudgetPriorityIsAcceptedForStrategy(strategy),
                            When.isValidAnd(When.isTrue(strategy != null)));

            v.item(bid.getPriceContext(), SetBidItem.PRICE_CONTEXT.name())
                    .weakCheck(priceContextIsAcceptedForStrategy(strategy),
                            When.isValidAnd(When.isTrue(strategy != null)));

            return v.getResult();
        };
    }

    private Validator<SetBidItem, Defect> bidsValidator(Currency currency) {
        return bid -> {
            ModelItemValidationBuilder<SetBidItem> v = ModelItemValidationBuilder.of(bid);

            v.item(SetBidItem.PRICE_SEARCH)
                    .checkBy(priceValidator(currency));

            v.item(SetBidItem.PRICE_CONTEXT)
                    .checkBy(priceValidator(currency));

            return v.getResult();
        };
    }

    private Validator<BigDecimal, Defect> priceValidator(Currency currency) {
        return price -> {
            ItemValidationBuilder<BigDecimal, Defect> v = ItemValidationBuilder.of(price);
            v
                    .checkBy(new PriceValidator(currency, AdGroupType.DYNAMIC),
                            When.notNull());
            return v.getResult();
        };
    }

    private Long getCampaignId(RequestSetBidType requestType,
                               SetBidItem setBidItem, Map<Long, Long> campaignIdByAdGroupId,
                               Map<Long, Long> campaignIdByRetargetingId) {
        Long bidId = requestType.getId(setBidItem);
        switch (requestType) {
            case ID:
                return campaignIdByRetargetingId.get(bidId);
            case ADGROUP_ID:
                return campaignIdByAdGroupId.get(bidId);
            case CAMPAIGN_ID:
                return bidId;
            default:
                throw new IllegalStateException("Неизвестный тип для " + requestType);
        }
    }
}

