package ru.yandex.direct.core.entity.dynamictextadtarget.service.validation;

import java.util.List;
import java.util.Map;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.campaign.service.accesschecker.AccessDefectPresets;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignAccessDefects;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignSubObjectAccessCheckerFactory;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignSubObjectAccessConstraint;
import ru.yandex.direct.core.entity.campaign.service.validation.CampaignAccessType;
import ru.yandex.direct.core.entity.dynamictextadtarget.model.DynamicFeedAdTarget;
import ru.yandex.direct.core.entity.dynamictextadtarget.repository.DynamicTextAdTargetRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.multitype.entity.LimitOffset;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.core.entity.dynamictextadtarget.service.validation.DynamicTextAdTargetDefects.alreadySuspended;
import static ru.yandex.direct.core.entity.dynamictextadtarget.service.validation.DynamicTextAdTargetDefects.isNotSuspended;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.unique;
import static ru.yandex.direct.validation.constraint.CommonConstraints.validId;
import static ru.yandex.direct.validation.defect.CollectionDefects.duplicatedObject;
import static ru.yandex.direct.validation.defect.CommonDefects.objectNotFound;

@Service
public class SuspendResumeDynamicFeedAdTargetValidationService {
    private static final CampaignAccessDefects ACCESS_DEFECTS = AccessDefectPresets.DEFAULT_DEFECTS.toBuilder()
            .withNotVisible(objectNotFound())
            .build();

    private final CampaignSubObjectAccessCheckerFactory accessCheckerFactory;
    private final DynamicTextAdTargetRepository dynamicTextAdTargetRepository;

    @Autowired
    public SuspendResumeDynamicFeedAdTargetValidationService(
            CampaignSubObjectAccessCheckerFactory accessCheckerFactory,
            DynamicTextAdTargetRepository dynamicTextAdTargetRepository) {
        this.accessCheckerFactory = accessCheckerFactory;
        this.dynamicTextAdTargetRepository = dynamicTextAdTargetRepository;
    }

    public ValidationResult<List<Long>, Defect> validate(int shard, ClientId clientId, long operatorUid, List<Long> ids,
                                                         boolean resume) {
        ListValidationBuilder<Long, Defect> lvb = ListValidationBuilder
                .of(ids, Defect.class);

        CampaignSubObjectAccessConstraint accessConstraint = accessCheckerFactory
                .newDynamicFeedAdTargetChecker(operatorUid, clientId, ids)
                .createValidator(CampaignAccessType.READ_WRITE, ACCESS_DEFECTS)
                .getAccessConstraint();

        List<DynamicFeedAdTarget> dynamicFeedAdTargets = dynamicTextAdTargetRepository
                .getDynamicFeedAdTargets(shard, clientId, ids, true, LimitOffset.maxLimited());

        Map<Long, DynamicFeedAdTarget> dynamicFeedAdTargetById =
                listToMap(dynamicFeedAdTargets, DynamicFeedAdTarget::getDynamicConditionId);

        lvb.checkEach(validId(), When.isValid())
                .weakCheckEach(unique(), duplicatedObject(), When.isValid())
                .checkEach(accessConstraint, When.isValid())
                .weakCheckEach(dynamicFeedAdTargetIsNotSuspended(dynamicFeedAdTargetById),
                        When.isValidAnd(When.isTrue(!resume)))
                .weakCheckEach(dynamicFeedAdTargetIsSuspended(dynamicFeedAdTargetById),
                        When.isValidAnd(When.isTrue(resume)));

        return lvb.getResult();
    }

    private Constraint<Long, Defect> dynamicFeedAdTargetIsNotSuspended(
            Map<Long, DynamicFeedAdTarget> dynamicFeedAdTargets) {
        return Constraint.fromPredicate(
                id -> dynamicFeedAdTargets.containsKey(id) && !dynamicFeedAdTargets.get(id)
                        .getIsSuspended(), alreadySuspended());
    }

    private Constraint<Long, Defect> dynamicFeedAdTargetIsSuspended(
            Map<Long, DynamicFeedAdTarget> dynamicFeedAdTargets) {
        return Constraint.fromPredicate(
                id -> dynamicFeedAdTargets.containsKey(id) && dynamicFeedAdTargets.get(id)
                        .getIsSuspended(), isNotSuspended());
    }
}
