package ru.yandex.direct.core.entity.dynamictextadtarget.service.validation;

import java.util.List;
import java.util.Map;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.campaign.service.accesschecker.AccessDefectPresets;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignAccessDefects;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignSubObjectAccessCheckerFactory;
import ru.yandex.direct.core.entity.campaign.service.accesschecker.CampaignSubObjectAccessConstraint;
import ru.yandex.direct.core.entity.campaign.service.validation.CampaignAccessType;
import ru.yandex.direct.core.entity.dynamictextadtarget.model.DynamicTextAdTarget;
import ru.yandex.direct.core.entity.dynamictextadtarget.repository.DynamicTextAdTargetRepository;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.multitype.entity.LimitOffset;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.core.entity.dynamictextadtarget.service.validation.DynamicTextAdTargetDefects.alreadySuspended;
import static ru.yandex.direct.core.entity.dynamictextadtarget.service.validation.DynamicTextAdTargetDefects.isNotSuspended;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.validation.constraint.CollectionConstraints.unique;
import static ru.yandex.direct.validation.constraint.CommonConstraints.validId;
import static ru.yandex.direct.validation.defect.CollectionDefects.duplicatedObject;
import static ru.yandex.direct.validation.defect.CommonDefects.objectNotFound;

@Service
public class SuspendResumeDynamicTextAdTargetValidationService {

    private static final CampaignAccessDefects ACCESS_DEFECTS = AccessDefectPresets.DEFAULT_DEFECTS.toBuilder()
            .withNotVisible(objectNotFound())
            .build();

    private final CampaignSubObjectAccessCheckerFactory accessCheckerFactory;
    private final DynamicTextAdTargetRepository dynamicTextAdTargetRepository;

    @Autowired
    public SuspendResumeDynamicTextAdTargetValidationService(
            CampaignSubObjectAccessCheckerFactory accessCheckerFactory,
            DynamicTextAdTargetRepository dynamicTextAdTargetRepository) {
        this.accessCheckerFactory = accessCheckerFactory;
        this.dynamicTextAdTargetRepository = dynamicTextAdTargetRepository;
    }

    public ValidationResult<List<Long>, Defect> validate(int shard, ClientId clientId, long operatorUid, List<Long> ids,
                                                         boolean resume) {
        ListValidationBuilder<Long, Defect> lvb = ListValidationBuilder
                .of(ids, Defect.class);

        CampaignSubObjectAccessConstraint accessConstraint = accessCheckerFactory
                .newDynamicTextAdTargetChecker(operatorUid, clientId, ids)
                .createValidator(CampaignAccessType.READ_WRITE, ACCESS_DEFECTS)
                .getAccessConstraint();

        List<DynamicTextAdTarget> dynamicTextAdTargets = dynamicTextAdTargetRepository
                .getDynamicTextAdTargetsWithDomainType(shard, clientId, ids, true, LimitOffset.maxLimited());

        Map<Long, DynamicTextAdTarget> dynamicTextAdTargetById =
                listToMap(dynamicTextAdTargets, DynamicTextAdTarget::getDynamicConditionId);

        lvb.checkEach(validId(), When.isValid())
                .weakCheckEach(unique(), duplicatedObject(), When.isValid())
                .checkEach(accessConstraint, When.isValid())
                .weakCheckEach(dynamicTextAdTargetIsNotSuspended(dynamicTextAdTargetById),
                        When.isValidAnd(When.isTrue(!resume)))
                .weakCheckEach(dynamicTextAdTargetIsSuspended(dynamicTextAdTargetById),
                        When.isValidAnd(When.isTrue(resume)));

        return lvb.getResult();
    }

    private Constraint<Long, Defect> dynamicTextAdTargetIsNotSuspended(
            Map<Long, DynamicTextAdTarget> dynamicTextAdTargets) {
        return Constraint.fromPredicate(
                id -> dynamicTextAdTargets.containsKey(id) && !dynamicTextAdTargets.get(id)
                        .getIsSuspended(),
                alreadySuspended());
    }

    private Constraint<Long, Defect> dynamicTextAdTargetIsSuspended(
            Map<Long, DynamicTextAdTarget> dynamicTextAdTargets) {
        return Constraint.fromPredicate(
                id -> dynamicTextAdTargets.containsKey(id) && dynamicTextAdTargets.get(id)
                        .getIsSuspended(), isNotSuspended());
    }
}
