package ru.yandex.direct.core.entity.dynamictextadtarget.utils;

import java.math.BigInteger;
import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;

import ru.yandex.direct.core.entity.dynamictextadtarget.model.DynamicFeedRule;
import ru.yandex.direct.core.entity.dynamictextadtarget.model.WebpageRule;
import ru.yandex.direct.core.entity.dynamictextadtarget.repository.DynamicTextAdTargetMapping;
import ru.yandex.direct.utils.HashingUtils;

import static ru.yandex.direct.core.entity.dynamictextadtarget.repository.DynamicTextAdTargetMapping.serializeForDynamicConditionHash;

public class DynamicTextAdTargetHashUtils {

    /**
     * Считает хеш от массива правил. В таблице dynamic_conditions condition_hash == getHash(condition_json)
     * <p>
     * _condition_hash в перле
     *
     * @param conditionJson текстовое представление json условия
     * @return хеш
     */
    public static BigInteger getHash(String conditionJson) {
        return HashingUtils.getMd5HalfHashUtf8(conditionJson);
    }

    /**
     * Считает хеш от массива правил. В таблице dynamic_conditions condition_hash == getHash(condition_json)
     * <p>
     * _condition_hash в перле
     *
     * @param rules массив правил, для которых считается хеш
     * @return хеш
     */
    public static BigInteger getHash(List<WebpageRule> rules) {
        return getHash(DynamicTextAdTargetMapping.webpageRulesToJson(rules));
    }

    public static BigInteger getHashForDynamicFeedRules(List<DynamicFeedRule> rules) {
        return getHash(serializeForDynamicConditionHash(rules));
    }

    /**
     * Этот хеш нужен для быстрого сравнения правил.
     * <p>
     * Набор правил нормализуется:
     * 1) Из набора values убираются повторы и он упорядочивается
     * 2) Из набора правил убираются повторы и он упорядочивается
     * <p>
     * Таким образом для структурно одинаковых наборов правил возвращется одинаковый хеш
     * <p>
     * __condition_uhash в перле
     *
     * @param rules массив правил, для которых считается хеш
     * @return хеш
     */
    public static int getUniqHash(List<WebpageRule> rules) {
        if (rules == null) {
            return getUniqHash(Collections.emptyList());
        }

        List<String> uniqRules = rules.stream()
                .map(DynamicTextAdTargetHashUtils::convertRuleOrdered)
                .distinct()
                .sorted()
                .collect(Collectors.toList());

        return uniqRules.hashCode();
    }

    /**
     * Этот хеш нужен для быстрого сравнения правил.
     * <p>
     * Набор правил нормализуется:
     * 1) Из набора values убираются повторы и он упорядочивается
     * 2) Из набора правил убираются повторы и он упорядочивается
     * <p>
     * Таким образом для структурно одинаковых наборов правил возвращется одинаковый хеш
     * <p>
     * __condition_uhash в перле
     *
     * @param conditionJson текстовое представление json условия
     * @return хеш
     */
    public static int getUniqHash(String conditionJson) {
        if (conditionJson == null) {
            return getUniqHash(Collections.emptyList());
        }
        return getUniqHash(DynamicTextAdTargetMapping.webpageRulesFromJson(conditionJson));
    }

    private static String convertRuleOrdered(WebpageRule rule) {
        StringBuilder sb = new StringBuilder();
        sb.append(rule.getType());
        sb.append(rule.getKind());

        List<String> values = rule.getValue();
        if (values != null) {
            // суммирование хешей дает одинаковый хеш для набора значений в разном порядке
            int valuesHash = values.stream()
                    .mapToInt(String::hashCode)
                    .distinct()
                    .sum();

            sb.append(valuesHash);
        }

        return sb.toString();
    }
}
