package ru.yandex.direct.core.entity.essblacklist;

import java.util.List;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.impl.DSL;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.essblacklist.model.EssBlacklistItem;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;

import static ru.yandex.direct.dbschema.ppcdict.tables.EssLogicObjectsBlacklist.ESS_LOGIC_OBJECTS_BLACKLIST;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.direct.jooqmapper.read.ReaderBuilders.fromField;
import static ru.yandex.direct.utils.FunctionalUtils.mapSet;

@Repository
@ParametersAreNonnullByDefault
public class EssLogicObjectsBlacklistRepository {
    private final DslContextProvider dslContextProvider;
    private JooqMapperWithSupplier<EssBlacklistItem> mapper = createMapper();

    public EssLogicObjectsBlacklistRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
    }

    private JooqMapperWithSupplier<EssBlacklistItem> createMapper() {
        return JooqMapperWithSupplierBuilder.builder(EssBlacklistItem::new)
                .readProperty(EssBlacklistItem.ID, fromField(ESS_LOGIC_OBJECTS_BLACKLIST.ID))
                .map(property(EssBlacklistItem.LOGIC_PROCESS_NAME, ESS_LOGIC_OBJECTS_BLACKLIST.LOGIC_PROCESS_NAME))
                .map(property(EssBlacklistItem.FILTER_SPEC, ESS_LOGIC_OBJECTS_BLACKLIST.FILTER_SPEC))
                .map(property(EssBlacklistItem.AUTHOR_LOGIN, ESS_LOGIC_OBJECTS_BLACKLIST.AUTHOR_LOGIN))
                .readProperty(EssBlacklistItem.CREATE_TIME, fromField(ESS_LOGIC_OBJECTS_BLACKLIST.CREATE_TIME))
                .build();
    }

    public List<EssBlacklistItem> getAll() {
        return dslContextProvider.ppcdict()
                .select(mapper.getFieldsToRead())
                .from(ESS_LOGIC_OBJECTS_BLACKLIST)
                .fetch()
                .map(mapper::fromDb);
    }

    public List<EssBlacklistItem> getItemsForLogicProcess(String logicProcessName) {
        return dslContextProvider.ppcdict()
                .select(mapper.getFieldsToRead())
                .from(ESS_LOGIC_OBJECTS_BLACKLIST)
                .where(ESS_LOGIC_OBJECTS_BLACKLIST.LOGIC_PROCESS_NAME.equal(logicProcessName))
                .fetch()
                .map(mapper::fromDb);
    }

    public void addItems(String logicProcessName, Set<String> filterSpecs, String domainLogin) {
        var itemsToInsert = mapSet(filterSpecs, spec -> new EssBlacklistItem()
                .withLogicProcessName(logicProcessName)
                .withFilterSpec(spec)
                .withAuthorLogin(domainLogin)
        );

        new InsertHelper<>(dslContextProvider.ppcdict(), ESS_LOGIC_OBJECTS_BLACKLIST)
                .addAll(mapper, itemsToInsert)
                .execute();
    }

    public void removeItems(String logicProcessName, Set<Long> ids) {
        var whereCondition = DSL.and(
                ESS_LOGIC_OBJECTS_BLACKLIST.LOGIC_PROCESS_NAME.equal(logicProcessName),
                ESS_LOGIC_OBJECTS_BLACKLIST.ID.in(ids)
        );

        dslContextProvider.ppcdict()
                .deleteFrom(ESS_LOGIC_OBJECTS_BLACKLIST)
                .where(whereCondition)
                .execute();
    }
}
