package ru.yandex.direct.core.entity.eventlog.repository;

import java.time.LocalDateTime;
import java.util.Collection;
import java.util.List;
import java.util.Objects;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.Field;
import org.jooq.Record3;
import org.jooq.impl.DSL;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.eventlog.model.EventCampaignAndTimeData;
import ru.yandex.direct.core.entity.eventlog.model.EventLog;
import ru.yandex.direct.core.entity.eventlog.model.EventLogParams;
import ru.yandex.direct.core.entity.eventlog.model.EventLogType;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;
import ru.yandex.direct.utils.JsonUtils;

import static java.util.Collections.singletonList;
import static ru.yandex.direct.dbschema.ppc.Tables.EVENTLOG;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Репозиторий для работы с таблицей ppc.eventlog
 */
@ParametersAreNonnullByDefault
@Repository
public class EventLogRepository {

    private final DslContextProvider dslContextProvider;
    private final JooqMapperWithSupplier<EventLog> jooqMapper;
    private final Collection<Field<?>> allFieldsToRead;
    private static final Field<LocalDateTime> MAX_EVENTTIME = DSL.field("maxEventTime", LocalDateTime.class);

    @Autowired
    public EventLogRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
        this.jooqMapper = JooqMapperWithSupplierBuilder.builder(EventLog::new)
                .map(property(EventLog.ID, EVENTLOG.ID))
                .map(property(EventLog.CLIENT_ID, EVENTLOG.CLIENT_ID))
                .map(property(EventLog.CAMPAIGN_ID, EVENTLOG.CID))
                .map(property(EventLog.BANNER_ID, EVENTLOG.BID))
                .map(property(EventLog.BIDS_ID, EVENTLOG.BIDS_ID))
                .map(property(EventLog.EVENT_TIME, EVENTLOG.EVENTTIME))
                .map(convertibleProperty(EventLog.TYPE, EVENTLOG.TYPE,
                        EventLogType::fromTypedValue,
                        EventLogType::getTypedValue))
                .map(convertibleProperty(EventLog.PARAMS, EVENTLOG.PARAMS,
                        json -> json != null ? JsonUtils.fromJson(json, EventLogParams.class) : null,
                        JsonUtils::toJsonNullable))
                .build();
        this.allFieldsToRead = jooqMapper.getFieldsToRead();
    }

    /**
     * Добавить лог события в таблицу ppc.eventlog
     */
    public void addEventLog(int shard, EventLog eventLog) {
        addEventLogs(shard, singletonList(eventLog));
    }

    /**
     * Добавить логи событий в таблицу ppc.eventlog
     */
    public void addEventLogs(int shard, Collection<EventLog> eventLogs) {
        if (eventLogs.isEmpty()) {
            return;
        }

        new InsertHelper<>(dslContextProvider.ppc(shard), EVENTLOG)
                .addAll(jooqMapper, eventLogs)
                .execute();
    }

    /**
     * Получить лог событий по clientId и campaignId
     */
    public List<EventLog> getEventLogsByClientIdAndCampaignId(int shard, Long clientId, Long campaignId) {
        return dslContextProvider.ppc(shard)
                .select(allFieldsToRead)
                .from(EVENTLOG)
                .where(EVENTLOG.CLIENT_ID.eq(clientId)
                        .and(EVENTLOG.CID.eq(campaignId)))
                .fetch()
                .map(jooqMapper::fromDb);
    }

    /**
     * Для каждого номера кампании и типа события выбирается максимальное время события, возвращается тип события,
     * номер кампании и время события
     *
     * @param fromCampaignId номер кампании, начиная с которого(не включительно) отбираются события
     * @param eventLogTypes типы событий, что должны быть отобраны
     * @param fromDateTime время события, от которого отбираются события
     * @param toDateTime время события, до которого отбираются события
     *
     * @return тип события, его время и номер кампании, соответствующие указанным критериям
     */
    public List<EventCampaignAndTimeData> getEventCampaignAndTimeDataWithMaxTime(int shard,
                                                                                 long fromCampaignId,
                                                                                 LocalDateTime fromDateTime,
                                                                                 LocalDateTime toDateTime,
                                                                                 List<EventLogType> eventLogTypes) {
        return dslContextProvider.ppc(shard)
                .select(EVENTLOG.CID, EVENTLOG.TYPE, DSL.max(EVENTLOG.EVENTTIME).as(MAX_EVENTTIME))
                .from(EVENTLOG)
                .where(EVENTLOG.CID.gt(fromCampaignId))
                .and(EVENTLOG.TYPE.in(mapList(eventLogTypes, EventLogType::getTypedValue)))
                .and(EVENTLOG.EVENTTIME.between(fromDateTime, toDateTime))
                .groupBy(EVENTLOG.CID, EVENTLOG.TYPE)
                .orderBy(EVENTLOG.CID, EVENTLOG.TYPE)
                .fetch(this::extractEventCampaignAndTimeData);
    }

    @Nullable
    public EventCampaignAndTimeData getLastEventByCliendIdAndCampaignId(int shard,
                                                                                ClientId clientId,
                                                                                long campaignId,
                                                                                EventLogType type) {
        EventCampaignAndTimeData event = dslContextProvider.ppc(shard)
                .select(EVENTLOG.CID, EVENTLOG.TYPE, EVENTLOG.EVENTTIME)
                .from(EVENTLOG)
                .where(EVENTLOG.CLIENT_ID.eq(clientId.asLong()))
                .and(EVENTLOG.CID.eq(campaignId))
                .and(EVENTLOG.TYPE.eq(type.getTypedValue()))
                .orderBy(EVENTLOG.EVENTTIME.desc())
                .limit(1)
                .fetch(this::extractEventCampaignAndTimeDataWithoutMaxEventtime).stream().findFirst().orElse(
                        new EventCampaignAndTimeData());
        return Objects.nonNull(event.getCampaignId()) ? event : null;
    }

    private EventCampaignAndTimeData extractEventCampaignAndTimeData(Record3<Long, Long, LocalDateTime> record) {
        return new EventCampaignAndTimeData()
                .withEventTime(record.get(MAX_EVENTTIME))
                .withCampaignId(record.get(EVENTLOG.CID))
                .withType(EventLogType.fromTypedValue(record.get(EVENTLOG.TYPE)));
    }

    private EventCampaignAndTimeData extractEventCampaignAndTimeDataWithoutMaxEventtime(
            Record3<Long, Long, LocalDateTime> record) {
        return new EventCampaignAndTimeData()
                .withEventTime(record.get(EVENTLOG.EVENTTIME))
                .withCampaignId(record.get(EVENTLOG.CID))
                .withType(EventLogType.fromTypedValue(record.get(EVENTLOG.TYPE)));
    }
}
