package ru.yandex.direct.core.entity.eventlog.service;

import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.eventlog.model.DaysLeftNotificationType;
import ru.yandex.direct.core.entity.eventlog.model.EventLog;
import ru.yandex.direct.core.entity.eventlog.model.EventLogParams;
import ru.yandex.direct.core.entity.eventlog.model.EventLogType;
import ru.yandex.direct.core.entity.eventlog.repository.EventLogRepository;
import ru.yandex.direct.currency.Money;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;
import ru.yandex.direct.dbutil.sharding.ShardedData;

import static ru.yandex.direct.utils.CommonUtils.nvl;

/**
 * Сервис для добавления лога событий
 */
@Service
@ParametersAreNonnullByDefault
public class EventLogService {
    /*
    Дефолтное значение для полей cid, bid, bids_id в таблице ppc.eventlog
     */
    public static final Long EMPTY_ID = 0L;
    private static final int MASS_INSERT_CHUNK_SIZE = 2000;
    private static final DateTimeFormatter PERL_DATE_FORMAT = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss");

    private final EventLogRepository eventLogRepository;
    private final ShardHelper shardHelper;

    @Autowired
    public EventLogService(EventLogRepository eventLogRepository, ShardHelper shardHelper) {
        this.eventLogRepository = eventLogRepository;
        this.shardHelper = shardHelper;
    }

    /**
     * Добавить лог события {@link EventLogType#MONEY_IN} или {@link EventLogType#MONEY_IN_WALLET}
     * <p>
     * Для общего счета добавляем событие с типом {@link EventLogType#MONEY_IN_WALLET}, а для остальных кампаний
     * событие с типом {@link EventLogType#MONEY_IN}
     */
    public void addMoneyInEventLog(Long campaignId, CampaignType type, Money sumPayed, Long clientId) {
        EventLog eventLog = new EventLog()
                .withType(CampaignType.WALLET.equals(type) ? EventLogType.MONEY_IN_WALLET : EventLogType.MONEY_IN)
                .withClientId(clientId)
                .withCampaignId(campaignId)
                .withParams(new EventLogParams()
                        .withSumPayed(sumPayed.bigDecimalValue().toPlainString())
                        .withCurrency(sumPayed.getCurrencyCode())
                );
        addWithDefaults(eventLog, clientId);
    }

    /**
     * Добавить лог события {@link EventLogType#MONEY_OUT} или {@link EventLogType#MONEY_OUT_WALLET}
     * <p>
     * Для общего счета добавляем событие с типом {@link EventLogType#MONEY_OUT_WALLET}, а для остальных кампаний
     * событие с типом {@link EventLogType#MONEY_OUT}
     */
    public void addMoneyOutEventLog(Long campaignId, CampaignType type, Long clientId) {
        EventLog eventLog = new EventLog()
                .withType(CampaignType.WALLET.equals(type) ? EventLogType.MONEY_OUT_WALLET : EventLogType.MONEY_OUT)
                .withClientId(clientId)
                .withCampaignId(campaignId);
        addWithDefaults(eventLog, clientId);
    }

    /**
     * Добавить лог события {@link EventLogType#MONEY_WARNING_WALLET} для случая,
     * когда на счёте осталось денег на 1 или 3 дня
     * <p>
     *
     * @param notificationType - тип события в зависимости от количества дней
     */
    public void addMoneyWarningForDaysEventLog(Long campaignId,
                                               Long clientId,
                                               DaysLeftNotificationType notificationType,
                                               Money restMoney) {
        EventLog eventLog = new EventLog()
                .withType(EventLogType.MONEY_WARNING_WALLET)
                .withClientId(clientId)
                .withCampaignId(campaignId)
                .withParams(new EventLogParams()
                        .withDaysLeft(notificationType)
                        .withSumRest(restMoney.bigDecimalValue())
                        .withCurrency(restMoney.getCurrencyCode()));
        addWithDefaults(eventLog, clientId);
    }

    /**
     * Добавить лог события {@link EventLogType#CAMP_FINISHED}
     */
    public void addCampFinishedEventLog(Long campaignId, LocalDate finishDate, Long clientId) {
        EventLog eventLog = new EventLog()
                .withType(EventLogType.CAMP_FINISHED)
                .withCampaignId(campaignId)
                .withClientId(clientId)
                .withParams(new EventLogParams()
                        .withFinishDate(finishDate.toString())
                );
        addWithDefaults(eventLog, clientId);
    }

    /**
     * Добавить лог события {@link EventLogType#PAUSED_BY_DAY_BUDGET} или
     * {@link EventLogType#PAUSED_BY_DAY_BUDGET_WALLET}
     *
     * @param campaignType      тип кампании, которую остановили
     * @param clientId          id клиента
     * @param campaignId        id кампании
     * @param dayBudgetStopTime время остановки по дневному бюджету
     * @see
     * <a href="https://a.yandex-team.ru/arc/trunk/arcadia/direct/perl/protected/Notification.pm?rev=r8434756#L1310-1318">
     * Оригинал
     * </a>
     */
    public void addPausedByDayBudgetEventLog(
            CampaignType campaignType, Long clientId, Long campaignId, LocalDateTime dayBudgetStopTime) {
        // Формат времени из перла
        String dayBudgetStopTimeAsFormattedString = dayBudgetStopTime.format(PERL_DATE_FORMAT);
        EventLog eventLog = new EventLog()
                .withType(CampaignType.WALLET.equals(campaignType) ?
                        EventLogType.PAUSED_BY_DAY_BUDGET_WALLET : EventLogType.PAUSED_BY_DAY_BUDGET)
                .withClientId(clientId)
                .withCampaignId(campaignId)
                .withParams(new EventLogParams()
                        .withBsStopTime(dayBudgetStopTimeAsFormattedString)
                );
        addWithDefaults(eventLog, clientId);
    }

    /**
     * Записать событие типа EventLogType.CUSTOM_MESSAGE_WITH_LINK на список клиентов кроссшардово. Для пушей.
     *
     * @param clientIds — список id клиентов
     * @param link      — полный URL ссылки
     * @param text      — текст, который будет отображаться в пуш-уведомлении
     */
    public void addCustomMessageWithLink(Collection<Long> clientIds, String link, String text) {
        ShardedData<Long> shardedData = shardHelper.groupByShard(clientIds, ShardKey.CLIENT_ID);
        shardedData.chunkedBy(MASS_INSERT_CHUNK_SIZE).forEach((shard, clientIdsList) -> {
            List<EventLog> events = new ArrayList<>();
            for (Long clientId : clientIdsList) {
                EventLog eventLog = new EventLog()
                        .withClientId(clientId)
                        .withType(EventLogType.CUSTOM_MESSAGE_WITH_LINK)
                        .withParams(new EventLogParams()
                                .withLink(link)
                                .withText(text)
                        );
                setDefaults(eventLog);
                events.add(eventLog);
            }
            eventLogRepository.addEventLogs(shard, events);
        });
    }

    private static void setDefaults(EventLog eventLog) {
        eventLog.withEventTime(LocalDateTime.now())
                .withCampaignId(nvl(eventLog.getCampaignId(), EMPTY_ID))
                .withBannerId(nvl(eventLog.getBannerId(), EMPTY_ID))
                .withBidsId(nvl(eventLog.getBidsId(), EMPTY_ID));
    }

    private void addWithDefaults(EventLog eventLog, Long clientId) {
        setDefaults(eventLog);
        eventLogRepository.addEventLog(shardHelper.getShardByClientId(ClientId.fromLong(clientId)), eventLog);
    }
}
