package ru.yandex.direct.core.entity.feature.repository;

import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import javax.annotation.Nullable;

import one.util.streamex.StreamEx;
import org.jooq.Condition;
import org.jooq.Record;
import org.jooq.Record3;
import org.jooq.Result;
import org.jooq.SelectConditionStep;
import org.jooq.impl.DSL;
import org.jooq.util.mysql.MySQLDSL;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.feature.model.ClientFeature;
import ru.yandex.direct.core.entity.feature.model.FeatureIdToClientId;
import ru.yandex.direct.core.entity.feature.model.FeatureState;
import ru.yandex.direct.dbschema.ppc.tables.records.ClientsFeaturesRecord;
import ru.yandex.direct.dbutil.QueryWithoutIndex;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.sharding.ShardKey;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;
import ru.yandex.direct.multitype.entity.LimitOffset;

import static org.jooq.impl.DSL.row;
import static ru.yandex.direct.common.jooqmapperex.ReaderWriterBuildersEx.clientIdProperty;
import static ru.yandex.direct.dbschema.ppc.tables.ClientsFeatures.CLIENTS_FEATURES;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Repository
public class ClientFeaturesRepository {
    private static final int LIMIT = 10000;
    private static final int DEFAULT_CHUNK_SIZE = 5000;
    private final JooqMapperWithSupplier<ClientFeature> featureMapper;
    private final DslContextProvider dslContextProvider;
    private final ShardHelper shardHelper;

    public ClientFeaturesRepository(DslContextProvider dslContextProvider, ShardHelper shardHelper) {
        this.dslContextProvider = dslContextProvider;
        this.shardHelper = shardHelper;
        this.featureMapper = JooqMapperWithSupplierBuilder.builder(ClientFeature::new)
                .map(property(ClientFeature.ID, CLIENTS_FEATURES.FEATURE_ID))
                .map(clientIdProperty(ClientFeature.CLIENT_ID, CLIENTS_FEATURES.CLIENT_ID))
                .map(convertibleProperty(ClientFeature.STATE, CLIENTS_FEATURES.IS_ENABLED,
                        this::longToFeatureState, this::featureStateToLong))
                .build();
    }

    public Map<Long, List<ClientFeature>> getClientsWithFeatures(Collection<Long> featureIds) {
        return getClientsWithFeatures(featureIds, null);
    }

    public Map<Long, List<ClientFeature>> getClientsWithFeatures(Collection<Long> featureIds,
                                                                 @Nullable FeatureState state) {
        return StreamEx.of(shardHelper.dbShards())
                .flatMap(shard -> clientsWithFeaturesByShard(shard, featureIds, state, new LimitOffset(LIMIT, 0)))
                .groupingBy(ClientFeature::getId);
    }

    public void deleteClientFeatures(List<ClientFeature> clientFeatures) {
        shardHelper.groupByShard(clientFeatures, ShardKey.CLIENT_ID, feature -> feature.getClientId().asLong())
                .forEach(this::deleteClientFeatures);
    }

    public void deleteClientFeatures(int shard, List<ClientFeature> clientFeatures) {
        dslContextProvider.ppc(shard)
                .deleteFrom(CLIENTS_FEATURES)
                .where(row(CLIENTS_FEATURES.CLIENT_ID, CLIENTS_FEATURES.FEATURE_ID)
                        .in(mapList(clientFeatures,
                                feature -> row(feature.getClientId().asLong(), feature.getId()))))
                .execute();
    }

    public void deleteFeatureFromAllClients(Long featureId) {
        shardHelper.forEachShard(shard -> deleteFeature(shard, featureId));
    }

    public void deleteClientsFeatures(Collection<? extends FeatureIdToClientId> clientsFeatures) {
        shardHelper.groupByShard(clientsFeatures, ShardKey.CLIENT_ID, FeatureIdToClientId::getClientId)
                .chunkedBy(DEFAULT_CHUNK_SIZE)
                .forEach(this::deleteClientsFeatures);
    }

    private void deleteClientsFeatures(int shard, Collection<? extends FeatureIdToClientId> clientsFeatures) {
        var rowsToDelete = mapList(clientsFeatures, f -> row(f.getId(), f.getClientId().asLong()));
        dslContextProvider.ppc(shard)
                .deleteFrom(CLIENTS_FEATURES)
                .where(row(CLIENTS_FEATURES.FEATURE_ID, CLIENTS_FEATURES.CLIENT_ID).in(rowsToDelete))
                .execute();
    }

    @QueryWithoutIndex("Используется во внутренних отчетах")
    private void deleteFeature(int shard, Long featureId) {
        dslContextProvider.ppc(shard)
                .deleteFrom(CLIENTS_FEATURES)
                .where(CLIENTS_FEATURES.FEATURE_ID.eq(featureId))
                .execute();
    }

    public Map<Long, FeatureState> getClientFeatureStatesById(int shard,
                                                              ClientId clientId,
                                                              Collection<Long> featuresIds) {
        return dslContextProvider.ppc(shard)
                .select(CLIENTS_FEATURES.FEATURE_ID, CLIENTS_FEATURES.IS_ENABLED)
                .from(CLIENTS_FEATURES)
                .where(CLIENTS_FEATURES.CLIENT_ID.eq(clientId.asLong())
                        .and(CLIENTS_FEATURES.FEATURE_ID.in(featuresIds)))
                .fetchMap(CLIENTS_FEATURES.FEATURE_ID, r -> longToFeatureState(r.get(CLIENTS_FEATURES.IS_ENABLED)));
    }

    public void addClientsFeatures(Collection<ClientFeature> ownersFeatures) {
        shardHelper.groupByShard(ownersFeatures, ShardKey.CLIENT_ID, ClientFeature::getClientId)
                .chunkedBy(DEFAULT_CHUNK_SIZE)
                .forEach(this::addClientsFeatures);
    }

    public List<ClientFeature> getClientsFeaturesStatus(Collection<? extends FeatureIdToClientId> clientsFeatures,
                                                        boolean filterFeatureIds) {
        return shardHelper
                .groupByShard(clientsFeatures, ShardKey.CLIENT_ID, FeatureIdToClientId::getClientId)
                .chunkedBy(DEFAULT_CHUNK_SIZE)
                .stream()
                .flatMapKeyValue(
                        (shard, chunk) -> clientsFeatures(shard,
                                StreamEx.of(chunk).map(FeatureIdToClientId::getClientId).map(ClientId::asLong).toSet(),
                                filterFeatureIds ?
                                        Optional.of(StreamEx.of(chunk).map(FeatureIdToClientId::getId).toSet()) :
                                        Optional.empty()
                        )
                )
                .toList();
    }

    public List<ClientFeature> getClientsFeaturesStatus(Collection<? extends FeatureIdToClientId> clientsFeatures) {
        return getClientsFeaturesStatus(clientsFeatures, true);
    }

    private void addClientsFeatures(int shard, List<ClientFeature> features) {
        InsertHelper<ClientsFeaturesRecord> insertHelper =
                new InsertHelper<>(dslContextProvider.ppc(shard), CLIENTS_FEATURES);
        for (ClientFeature feature : features) {
            insertHelper.add(featureMapper, feature).newRecord();
        }
        insertHelper.onDuplicateKeyUpdate()
                .set(CLIENTS_FEATURES.IS_ENABLED, MySQLDSL.values(CLIENTS_FEATURES.IS_ENABLED));
        insertHelper.execute();
    }

    private StreamEx<ClientFeature> clientsFeatures(int shard,
                                                    Set<Long> clientIds,
                                                    Optional<Set<Long>> maybeFeatureIds) {
        Condition cond = CLIENTS_FEATURES.CLIENT_ID.in(clientIds);
        if (maybeFeatureIds.isPresent()) {
            cond = cond.and(CLIENTS_FEATURES.FEATURE_ID.in(maybeFeatureIds.get()));
        }
        SelectConditionStep<Record> step = dslContextProvider.ppc(shard)
                .select(featureMapper.getFieldsToRead())
                .from(CLIENTS_FEATURES)
                .where(cond);

        Result<Record> result = step.fetch();
        return StreamEx.of(result).map(featureMapper::fromDb);

    }

    @QueryWithoutIndex("Используется во внутренних отчетах")
    private StreamEx<ClientFeature> clientsWithFeaturesByShard(int shard, Collection<Long> featureIds,
                                                               @Nullable FeatureState state,
                                                               LimitOffset limitOffset) {
        var condition = CLIENTS_FEATURES.FEATURE_ID.in(featureIds);
        if (state != null) {
            condition = condition
                    .and(CLIENTS_FEATURES.IS_ENABLED.eq(featureStateToLong(state)));
        }
        Result<Record> result = dslContextProvider.ppc(shard)
                .select(featureMapper.getFieldsToRead())
                .from(CLIENTS_FEATURES)
                .where(condition)
                .limit(limitOffset.limit())
                .offset(limitOffset.offset())
                .fetch();
        return StreamEx.of(result).map(featureMapper::fromDb);
    }

    private FeatureState longToFeatureState(Long featureState) {
        return Optional.ofNullable(featureState)
                .map(isEnabled -> isEnabled.equals(1L) ? FeatureState.ENABLED : FeatureState.DISABLED)
                .orElse(FeatureState.UNKNOWN);
    }

    private Long featureStateToLong(FeatureState state) {
        if (state == null) {
            throw new IllegalArgumentException("can't save state with null value");
        }
        return !FeatureState.UNKNOWN.equals(state) ? (FeatureState.ENABLED.equals(state) ? 1L : 0L) : null;
    }

    /**
     * Информация о кол-ве включенных/выключенных фич для клиентов (clients_features)
     */
    public Map<Long, Map<Boolean, Integer>> featuresStateSummary(int shard) {
        Result<Record3<Long, Long, Integer>> grouped = dslContextProvider.ppc(shard)
                .select(CLIENTS_FEATURES.FEATURE_ID, CLIENTS_FEATURES.IS_ENABLED, DSL.count().as("count"))
                .from(CLIENTS_FEATURES)
                .groupBy(CLIENTS_FEATURES.FEATURE_ID, CLIENTS_FEATURES.IS_ENABLED)
                .fetch();
        Map<Long, Map<Boolean, Integer>> result = new HashMap<>();
        for (Record3<Long, Long, Integer> record : grouped) {
            Map<Boolean, Integer> map = result.getOrDefault(record.component1(), new HashMap<>());
            boolean state = record.component2().equals(1L);
            map.put(state, record.component3());
            result.put(record.component1(), map);
        }
        return result;
    }
}
