package ru.yandex.direct.core.entity.feature.repository;

import java.util.Collection;
import java.util.List;
import java.util.Map;

import one.util.streamex.EntryStream;
import org.jooq.Field;
import org.jooq.Record1;
import org.jooq.Select;
import org.jooq.SelectConditionStep;
import org.jooq.impl.DSL;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.feature.model.EventType;
import ru.yandex.direct.core.entity.feature.model.FeatureHistory;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.utils.JsonUtils;

import static ru.yandex.direct.dbschema.ppcdict.tables.FeaturesHistory.FEATURES_HISTORY;
import static ru.yandex.direct.jooqmapper.JooqMapperUtils.makeCaseStatement;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;

@Repository
public class FeatureHistoryRepository {
    public static final String EMPTY_FEATURE_SETTINGS = "{}";

    private final JooqMapperWithSupplier<FeatureHistory> featureHistoryMapper;
    private final DslContextProvider dslContextProvider;

    public FeatureHistoryRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
        this.featureHistoryMapper = JooqMapperWithSupplierBuilder.builder(FeatureHistory::new)
                .map(property(FeatureHistory.ID, FEATURES_HISTORY.ID))
                .map(property(FeatureHistory.FEATURE_ID, FEATURES_HISTORY.FEATURE_ID))
                .map(property(FeatureHistory.FEATURE_TEXT_ID, FEATURES_HISTORY.FEATURE_TEXT_ID))
                .map(property(FeatureHistory.EVENT_TIME, FEATURES_HISTORY.EVENT_TIME))
                .map(convertibleProperty(FeatureHistory.EVENT_TYPE, FEATURES_HISTORY.EVENT_TYPE,
                        EventType::fromSource, EventType::toSource))
                .map(property(FeatureHistory.OPERATOR_UID, FEATURES_HISTORY.OPERATOR_UID))
                .map(convertibleProperty(FeatureHistory.NEW_SETTINGS, FEATURES_HISTORY.NEW_SETTINGS,
                        FeatureMappings::fromDb, JsonUtils::toJson))
                .build();
    }

    /**
     * Возвращает историю фичи по списку featureId
     * свежие записи идут первыми в списке
     */
    public Map<Long, List<FeatureHistory>> getFeatureIdToHistory(Collection<Long> featureIds) {
        return dslContextProvider.ppcdict()
                .select(featureHistoryMapper.getFieldsToRead())
                .from(FEATURES_HISTORY)
                .where(FEATURES_HISTORY.FEATURE_ID.in(featureIds))
                .and(FEATURES_HISTORY.NEW_SETTINGS.notEqual(EMPTY_FEATURE_SETTINGS))
                .orderBy(FEATURES_HISTORY.ID.desc())
                .fetchGroups(FEATURES_HISTORY.FEATURE_ID, featureHistoryMapper::fromDb);
    }

    public FeatureHistory getFeatureHistory(Long featureHistoryId) {
        return dslContextProvider.ppcdict()
                .select(featureHistoryMapper.getFieldsToRead())
                .from(FEATURES_HISTORY)
                .where(FEATURES_HISTORY.ID.eq(featureHistoryId))
                .fetchOne(featureHistoryMapper::fromDb);
    }

    /**
     * Достать json настроек фичи для featureHistoryId
     */
    public String getPlainFeatureSettings(Long featureHistoryId) {
        return dslContextProvider.ppcdict()
                .select(FEATURES_HISTORY.NEW_SETTINGS)
                .from(FEATURES_HISTORY)
                .where(FEATURES_HISTORY.ID.eq(featureHistoryId))
                .fetchOne(FEATURES_HISTORY.NEW_SETTINGS);
    }

    /**
     * Возвращает историю фичей <b>после</b> события {@code id}
     * @return feature_id -> [events]
     */
    public Map<Long, List<FeatureHistory>> getFeatureHistorySince(Long id) {
        return dslContextProvider.ppcdict()
                .select(featureHistoryMapper.getFieldsToRead())
                .from(FEATURES_HISTORY)
                .where(FEATURES_HISTORY.ID.gt(id))
                .orderBy(FEATURES_HISTORY.ID)
                .fetchGroups(FEATURES_HISTORY.FEATURE_ID, featureHistoryMapper::fromDb);
    }

    /**
     * Выбирает предыдущую запись для каждой из переданных фич
     */
    public Map<Long, FeatureHistory> getPrevFeatureSettings(Map<Long, FeatureHistory> itemsByFeatureId) {
        Map<Long, Long> featureIdToItemId = EntryStream.of(itemsByFeatureId)
                .mapValues(FeatureHistory::getId)
                .toMap();
        Field<Long> itemIdField =
                makeCaseStatement(FEATURES_HISTORY.FEATURE_ID, FEATURES_HISTORY.ID, featureIdToItemId);

        Select<Record1<Long>> idSelect = dslContextProvider.ppcdict()
                .select(DSL.max(FEATURES_HISTORY.ID))
                .from(FEATURES_HISTORY)
                .where(FEATURES_HISTORY.ID.lt(itemIdField))
                .and(FEATURES_HISTORY.NEW_SETTINGS.notEqual(EMPTY_FEATURE_SETTINGS))
                .groupBy(FEATURES_HISTORY.FEATURE_ID);

        return dslContextProvider.ppcdict()
                .select(featureHistoryMapper.getFieldsToRead())
                .from(FEATURES_HISTORY)
                .where(FEATURES_HISTORY.ID.in(idSelect))
                .fetchMap(FEATURES_HISTORY.FEATURE_ID, featureHistoryMapper::fromDb);
    }

    /**
     * Достать json настроек фичи соотвествующий состоянию фичи которое было перед lastFeatureHistoryId.
     * Пустые настройки "{}" игноируются
     */
    public String getPrevPlainFeatureSettings(String featureTextId, Long lastFeatureHistoryId) {
        SelectConditionStep<Record1<Long>> nestedMaxIdSelect = dslContextProvider.ppcdict()
                .select(DSL.max(FEATURES_HISTORY.ID))
                .from(FEATURES_HISTORY)
                .where(FEATURES_HISTORY.FEATURE_TEXT_ID.eq(featureTextId))
                .and(FEATURES_HISTORY.ID.lessThan(lastFeatureHistoryId))
                .and(FEATURES_HISTORY.NEW_SETTINGS.notEqual(EMPTY_FEATURE_SETTINGS));

        return dslContextProvider.ppcdict()
                .select(FEATURES_HISTORY.NEW_SETTINGS)
                .from(FEATURES_HISTORY)
                .where(FEATURES_HISTORY.ID.eq(nestedMaxIdSelect))
                .fetchOne(FEATURES_HISTORY.NEW_SETTINGS);
    }
}
