package ru.yandex.direct.core.entity.feature.repository;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import one.util.streamex.EntryStream;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.core.entity.feature.model.Feature;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;
import ru.yandex.direct.jooqmapperhelper.JooqUpdateBuilder;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.utils.JsonUtils;

import static org.jooq.impl.DSL.choose;
import static ru.yandex.direct.dbschema.ppcdict.tables.Features.FEATURES;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Repository
public class FeatureRepository {
    private final JooqMapperWithSupplier<Feature> featureMapper;
    private final DslContextProvider dslContextProvider;

    public FeatureRepository(DslContextProvider dslContextProvider) {
        this.dslContextProvider = dslContextProvider;
        this.featureMapper = JooqMapperWithSupplierBuilder.builder(Feature::new)
                .map(property(Feature.ID, FEATURES.FEATURE_ID))
                .map(property(Feature.FEATURE_TEXT_ID, FEATURES.FEATURE_TEXT_ID))
                .map(property(Feature.FEATURE_PUBLIC_NAME, FEATURES.FEATURE_PUBLIC_NAME))
                .map(property(Feature.FEATURE_TICKET, FEATURES.TICKET_ID))
                .map(convertibleProperty(Feature.SETTINGS, FEATURES.SETTINGS,
                        FeatureMappings::fromDb, JsonUtils::toJson))
                .build();
    }

    /**
     * Возвращает полный список фич.
     */
    public List<Feature> get() {
        var records = dslContextProvider.ppcdict()
                .select(featureMapper.getFieldsToRead())
                .from(FEATURES)
                .fetch();
        return mapList(records, featureMapper::fromDb);
    }

    /**
     * Возвращает список фич с конкретнымы id.
     */
    public List<Feature> get(Collection<Long> featureIds) {
        var record = dslContextProvider.ppcdict()
                .select(featureMapper.getFieldsToRead())
                .from(FEATURES)
                .where(FEATURES.FEATURE_ID.in(featureIds))
                .fetch();
        return mapList(record, featureMapper::fromDb);
    }

    /**
     * Возвращает фичу по featureTextId
     */
    public Optional<Feature> getByFeatureTextId(String featureTextId) {
       return dslContextProvider.ppcdict()
                .select(featureMapper.getFieldsToRead())
                .from(FEATURES)
                .where(FEATURES.FEATURE_TEXT_ID.eq(featureTextId))
                .fetchOptional(featureMapper::fromDb);
    }

    /**
     * Обновляет сторожевой тикет для фичи.
     */
    public void updateTicketId(String featureTextId, String ticketId) {
        dslContextProvider.ppcdict()
                .update(FEATURES)
                .set(FEATURES.TICKET_ID, ticketId)
                .where(FEATURES.FEATURE_TEXT_ID.eq(featureTextId))
                .execute();
    }

    /**
     * Массовое обновление сторожевого тикета для фичи.
     */
    public void updateTicketsIds(Map<String, String> featureTextIdToTicketId) {
        if (featureTextIdToTicketId.isEmpty()) {
            return;
        }


        dslContextProvider.ppcdict()
                .update(FEATURES)
                .set(FEATURES.TICKET_ID,
                        choose(FEATURES.FEATURE_TEXT_ID).mapValues(featureTextIdToTicketId).otherwise(FEATURES.TICKET_ID))
                .where(FEATURES.FEATURE_TEXT_ID.in(featureTextIdToTicketId.keySet()))
                .execute();
    }

    /**
     * Обновляет фичу.
     */
    public void update(Collection<AppliedChanges<Feature>> appliedChanges) {
        var updateBuilder = new JooqUpdateBuilder<>(FEATURES.FEATURE_ID, appliedChanges);
        updateBuilder.processProperty(Feature.SETTINGS, FEATURES.SETTINGS, JsonUtils::toJson);
        dslContextProvider.ppcdict()
                .update(FEATURES)
                .set(updateBuilder.getValues())
                .where(FEATURES.FEATURE_ID.in(updateBuilder.getChangedIds()))
                .execute();
    }

    /**
     * Добавляет список фич.
     */
    public void add(List<Feature> featureList) {
        var insertHelper = new InsertHelper<>(dslContextProvider.ppcdict(), FEATURES);
        insertHelper.addAll(featureMapper, featureList);
        List<Long> featureIds = insertHelper.executeIfRecordsAddedAndReturn(FEATURES.FEATURE_ID);
        EntryStream.zip(featureList, featureIds)
                .forKeyValue(Feature::withId);
    }

    /**
     * Удаляет фичу по id.
     *
     * @return true, если фича была удалена
     */
    public boolean delete(Long featureId) {
        return dslContextProvider.ppcdict()
                .deleteFrom(FEATURES)
                .where(FEATURES.FEATURE_ID.equal(featureId))
                .execute() > 0;
    }
}
