package ru.yandex.direct.core.entity.feature.service;

import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Predicate;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.feature.container.ClientRealFeature;
import ru.yandex.direct.core.entity.feature.container.FeatureRequest;
import ru.yandex.direct.core.entity.feature.container.FeaturesWithExpBoxes;
import ru.yandex.direct.core.entity.feature.model.ClientFeature;
import ru.yandex.direct.core.entity.feature.model.Feature;
import ru.yandex.direct.core.entity.feature.model.FeatureState;
import ru.yandex.direct.dbutil.model.ClientId;

import static com.google.common.base.Preconditions.checkState;

/**
 * Хранилище для работы с фичами для клиентов.
 * Внутри хранит фичи, как мапу clientId -> featureId -> clientRealFeature
 */
class ClientFeaturesStorage {

    private Map<ClientId, FeaturesWithExpBoxes> clientIdFeaturesWithExpBoxesMap;

    ClientFeaturesStorage() {
        clientIdFeaturesWithExpBoxesMap = new HashMap<>();
    }

    ClientFeaturesStorage(Collection<FeatureRequest> featureRequests, Collection<Feature> features) {
        clientIdFeaturesWithExpBoxesMap = StreamEx.of(featureRequests)
                .filter(featureRequest -> Objects.nonNull(featureRequest.getClientId()))
                .distinct()
                .mapToEntry(FeatureRequest::getClientId,
                        featureRequest -> new FeaturesWithExpBoxes(StreamEx.of(features)
                                .mapToEntry(state -> FeatureState.UNKNOWN)
                                .mapToValue((feature, state) -> new ClientRealFeature()
                                        .withClientId(featureRequest.getClientId())
                                        .withFeature(feature)
                                        .withFeatureState(FeatureState.UNKNOWN))
                                .mapKeys(Feature::getId)
                                .toMap()))
                .toMap();
    }

    /**
     * Возвращает список клиентских реальных фич
     */
    List<ClientRealFeature> getClientRealFeatureList() {
        return StreamEx.ofValues(clientIdFeaturesWithExpBoxesMap)
                .flatMap(featuresWithExpBoxes -> featuresWithExpBoxes.getIdToFeatureMap().values().stream())
                .toList();
    }

    Map<ClientId, FeaturesWithExpBoxes> getEnabled() {
        return getFilteredFeatures(featureState -> featureState == FeatureState.ENABLED);
    }

    FeaturesWithExpBoxes getEnabled(ClientId clientId) {
        return getFilteredFeatures(clientId,
                clientRealFeature -> clientRealFeature.getFeatureState() == FeatureState.ENABLED);
    }

    /**
     * Возвращает включенные клиентские  фичи
     */
    List<ClientRealFeature> getEnabledFeatures() {
        return getFilteredClientFeatures(clientRealFeature -> FeatureState.ENABLED == clientRealFeature.getFeatureState());
    }

    Map<ClientId, FeaturesWithExpBoxes> getUnknown() {
        return getFilteredFeatures(featureState -> featureState == FeatureState.UNKNOWN);
    }

    FeaturesWithExpBoxes getUnknown(ClientId clientId) {
        return getFilteredFeatures(clientId,
                clientRealFeature -> clientRealFeature.getFeatureState() == FeatureState.UNKNOWN);
    }

    /**
     * Возвращает только клиентские фичи с неопределенным статусом
     */
    List<ClientRealFeature> getUnknownFeatures() {
        return getFilteredClientFeatures(clientRealFeature -> FeatureState.UNKNOWN == clientRealFeature.getFeatureState());
    }

    /**
     * Возвращает только клиентские фичи с определенным статусом
     */
    List<ClientRealFeature> getKnownFeatures() {
        return getFilteredClientFeatures(clientRealFeature -> FeatureState.UNKNOWN != clientRealFeature.getFeatureState());
    }

    /**
     * Возвращает только включенные и публичные клиентские фичи
     */
    List<ClientRealFeature> getPublicFeaturesList() {
        return getFilteredClientFeatures(clientRealFeature ->
                clientRealFeature.getFeatureState() == FeatureState.ENABLED
                        && clientRealFeature.getFeature().getSettings().getIsPublic());
    }

    /**
     * Обновляет клиентские фичи в хранилище
     *
     * @param clientFeatures список клиентов фич
     */
    void update(Collection<ClientFeature> clientFeatures) {
        clientFeatures.forEach(clientFeature -> find(clientFeature.getClientId(), clientFeature.getId())
                .withFeatureState(clientFeature.getState()));
    }

    /**
     * Обновляет клиентские фичи в хранилище, пропускает фичи из переданного списка, которых в хранилище нет
     *
     * @param clientFeatures список клиентов фич
     */
    void updateIfPresentAndUnknown(Collection<ClientFeature> clientFeatures) {
        clientFeatures.forEach((clientFeature) -> {
            var found = findOrNull(clientFeature.getClientId(), clientFeature.getId());
            if (found != null && found.getFeatureState() == FeatureState.UNKNOWN) {
                found.withFeatureState(clientFeature.getState());
            }
        });
    }

    void setExpBoxes(ClientId clientId, String exbBoxes, String expBoxesCrypted) {
        var featureWithExpBoxes = clientIdFeaturesWithExpBoxesMap.get(clientId);
        checkState(featureWithExpBoxes != null, "client with clientId: {} not found", clientId);
        featureWithExpBoxes.setExpBoxes(exbBoxes);
        featureWithExpBoxes.setExpBoxesCrypted(expBoxesCrypted);
    }

    private ClientRealFeature findOrNull(ClientId clientId, Long featureId) {
        return Optional.ofNullable(clientIdFeaturesWithExpBoxesMap.get(clientId))
                .map(featuresWithExpBoxes -> featuresWithExpBoxes.getIdToFeatureMap().get(featureId))
                .orElse(null);
    }

    private ClientRealFeature find(ClientId clientId, Long featureId) {
        return Optional.ofNullable(clientIdFeaturesWithExpBoxesMap.get(clientId))
                .map(featuresWithExpBoxes -> featuresWithExpBoxes.getIdToFeatureMap().get(featureId))
                .orElseThrow(() -> new IllegalArgumentException(
                        "feature with feature_id:" + featureId + " client with clientId: "
                                + clientId + " not found"));
    }

    private Map<ClientId, FeaturesWithExpBoxes> getFilteredFeatures(Predicate<FeatureState> featureStatePredicate) {
        return EntryStream.of(clientIdFeaturesWithExpBoxesMap)
                .mapValues(clientIdFeaturesWithExpBoxesEntry -> new FeaturesWithExpBoxes(EntryStream.of(clientIdFeaturesWithExpBoxesEntry.getIdToFeatureMap())
                        .filterValues(clientRealFeature -> featureStatePredicate.test(clientRealFeature.getFeatureState()))
                        .toMap()).withExpBoxesCrypted(clientIdFeaturesWithExpBoxesEntry.getExpBoxesCrypted()).withExpBoxes(clientIdFeaturesWithExpBoxesEntry.getExpBoxes()))
                .toMap();
    }

    private FeaturesWithExpBoxes getFilteredFeatures(ClientId clientId,
                                                     Predicate<ClientRealFeature> clientRealFeaturePredicate) {
        var featuresWithExpBoxesOld = clientIdFeaturesWithExpBoxesMap.get(clientId);
        return new FeaturesWithExpBoxes(EntryStream.of(featuresWithExpBoxesOld.getIdToFeatureMap())
                .filterValues(clientRealFeaturePredicate)
                .toMap()).withExpBoxesCrypted(featuresWithExpBoxesOld.getExpBoxesCrypted()).withExpBoxes(featuresWithExpBoxesOld.getExpBoxes());
    }


    private List<ClientRealFeature> getFilteredClientFeatures(Predicate<ClientRealFeature> clientRealFeaturePredicate) {
        return StreamEx.of(clientIdFeaturesWithExpBoxesMap.values())
                .flatMap(clientIdFeaturesWithExpBoxesEntry -> StreamEx.of(clientIdFeaturesWithExpBoxesEntry.getIdToFeatureMap().values()))
                .filter(clientRealFeaturePredicate)
                .toList();
    }
}
