package ru.yandex.direct.core.entity.feature.service

import org.slf4j.LoggerFactory
import org.springframework.context.ApplicationListener
import org.springframework.security.core.context.SecurityContextHolder
import org.springframework.stereotype.Service
import org.springframework.web.context.support.ServletRequestHandledEvent
import ru.yandex.direct.core.security.DirectAuthentication
import ru.yandex.direct.dbutil.model.ClientId
import java.io.PrintWriter
import java.io.StringWriter

/**
 * Сервис для задания/получения клиента и оператора
 * Для внешних запросов используется клиент и оператор из [SecurityContextHolder]
 * Для всего остального в качестве хранилища используется [ThreadLocal]
 * Так же можно переопределить клиента и оператора для внешних запросов в случае необходимости
 * (например для валидации при копировании)
 */
@Service
class DirectAuthContextService : ContextSupplier, ApplicationListener<ServletRequestHandledEvent> {
    init {
        if (logger.isTraceEnabled) {
            try {
                throw RuntimeException()
            } catch (e: Exception) {
                val sw = StringWriter()
                e.printStackTrace(PrintWriter(sw))
                logger.debug(
                    "DirectAuthContextService initialized in thread: {}-{} as $this. Stacktrace: $sw",
                    Thread.currentThread().name, Thread.currentThread().id
                )
            }
        }
    }

    private val threadLocalClientId = ThreadLocal<ClientId?>()
    private val threadLocalUid = ThreadLocal<Long?>()

    /**
     * Задать клиента для проверки фичей
     */
    fun usingClientId(clientId: ClientId?) {
        threadLocalClientId.set(clientId)
        logger.debug(
            "DirectAuthContextService instance is $this, set localthread client to {}",
            threadLocalClientId.get()
        )
    }

    /**
     * Задать оператора для проверки фичей
     */
    fun usingOperatorUid(uid: Long?) {
        threadLocalUid.set(uid)
        logger.debug("DirectAuthContextService instance is $this, set localthread uid to {}", threadLocalUid.get())
    }

    /**
     * Очистить текущие значения клиента и оператора из ThreadLocal
     */
    fun clearThreadLocal() {
        threadLocalClientId.remove()
        threadLocalUid.remove()
    }

    /**
     * Получить clientId клиента из текущего контекста. Приоритетно возвращается предварительно установленный клиент в [usingClientId]
     * Если он не установлен, то получается из SecurityContextHolder(который заполняется если запрос пришёл с аутентификацией(web или api))
     * Если клиент не установлен, кидается исключение
     * @throws EmptyAuthenticationException
     */
    override val clientId: ClientId
        get() {
            logger.debug("DirectAuthContextService instance $this, localthread clientid {}", threadLocalClientId.get())
            if (threadLocalClientId.get() != null) {
                return threadLocalClientId.get()!!
            } else {
                val auth = authentication()
                    ?: throw EmptyAuthenticationException(
                        "В SecurityContextHolder нет информации об аутентификации. " +
                            "В тестах используйте FeatureSteps.setCurrentClient. " +
                            "В приложениях без внешних клиентских запросов DirectAuthContextService.usingClientId"
                    )
                val user = auth.subjectUser
                    ?: throw EmptyAuthenticationException("В SecurityContextHolder нет информации о клиенте")
                return user.clientId
            }
        }

    /**
     * Получить uid оператора из текущего контекста. Приоритетно возвращается предварительно установленный оператор в [usingOperatorUid]
     * Если он не установлен, то получается из SecurityContextHolder(который заполняется если запрос пришёл с аутентификацией(web или api))
     * Если оператор не установлен, кидается исключение
     * @throws EmptyAuthenticationException
     */
    override val operatorUid: Long
        get() {
            logger.debug("DirectAuthContextService instance $this, localthread uid {}", threadLocalUid.get())
            if (threadLocalUid.get() != null) {
                return threadLocalUid.get()!!
            } else {
                val auth = authentication()
                    ?: throw EmptyAuthenticationException(
                        "В SecurityContextHolder нет информации об аутентификации. " +
                            "В тестах используйте FeatureSteps.setCurrentOperator. " +
                            "В приложениях без внешних клиентских запросов DirectAuthContextService.usingUid"
                    )
                val operator = auth.operator
                    ?: throw EmptyAuthenticationException("В SecurityContextHolder нет информации об операторе")
                return operator.uid
            }
        }

    companion object {
        private val logger = LoggerFactory.getLogger(DirectAuthContextService::class.java)

        fun authentication(): DirectAuthentication? {
            val authentication = SecurityContextHolder.getContext().authentication
            return authentication as? DirectAuthentication
        }
    }

    /**
     * Очищать LocalThread при любом событии [ServletRequestHandledEvent]
     * (т.е. когда работа идёт в контексте spring web, это api и web)
     */
    override fun onApplicationEvent(event: ServletRequestHandledEvent) {
        clearThreadLocal()
        logger.debug("localthread cleared")
    }
}

interface ContextSupplier {
    val clientId: ClientId
    val operatorUid: Long
}
