package ru.yandex.direct.core.entity.feature.service;

import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Supplier;
import com.google.common.base.Suppliers;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.feature.model.Feature;
import ru.yandex.direct.core.entity.feature.repository.FeatureRepository;

/**
 * Кэш для элементов таблицы фич.
 * Является надстройкой над {@link FeatureRepository},
 */
@Service
@ParametersAreNonnullByDefault
public class FeatureCache {
    private static final int CACHE_RELOAD_INTERVAL_IN_SECONDS = 10;

    private final FeatureRepository featureRepository;

    private volatile Supplier<List<Feature>> featureSupplier;

    @Autowired
    public FeatureCache(FeatureRepository featureRepository) {
        Objects.requireNonNull(featureRepository, "featureRepository");
        this.featureRepository = featureRepository;
        invalidate();
    }

    /**
     * Инвалидирует кэш при изменении в базе данных фич.
     */
    public void invalidate() {
        this.featureSupplier = Suppliers.memoizeWithExpiration(
                this::getData,
                CACHE_RELOAD_INTERVAL_IN_SECONDS,
                TimeUnit.SECONDS);
    }

    private List<Feature> getData() {
        return featureRepository.get().stream().collect(Collectors.toUnmodifiableList());
    }

    /**
     * Возвращает кэшированый список фич из базы.
     */
    public List<Feature> getCached() {
        return featureSupplier.get();
    }

    /**
     * Возвращает кэшированый список фич с конкретными id из базы.
     */
    public List<Feature> getCached(Collection<Long> featureIds) {
        var featureIdsSet = new HashSet<>(featureIds);
        return this.getCached().stream()
                .filter(f -> featureIdsSet.contains(f.getId()))
                .collect(Collectors.toUnmodifiableList());
    }

    /**
     * Возвращает кэшированый список фич по id текста из базы.
     */
    public List<Feature> getCachedByTextId(Collection<String> featureTextIds) {
        var featureTextIdsSet = new HashSet<>(featureTextIds);
        return this.getCached().stream()
                .filter(f -> featureTextIdsSet.contains(f.getFeatureTextId()))
                .collect(Collectors.toUnmodifiableList());
    }

    /**
     * Возвращает кэшированый словарь id фич по id текста из базы.
     */
    public Map<String, Long> getIdsByTextId(Collection<String> featureTextIds) {
        var featureTextIdsSet = new HashSet<>(featureTextIds);
        return this.getCached().stream()
                .filter(f -> featureTextIdsSet.contains(f.getFeatureTextId()))
                .collect(Collectors.toMap(Feature::getFeatureTextId, Feature::getId));
    }
}
