package ru.yandex.direct.core.entity.feature.service

import org.springframework.context.annotation.Lazy
import org.springframework.security.core.context.SecurityContextHolder
import org.springframework.stereotype.Service
import ru.yandex.direct.feature.FeatureName
import ru.yandex.direct.validation.builder.When
import java.util.function.Supplier
import javax.annotation.PreDestroy

fun FeatureName.enabled(): Boolean = FeatureHelper.feature(this).enabled()

fun FeatureName.disabled(): Boolean = FeatureHelper.feature(this).disabled()

fun <T, D> FeatureName.whenEnabled(): When<T, D> = FeatureHelper.whenFeatureEnabled(this)

fun <T, D> FeatureName.whenDisabled(): When<T, D> = FeatureHelper.whenFeatureDisabled(this)

fun <T, D> FeatureName.whenEnabledForOperator(): When<T, D> = FeatureHelper.whenOperatorFeatureEnabled(this)

fun <T, D> FeatureName.whenDisabledForOperator(): When<T, D> = FeatureHelper.whenOperatorFeatureDisabled(this)

/**
 * Позволяет в статическом контексте проверять доступность фичей для клиента и оператора
 * Информация о клиенте и операторе получается из [ClientIdSupplier] и [UidSupplier] определяемых в спринговой конфигурации.
 * Для приложений, где есть контекст запроса используется [SecurityContextHolder]
 * В остальных случаях необходимо явно указывать с какими оперетором и клиентом необходимо работать
 *
 * Пример использования в kotlin
 * ```(kotlin)
 * if (FeatureName.TARGET_TAGS_ALLOWED.enabled()) {
 * //do smthg
 * }
 * ```
 *
 * ```(kotlin)
 * (ItemValidationBuilder) builder.check(CommonConstraints.notNull(), FeatureName.TARGET_TAGS_ALLOWED.whenEnabled())
 * ```
 *
 * в java используется статический метод
 * ```(java)
 * FeatureHelper.feature(FeatureName.TARGET_TAGS_ALLOWED).enabled()
 * ```
 */
@Service
@Lazy(false)
class FeatureHelper(featureService: FeatureService, contextSupplier: ContextSupplier) {
    init {
        if (Companion.featureService != null || Companion.contextSupplier != null) {
            throw IllegalStateException(
                "Cannot initialize FeatureHelper twice, check Spring Configuration, must be one per jvm run."
            )
        }
        Companion.featureService = featureService
        Companion.contextSupplier = contextSupplier
    }

    class FeatureSupplier(private val supplier: Supplier<Boolean>) : Supplier<Boolean> {
        fun enabled(): Boolean {
            return supplier.get()
        }

        fun disabled(): Boolean {
            return !supplier.get()
        }

        override fun get(): Boolean {
            return supplier.get()
        }
    }

    companion object {
        private var featureService: FeatureService? = null
        private var contextSupplier: ContextSupplier? = null

        @JvmStatic
        fun <T, D> whenFeatureEnabled(featureName: FeatureName): When<T, D> {
            return When.isTrue {
                feature(
                    featureName
                ).enabled()
            }
        }

        @JvmStatic
        fun <T, D> whenFeatureDisabled(featureName: FeatureName): When<T, D> {
            return When.isTrue {
                feature(
                    featureName
                ).disabled()
            }
        }

        @JvmStatic
        fun <T, D> whenOperatorFeatureEnabled(featureName: FeatureName): When<T, D> {
            return When.isTrue {
                operatorFeature(
                    featureName
                ).enabled()
            }
        }

        @JvmStatic
        fun <T, D> whenOperatorFeatureDisabled(featureName: FeatureName): When<T, D> {
            return When.isTrue {
                operatorFeature(
                    featureName
                ).disabled()
            }
        }

        @JvmStatic
        fun feature(featureName: FeatureName): FeatureSupplier {
            return FeatureSupplier {
                isFeatureEnabled(
                    featureName
                )
            }
        }

        @JvmStatic
        fun operatorFeature(featureName: FeatureName): FeatureSupplier {
            return FeatureSupplier {
                isFeatureEnabledForOperator(
                    featureName
                )
            }
        }

        private fun isFeatureEnabled(feature: FeatureName): Boolean {
            return featureService!!.isEnabledForClientId(
                contextSupplier!!.clientId, feature
            )
        }

        private fun isFeatureEnabledForOperator(feature: FeatureName): Boolean {
            return featureService!!.isEnabledForUid(
                contextSupplier!!.operatorUid, listOf(feature)
            )
        }
    }

    @PreDestroy
    fun cleanUp() {
        featureService = null
        contextSupplier = null
    }
}
