package ru.yandex.direct.core.entity.feature.service;

import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Nullable;

import com.google.common.collect.Sets;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.feature.FeatureAssignmentException;
import ru.yandex.direct.core.entity.feature.container.ChiefRepresentativeWithClientFeature;
import ru.yandex.direct.core.entity.feature.container.FeatureTextIdToClientIdState;
import ru.yandex.direct.core.entity.feature.container.FeatureTextIdToLoginState;
import ru.yandex.direct.core.entity.feature.container.FeatureTextIdToPercent;
import ru.yandex.direct.core.entity.feature.container.FeatureTextIdToRole;
import ru.yandex.direct.core.entity.feature.container.LoginClientIdChiefLoginWithFeature;
import ru.yandex.direct.core.entity.feature.container.LoginClientIdChiefLoginWithState;
import ru.yandex.direct.core.entity.feature.model.ClientFeature;
import ru.yandex.direct.core.entity.feature.model.Feature;
import ru.yandex.direct.core.entity.feature.model.FeatureConverter;
import ru.yandex.direct.core.entity.feature.model.FeatureSettings;
import ru.yandex.direct.core.entity.feature.model.FeatureState;
import ru.yandex.direct.core.entity.feature.repository.ClientFeaturesRepository;
import ru.yandex.direct.core.entity.feature.repository.FeatureRepository;
import ru.yandex.direct.core.entity.feature.service.validation.FeatureAddValidationService;
import ru.yandex.direct.core.entity.feature.service.validation.FeaturePercentUpdateValidationService;
import ru.yandex.direct.core.entity.feature.service.validation.FeatureRoleUpdateValidationService;
import ru.yandex.direct.core.entity.feature.service.validation.FeatureTextIdValidationService;
import ru.yandex.direct.core.entity.feature.service.validation.SwitchFeatureByClientIdValidationService;
import ru.yandex.direct.core.entity.feature.service.validation.SwitchFeatureByLoginValidationService;
import ru.yandex.direct.core.entity.user.service.UserService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.rbac.RbacRole;
import ru.yandex.direct.rbac.RbacService;
import ru.yandex.direct.result.Result;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.constraint.CommonConstraints;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static java.util.Collections.emptySet;
import static org.apache.commons.collections4.IterableUtils.first;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.defect.CommonDefects.inconsistentStateAlreadyExists;
import static ru.yandex.direct.validation.defect.CommonDefects.objectNotFound;

@Service
public class FeatureManagingService {
    private final ClientFeaturesRepository clientFeaturesRepository;
    private final FeatureRepository featureRepository;
    private final FeatureCache featureCache;

    private final SwitchFeatureByLoginValidationService switchFeatureToLoginValidationService;
    private final SwitchFeatureByClientIdValidationService switchFeatureToClientIdValidationService;
    private final FeaturePercentUpdateValidationService featurePercentUpdateValidationService;
    private final FeatureTextIdValidationService featureTextIdValidationService;
    private final FeatureRoleUpdateValidationService featureRoleUpdateValidationService;
    private final FeatureAddValidationService featureAddValidationService;

    private final ShardHelper shardHelper;
    private final RbacService rbacService;
    private final UserService userService;
    private final FeatureService featureService;

    @Autowired
    public FeatureManagingService(
            SwitchFeatureByLoginValidationService switchFeatureToLoginValidationService,
            FeatureTextIdValidationService featureTextIdValidationService,
            SwitchFeatureByClientIdValidationService switchFeatureToClientIdValidationService,
            FeatureRepository featureRepository,
            FeatureCache featureCache,
            ClientFeaturesRepository clientFeaturesRepository,
            FeaturePercentUpdateValidationService featurePercentUpdateValidationService,
            FeatureRoleUpdateValidationService featureRoleUpdateValidationService,
            FeatureAddValidationService featureAddValidationService,
            ShardHelper shardHelper,
            RbacService rbacService, UserService userService, FeatureService featureService) {
        this.clientFeaturesRepository = clientFeaturesRepository;
        this.featureRepository = featureRepository;
        this.featureCache = featureCache;

        this.switchFeatureToLoginValidationService = switchFeatureToLoginValidationService;
        this.switchFeatureToClientIdValidationService = switchFeatureToClientIdValidationService;
        this.featureTextIdValidationService = featureTextIdValidationService;
        this.featurePercentUpdateValidationService = featurePercentUpdateValidationService;
        this.featureRoleUpdateValidationService = featureRoleUpdateValidationService;
        this.featureAddValidationService = featureAddValidationService;

        this.shardHelper = shardHelper;
        this.rbacService = rbacService;
        this.userService = userService;
        this.featureService = featureService;
    }

    public static FeatureSettings getDefaultFeatureSettings() {
        return new FeatureSettings()
                .withIsAccessibleAfterDisabling(false)
                .withRoles(emptySet())
                .withPercent(0)
                .withIsPublic(false)
                .withCanDisable(Sets.newHashSet(RbacRole.SUPER.name(), RbacRole.SUPERREADER.name()))
                .withCanEnable(Sets.newHashSet(RbacRole.SUPER.name(), RbacRole.SUPERREADER.name()));
    }

    /**
     * Возвращает все фичи
     */
    public List<Feature> getFeatures() {
        return featureRepository.get();
    }

    /**
     * Возвращает все фичи из кэша
     */
    public List<Feature> getCachedFeatures() {
        return featureCache.getCached();
    }

    /**
     * Возвращает фичи для переданных uids
     */
    public List<Feature> getCachedFeatures(List<Long> uids) {
        return featureCache.getCached(uids);
    }

    /**
     * Добавляет новые фичи
     */
    public void addFeatures(List<Feature> featureList) {
        featureRepository.add(featureList);
        clearCaches();
    }

    /**
     * Добавляет новые фичи с настройками по умолчанию
     */
    public Result<List<Feature>> addFeaturesWithDefaultSettings(List<Feature> featureList) {
        return addFeaturesWithSettings(featureList, getDefaultFeatureSettings());
    }

    /**
     * Добавляет новые фичи с настройками
     */
    public Result<List<Feature>> addFeaturesWithSettings(List<Feature> featureList, FeatureSettings featureSettings) {
        ValidationResult<List<Feature>, Defect> validation = featureAddValidationService.validate(featureList);
        if (validation.hasAnyErrors()) {
            return Result.broken(validation);
        }
        Map<String, Feature> existingFeatures = listToMap(getCachedFeatures(), Feature::getFeatureTextId);
        for (Feature feature : featureList) {
            if (existingFeatures.containsKey(feature.getFeatureTextId())) {
                return Result.broken(ValidationResult.failed(feature, inconsistentStateAlreadyExists()));
            }
            feature.withSettings(featureSettings);
        }
        featureRepository.add(featureList);
        clearCaches();
        return Result.successful(featureList);
    }

    /**
     * Удаляет саму фичу из базы и все привязки к этой фиче у клиентов (из таблицы clients_features)
     */
    public Result<Long> deleteFeature(Long featureId) {
        Map<Long, Feature> existingFeatures = listToMap(getCachedFeatures(), Feature::getId);
        if (!existingFeatures.containsKey(featureId)) {
            return Result.broken(ValidationResult.failed(featureId, objectNotFound()));
        }
        clientFeaturesRepository.deleteFeatureFromAllClients(featureId);
        featureRepository.delete(featureId);
        clearCaches();
        return Result.successful(featureId);
    }

    /**
     * обновляет процент для фичи
     *
     * @param featurePercentList список textId фич к выставляемым процентам
     * @return ids обновленных фич
     */
    public Result<Collection<Long>> updateFeaturePercent(List<FeatureTextIdToPercent> featurePercentList) {
        Map<String, Feature> featuresByTextId = listToMap(getFeatures(), Feature::getFeatureTextId);
        ValidationResult<List<FeatureTextIdToPercent>, Defect> validation =
                featurePercentUpdateValidationService.validate(featurePercentList, featuresByTextId.values());
        if (validation.hasAnyErrors()) {
            return Result.broken(validation);
        }

        List<Feature> featuresToUpdate = featureRepository.get(mapList(featurePercentList,
                featureToPercent -> featuresByTextId.get(featureToPercent.getTextId()).getId()));

        Map<String, Integer> featurePercentMap = StreamEx.of(featurePercentList)
                .mapToEntry(FeatureTextIdToPercent::getTextId, FeatureTextIdToPercent::getPercent)
                .toMap();

        StreamEx.of(featuresToUpdate).forEach(
                feature -> feature.getSettings().setPercent(featurePercentMap.get(feature.getFeatureTextId())));

        List<ModelChanges<Feature>> modelChanges = StreamEx.of(featuresToUpdate)
                .map(feature -> new ModelChanges<>(feature.getId(), Feature.class)
                        .process(feature.getSettings(), Feature.SETTINGS))
                .toList();

        List<Long> validModelIds = mapList(modelChanges, ModelChanges::getId);
        Map<Long, Feature> featurePercents = StreamEx.of(featureRepository.get(validModelIds))
                .mapToEntry(Feature::getId)
                .invert()
                .toMap();

        List<AppliedChanges<Feature>> appliedChangesList = StreamEx.of(modelChanges)
                .map(x -> x.applyTo(featurePercents.get(x.getId())))
                .toList();

        featureRepository.update(appliedChangesList);
        clearCaches();
        return Result.successful(validModelIds, validation);
    }

    /**
     * добавляет или удаляет роли из списка доступных для фичи ролей
     *
     * @param featuresToRoles список textId фич к изменяемым ролям
     * @param isEnable        включать доступ у роли к фичи
     * @return ids обновленных фич
     */
    public Result<List<Long>> updateFeaturesRoles(List<FeatureTextIdToRole> featuresToRoles, Boolean isEnable) {
        Result<List<Long>> operationResult;
        if (isEnable) {
            operationResult = enableFeaturesForRoles(featuresToRoles);
        } else {
            operationResult = disableFeaturesForRoles(featuresToRoles);
        }
        return operationResult;
    }

    public Result<List<LoginClientIdChiefLoginWithState>> deleteFeaturesFromClientIds(
            List<String> features,
            List<ClientId> clientIds
    ) {
        var featuresByTextId = listToMap(getCachedFeatures(), Feature::getFeatureTextId);
        var featuresValidation = featureTextIdValidationService.validate(featuresByTextId.keySet(), features);
        if (featuresValidation.hasAnyErrors()) {
            return Result.broken(featuresValidation);
        }

        var clientIdsValidation = switchFeatureToClientIdValidationService.validateClientIds(clientIds);
        if (clientIdsValidation.hasAnyErrors()) {
            return Result.broken(clientIdsValidation);
        }

        var clientFeatureList = StreamEx.of(features).cross(clientIds)
                .mapKeyValue((feature, clientId) -> new ClientFeature()
                                .withId(featuresByTextId.get(feature).getId())
                                .withClientId(clientId))
                .toList();
        clientFeaturesRepository.deleteClientsFeatures(clientFeatureList);
        clearCaches();

        var chiefLoginsByClientIds = userService.getChiefsLoginsByClientIds(clientIds);

        var clientIdToFeature = StreamEx.of(clientIds)
                .map(clientId -> new LoginClientIdChiefLoginWithState()
                        .withClientId(clientId.asLong())
                        .withChiefLogin(chiefLoginsByClientIds.get(clientId))
                        .withFeatureState(FeatureState.DISABLED)
                )
                .toList();
        return Result.successful(clientIdToFeature);
    }

    public Result<List<LoginClientIdChiefLoginWithFeature>> deleteFeaturesFromClientIdsByLogins(
            List<String> features,
            List<String> logins
    ) {
        var featuresByTextId = listToMap(getCachedFeatures(), Feature::getFeatureTextId);
        var featuresValidation = featureTextIdValidationService.validate(featuresByTextId.keySet(), features);
        if (featuresValidation.hasAnyErrors()) {
            return Result.broken(featuresValidation);
        }

        var clientIdsByLogins = shardHelper.getClientIdsByLogins(logins);
        var loginsValidation = switchFeatureToLoginValidationService.validateLogins(
                clientIdsByLogins,
                logins
        );
        if (loginsValidation.hasAnyErrors()) {
            return Result.broken(loginsValidation);
        }

        var clientFeatureList = StreamEx.of(features).cross(logins)
                .mapKeyValue((feature, login) -> new ClientFeature()
                        .withId(featuresByTextId.get(feature).getId())
                        .withClientId(ClientId.fromLong(clientIdsByLogins.get(login))))
                .toList();
        clientFeaturesRepository.deleteClientsFeatures(clientFeatureList);
        clearCaches();

        var chiefLoginsByClientIds = userService
                .getChiefsLoginsByClientIds(mapList(logins, login -> ClientId.fromLong(clientIdsByLogins.get(login))));

        var loginToFeature = StreamEx.of(features).cross(logins)
                .mapKeyValue((feature, login) -> {
                    var clientId = clientIdsByLogins.get(login);
                    return new LoginClientIdChiefLoginWithFeature()
                            .withLogin(login)
                            .withClientId(clientId)
                            .withChiefLogin(chiefLoginsByClientIds.get(ClientId.fromLong(clientId)))
                            .withFeatureTextId(feature)
                            .withFeatureState(FeatureState.DISABLED);
                })
                .toList();
        return Result.successful(loginToFeature);
    }


    /**
     * переключает фичи для конкретных clientId, вычисляя его по login
     *
     * @param featureTextIdToLoginStateList список textId фич к логину
     * @return список(логины клиент, логин шефпредставителя, clientId)
     */
    public Result<List<LoginClientIdChiefLoginWithFeature>> switchFeaturesStateForClientIdsByLogins(
            List<FeatureTextIdToLoginState> featureTextIdToLoginStateList) {
        List<String> logins = mapList(featureTextIdToLoginStateList, FeatureTextIdToLoginState::getLogin);
        Map<String, Feature> featuresByTextId = listToMap(getCachedFeatures(), Feature::getFeatureTextId);
        Map<String, Long> clientIdsByLogins = shardHelper.getClientIdsByLogins(logins);

        List<String> obtainedFeatures = StreamEx.of(featureTextIdToLoginStateList)
                .map(FeatureTextIdToLoginState::getTextId)
                .distinct()
                .toList();

        ValidationResult<String, Defect> featuresValidation =
                switchFeatureToLoginValidationService.validateFeatureIds(obtainedFeatures, featuresByTextId.keySet());

        if (featuresValidation.hasAnyErrors()) {
            return Result.broken(featuresValidation);
        }

        ValidationResult<List<FeatureTextIdToLoginState>, Defect> validation = switchFeatureToLoginValidationService
                .validate(featureTextIdToLoginStateList, clientIdsByLogins);

        if (validation.hasAnyErrors()) {
            return Result.broken(validation);
        }

        List<ClientFeature> switchFeatureToClients =
                mapList(featureTextIdToLoginStateList, switchFeatureToLogin -> new ClientFeature()
                        .withState(switchFeatureToLogin.getState())
                        .withClientId(ClientId.fromLong(clientIdsByLogins.get(switchFeatureToLogin.getLogin())))
                        .withId(featuresByTextId.get(switchFeatureToLogin.getTextId()).getId()));

        clientFeaturesRepository.addClientsFeatures(switchFeatureToClients);
        featureService.clearCaches();

        Map<ClientId, String> chiefLoginsByClientIds =
                userService.getChiefsLoginsByClientIds(mapList(clientIdsByLogins.values(),
                        ClientId::fromLong));


        List<LoginClientIdChiefLoginWithFeature> loginToFeature =
                mapList(
                        featureTextIdToLoginStateList,
                        featureTextIdToLoginState -> new LoginClientIdChiefLoginWithFeature()
                                .withChiefLogin(
                                        chiefLoginsByClientIds.get(
                                                ClientId.fromLong(
                                                        clientIdsByLogins.get(featureTextIdToLoginState.getLogin())
                                                )
                                        )
                                )
                                .withClientId(clientIdsByLogins.get(featureTextIdToLoginState.getLogin()))
                                .withLogin(featureTextIdToLoginState.getLogin())
                                .withFeatureState(featureTextIdToLoginState.getState())
                                .withFeatureTextId(featureTextIdToLoginState.getTextId())
                );

        return Result.successful(loginToFeature, validation);
    }

    /**
     * включить одну фичу одному клиенту
     *
     * @param clientId    клиент
     * @param featureName фича
     * @throws FeatureAssignmentException если что-то пошло не так
     */
    public void enableFeatureForClient(ClientId clientId, FeatureName featureName) {
        enableFeatureForClient(clientId, featureName.getName());
    }

    /**
     * включить одну фичу одному клиенту
     *
     * @param clientId      клиент
     * @param featureTextId фича
     * @throws FeatureAssignmentException если что-то пошло не так
     */
    void enableFeatureForClient(ClientId clientId, String featureTextId) {
        changeFeatureStateForClient(clientId, featureTextId, FeatureState.ENABLED);
    }

    public void disableFeatureForClient(ClientId clientId, FeatureName featureName) {
        disableFeatureForClient(clientId, featureName.getName());
    }

    void disableFeatureForClient(ClientId clientId, String featureTextId) {
        changeFeatureStateForClient(clientId, featureTextId, FeatureState.DISABLED);
    }

    void changeFeatureStateForClient(ClientId clientId, String featureTextId, FeatureState featureState) {
        Result<List<LoginClientIdChiefLoginWithState>> result =
                switchFeaturesStateForClientIds(List.of(new FeatureTextIdToClientIdState()
                        .withTextId(featureTextId)
                        .withClientId(clientId)
                        .withState(featureState)));

        if (!result.isSuccessful()) {
            throw new FeatureAssignmentException("failed to enable feature: " + result.getErrors());
        }
    }

    /**
     * переключает фичи для конкретных clientId
     *
     * @param featureTextIdToClientIdStates список textId фич к clientId
     * @return список(логин шефпредставителя, clientId)
     */
    public Result<List<LoginClientIdChiefLoginWithState>> switchFeaturesStateForClientIds(
            List<FeatureTextIdToClientIdState> featureTextIdToClientIdStates) {
        return internalSwitchFeaturesStateForClientIds(null, featureTextIdToClientIdStates);
    }

    /**
     * Проверяет, может ли оператор переключить фичу и переключает фичи для конкретных clientId
     *
     * @param operatorUid                   - uid оператора
     * @param featureTextIdToClientIdStates - список textId фич к clientId
     * @return список(логин шефпредставителя, clientId)
     */
    public Result<List<LoginClientIdChiefLoginWithState>> switchFeaturesStateForClientIds(
            long operatorUid,
            List<FeatureTextIdToClientIdState> featureTextIdToClientIdStates) {
        return internalSwitchFeaturesStateForClientIds(operatorUid, featureTextIdToClientIdStates);
    }

    // если operatorUid != null - проверяем, что он имеер право переключать фичу
    private Result<List<LoginClientIdChiefLoginWithState>> internalSwitchFeaturesStateForClientIds(
            @Nullable Long operatorUid,
            List<FeatureTextIdToClientIdState> featureTextIdToClientIdStates
    ) {
        Map<String, Feature> featuresByTextId = listToMap(getCachedFeatures(), Feature::getFeatureTextId);

        List<String> obtainedFeatures = StreamEx.of(featureTextIdToClientIdStates)
                .map(FeatureTextIdToClientIdState::getTextId)
                .distinct()
                .toList();

        ValidationResult<String, Defect> featuresValidation =
                switchFeatureToLoginValidationService.validateFeatureIds(obtainedFeatures, featuresByTextId.keySet());

        if (featuresValidation.hasAnyErrors()) {
            return Result.broken(featuresValidation);
        }

        if (operatorUid != null) {
            ValidationResult<List<FeatureTextIdToClientIdState>, Defect> permissionsValidation =
                    switchFeatureToClientIdValidationService
                            .validateRolePermissions(featuresByTextId, rbacService.getUidRole(operatorUid),
                                    featureTextIdToClientIdStates);
            if (permissionsValidation.hasAnyErrors()) {
                return Result.broken(permissionsValidation);
            }
        }

        ValidationResult<List<FeatureTextIdToClientIdState>, Defect> validation =
                switchFeatureToClientIdValidationService
                        .validate(featureTextIdToClientIdStates);

        if (validation.hasAnyErrors()) {
            return Result.broken(validation);
        }

        Map<ClientId, List<FeatureTextIdToClientIdState>> clientIdToFeatures =
                StreamEx.of(featureTextIdToClientIdStates).groupingBy(FeatureTextIdToClientIdState::getClientId);

        List<ClientFeature> clientsFeaturesForInsert =
                mapList(featureTextIdToClientIdStates, switchFeatureToClient -> FeatureConverter
                        .clientFeatureFromFeatureTextIdToClientIdState(switchFeatureToClient, featuresByTextId));


        clientFeaturesRepository.addClientsFeatures(clientsFeaturesForInsert);
        featureService.clearCaches();

        List<Long> clientIds = StreamEx.of(StreamEx.of(featureTextIdToClientIdStates))
                .map(c -> c.getClientId().asLong()).toList();
        Map<ClientId, String> chiefLoginsByClientIds =
                userService.getChiefsLoginsByClientIds(mapList(clientIds, ClientId::fromLong));


        List<LoginClientIdChiefLoginWithState> loginsWithChiefLogins = EntryStream.of(chiefLoginsByClientIds)
                .filterValues(Objects::nonNull)
                .distinctKeys()
                .mapToValue((clientId, chiefLogin) -> new LoginClientIdChiefLoginWithState()
                        .withChiefLogin(chiefLogin)
                        .withFeatureState(clientIdToFeatures.get(clientId).get(0).getState())
                        .withClientId(clientId.asLong()))
                .values()
                .toList();


        return Result.successful(loginsWithChiefLogins, validation);
    }

    /**
     * Лимит на количество записей {@link ClientFeaturesRepository#}
     *
     * @return возвращает всех клиентов у которых включены/выключены фичи, для каждой фичи
     */
    public Result<Map<Long, List<ChiefRepresentativeWithClientFeature>>> getFeaturesClients(
            List<String> featureTextIds,
            FeatureState state) {
        Map<String, Feature> featuresByTextId = listToMap(getCachedFeatures(), Feature::getFeatureTextId);

        ValidationResult<List<String>, Defect> validation =
                featureTextIdValidationService.validate(featuresByTextId.keySet(), featureTextIds);
        if (validation.hasAnyErrors()) {
            return Result.broken(validation);
        }
        List<Long> featureIds = StreamEx.of(featureTextIds)
                .map(featuresByTextId::get)
                .map(Feature::getId)
                .toList();

        Map<Long, List<ClientFeature>> clientsWithFeaturesByFeatureId =
                clientFeaturesRepository.getClientsWithFeatures(featureIds, state);

        List<ClientId> clientIds = StreamEx.of(clientsWithFeaturesByFeatureId.values())
                .flatMap(StreamEx::of)
                .map(ClientFeature::getClientId)
                .toList();
        Map<ClientId, String> chiefLoginsByClientIds = userService.getChiefsLoginsByClientIds(clientIds);
        Map<ClientId, Long> chiefUidsByClientIds = rbacService.getChiefsByClientIds(clientIds);
        Map<Long, List<ChiefRepresentativeWithClientFeature>> clientsByFeatures =
                EntryStream.of(clientsWithFeaturesByFeatureId)
                        .mapValues(clientFeatureList -> mapList(clientFeatureList,
                                clientFeature -> new ChiefRepresentativeWithClientFeature()
                                        .withClientFeature(clientFeature)
                                        .withChiefUid(chiefUidsByClientIds
                                                .get(clientFeature.getClientId()))
                                        .withChiefLogin(chiefLoginsByClientIds
                                                .get(clientFeature.getClientId()))))
                        .toMap();
        return Result.successful(clientsByFeatures);
    }


    /**
     * включает фичу на роль
     *
     * @param featuresToRoles список связок фичи с ролью
     */
    private Result<List<Long>> enableFeaturesForRoles(List<FeatureTextIdToRole> featuresToRoles) {
        Map<String, Feature> featuresByTextId = listToMap(getCachedFeatures(), Feature::getFeatureTextId);

        ValidationResult<List<FeatureTextIdToRole>, Defect> validation =
                featureRoleUpdateValidationService.validate(featuresToRoles, featuresByTextId, true);
        if (validation.hasAnyErrors()) {
            return Result.broken(validation);
        }

        List<Long> featuresIds = StreamEx.of(featuresToRoles)
                .map(FeatureTextIdToRole::getTextId)
                .map(featuresByTextId::get)
                .map(Feature::getId)
                .toList();

        List<Feature> featuresToUpdate = featureRepository.get(featuresIds);

        Map<Long, List<String>> featurePercentMap = StreamEx.of(featuresToRoles)
                .mapToEntry(FeatureTextIdToRole::getTextId, FeatureTextIdToRole::getRoleName)
                .mapKeys(featuresByTextId::get)
                .mapKeys(Feature::getId)
                .collapseKeys()
                .toMap();

        StreamEx.of(featuresToUpdate)
                .filter(feature -> featurePercentMap.containsKey(feature.getId()))
                .forEach(feature -> feature.getSettings().getRoles().addAll(featurePercentMap.get(feature.getId())));

        List<ModelChanges<Feature>> modelChanges = StreamEx.of(featuresToUpdate)
                .map(feature -> new ModelChanges<>(feature.getId(), Feature.class)
                        .process(feature.getSettings(), Feature.SETTINGS))
                .toList();


        List<Long> validModelIds = mapList(modelChanges, ModelChanges::getId);
        Map<Long, Feature> featureRoles = StreamEx.of(featureRepository.get(validModelIds))
                .mapToEntry(Feature::getId)
                .invert()
                .toMap();

        List<AppliedChanges<Feature>> appliedChangesList = StreamEx.of(modelChanges)
                .map(x -> x.applyTo(featureRoles.get(x.getId())))
                .toList();

        featureRepository.update(appliedChangesList);
        clearCaches();
        return Result.successful(featuresIds, validation);
    }

    /**
     * выключает фичу на роль, удаляет записи в базе
     *
     * @param featuresToRoles список связок фичи с ролью
     */
    private Result<List<Long>> disableFeaturesForRoles(List<FeatureTextIdToRole> featuresToRoles) {
        Map<String, Feature> featuresByTextId = listToMap(getCachedFeatures(), Feature::getFeatureTextId);

        ValidationResult<List<FeatureTextIdToRole>, Defect> validation =
                featureRoleUpdateValidationService.validate(featuresToRoles, featuresByTextId, false);
        if (validation.hasAnyErrors()) {
            return Result.broken(validation);
        }

        List<Long> featuresIds = StreamEx.of(featuresToRoles)
                .map(FeatureTextIdToRole::getTextId)
                .map(featuresByTextId::get)
                .map(Feature::getId)
                .toList();

        List<Feature> featuresToUpdate = featureRepository.get(featuresIds);

        Map<Long, List<String>> featurePercentMap = StreamEx.of(featuresToRoles)
                .mapToEntry(FeatureTextIdToRole::getTextId, FeatureTextIdToRole::getRoleName)
                .mapKeys(featuresByTextId::get)
                .mapKeys(Feature::getId)
                .collapseKeys()
                .toMap();

        StreamEx.of(featuresToUpdate)
                .filter(feature -> featurePercentMap.containsKey(feature.getId()))
                .forEach(feature -> feature.getSettings().getRoles().removeAll(featurePercentMap.get(feature.getId())));

        List<ModelChanges<Feature>> modelChanges = StreamEx.of(featuresToUpdate)
                .map(feature -> new ModelChanges<>(feature.getId(), Feature.class)
                        .process(feature.getSettings(), Feature.SETTINGS))
                .toList();


        List<Long> validModelIds = mapList(modelChanges, ModelChanges::getId);
        Map<Long, Feature> featureRoles = StreamEx.of(featureRepository.get(validModelIds))
                .mapToEntry(Feature::getId)
                .invert()
                .toMap();

        List<AppliedChanges<Feature>> appliedChangesList = StreamEx.of(modelChanges)
                .map(x -> x.applyTo(featureRoles.get(x.getId())))
                .toList();
        featureRepository.update(appliedChangesList);
        clearCaches();

        return Result.successful(featuresIds, validation);
    }

    public Result<Long> setRolesAbleToChangeFeatureFromWeb(
            String featureName,
            Set<String> rolesAbleToEnable,
            Set<String> rolesAbleToDisable
    ) {
        ValidationResult<Void, Defect> requestValResult =
                validateUpdateWebEditability(rolesAbleToEnable, rolesAbleToDisable);
        if (requestValResult.hasAnyErrors()) {
            return Result.broken(requestValResult);
        }

        Map<String, Feature> featuresByTextId = listToMap(getCachedFeatures(), Feature::getFeatureTextId);
        Feature feature = featuresByTextId.get(featureName);
        if (feature == null) {
            return Result.broken(ValidationResult.failed(featureName, objectNotFound()));
        }
        Long featureId = feature.getId();

        // Get copy of `Settings` before modifying
        Feature featuresToUpdate = first(featureRepository.get(List.of(featureId)));
        FeatureSettings settings = featuresToUpdate.getSettings()
                .withCanEnable(rolesAbleToEnable)
                .withCanDisable(rolesAbleToDisable);

        ModelChanges<Feature> modelChanges =
                new ModelChanges<>(featureId, Feature.class)
                        .process(settings, Feature.SETTINGS);

        // Another model object for using in ModelChanges#applyTo
        Feature dbFeature = first(featureRepository.get(List.of(featureId)));
        AppliedChanges<Feature> appliedChanges = modelChanges.applyTo(dbFeature);
        featureRepository.update(List.of(appliedChanges));
        clearCaches();
        return Result.successful(featureId);
    }

    /**
     * Тривиальная валидация для списка ролей.
     * До самостоятельного валидатора пока не доросла
     */
    @SuppressWarnings("rawtypes")
    private static ValidationResult<Void, Defect> validateUpdateWebEditability(
            Set<String> rolesAbleToEnable,
            Set<String> rolesAbleToDisable
    ) {
        ItemValidationBuilder<Void, Defect> ivb = ItemValidationBuilder.of(null);
        Set<String> existingRoles = StreamEx.of(RbacRole.values()).map(Enum::name).toSet();
        ivb.item(rolesAbleToEnable, "rolesAbleToEnable")
                .check(CommonConstraints.eachInSet(existingRoles));
        ivb.item(rolesAbleToDisable, "rolesAbleToDisable")
                .check(CommonConstraints.eachInSet(existingRoles));
        return ivb.getResult();
    }

    /**
     * Выключает фичу всем, у кого она на момент запроса включена по ClientId
     *
     * @param featureName имя фичи
     */
    public void switchFeatureOffForIdentifiedUsers(String featureName) {
        Objects.requireNonNull(featureName);

        Map<String, Feature> featuresByTextId = listToMap(getCachedFeatures(), Feature::getFeatureTextId);
        Long featureId = featuresByTextId.get(featureName).getId();

        List<ClientFeature> features = clientFeaturesRepository.getClientsWithFeatures(List.of(featureId),
                FeatureState.ENABLED).get(featureId);

        List<ClientFeature> clientsFeaturesForInsert = features.stream()
                .map(cf -> new ClientFeature()
                        .withClientId(cf.getClientId())
                        .withId(cf.getId())
                        .withState(FeatureState.DISABLED))
                .collect(Collectors.toList());
        clientFeaturesRepository.addClientsFeatures(clientsFeaturesForInsert);
        featureService.clearCaches();
    }

    /**
     * Информация о кол-ве включенных/выключенных фич для клиентов (clients_features)
     */
    public Map<Long, Map<Boolean, Integer>> getFeaturesStateSummary() {
        Map<Long, Map<Boolean, Integer>> result = new HashMap<>();
        shardHelper.forEachShard(shard -> {
            Map<Long, Map<Boolean, Integer>> shardResult = clientFeaturesRepository.featuresStateSummary(shard);
            result.putAll(
                    EntryStream.of(shardResult)
                            .append(EntryStream.of(result))
                            .toMap((m1, m2) -> EntryStream.of(m1)
                                    .append(EntryStream.of(m2))
                                    .toMap(Integer::sum)));
        });
        return result;
    }

    public Result<Long> updateFeaturePublicity(String featureTextId, boolean isPublic) {
        Map<String, Feature> existingFeaturesByTextId = listToMap(featureRepository.get(), Feature::getFeatureTextId);
        ValidationResult<String, Defect> vr =
                featureTextIdValidationService.validate(existingFeaturesByTextId.keySet(), featureTextId);
        if (vr.hasAnyErrors()) {
            return Result.broken(vr);
        }
        var preparedFeature = existingFeaturesByTextId.get(featureTextId);
        preparedFeature.getSettings().setIsPublic(isPublic);
        var modelChanges = new ModelChanges<>(preparedFeature.getId(), Feature.class);
        modelChanges.process(preparedFeature.getSettings(), Feature.SETTINGS);
        var featureId = modelChanges.getId();
        var featureForUpdate = featureRepository.get(List.of(featureId)).get(0);
        var appliedChanges = modelChanges.applyTo(featureForUpdate);
        featureRepository.update(List.of(appliedChanges));
        clearCaches();
        return Result.successful(featureId);
    }

    public void clearCaches() {
        featureCache.invalidate();
        featureService.clearCaches();
    }
}
