package ru.yandex.direct.core.entity.feature.service;

import java.math.BigInteger;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.Iterables;
import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.context.event.EventListener;
import org.springframework.stereotype.Service;
import org.springframework.web.context.support.ServletRequestHandledEvent;

import ru.yandex.direct.core.entity.abt.container.AllowedFeatures;
import ru.yandex.direct.core.entity.abt.container.UaasInfoRequest;
import ru.yandex.direct.core.entity.abt.container.UaasInfoResponse;
import ru.yandex.direct.core.entity.abt.service.UaasInfoService;
import ru.yandex.direct.core.entity.client.model.AgencyClientRelation;
import ru.yandex.direct.core.entity.client.service.AgencyClientRelationService;
import ru.yandex.direct.core.entity.client.service.ClientService;
import ru.yandex.direct.core.entity.feature.container.ClientRealFeature;
import ru.yandex.direct.core.entity.feature.container.FeatureRequest;
import ru.yandex.direct.core.entity.feature.container.FeatureRequestFactory;
import ru.yandex.direct.core.entity.feature.container.FeaturesWithExpBoxes;
import ru.yandex.direct.core.entity.feature.model.ClientFeature;
import ru.yandex.direct.core.entity.feature.model.Feature;
import ru.yandex.direct.core.entity.feature.model.FeatureConverter;
import ru.yandex.direct.core.entity.feature.model.FeatureState;
import ru.yandex.direct.core.entity.feature.repository.ClientFeaturesRepository;
import ru.yandex.direct.core.entity.feature.service.FeatureUtils.TLCache;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.env.EnvironmentType;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.rbac.RbacRole;

import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static java.util.function.Function.identity;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;
import static java.util.stream.Collectors.toSet;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.utils.HashingUtils.getMd5HashUtf8;

@Service
@ParametersAreNonnullByDefault
@Lazy(false)
public class FeatureService {
    private static final BigInteger PERCENT = BigInteger.valueOf(100);
    private static final Logger logger = LoggerFactory.getLogger(FeatureService.class);
    private static final int CHUNK_SIZE = 1000;

    private final FeatureCache featureCache;
    private final ClientFeaturesRepository clientFeaturesRepository;

    private final ClientService clientService;
    private final ShardHelper shardHelper;
    private final AgencyClientRelationService agencyClientRelationService;
    private final UaasInfoService uaasInfoService;
    private final EnvironmentType environmentType;

    private final TLCache<FeatureRequest, Set<String>> featuresByClientIdCache;
    private final TLCache<FeatureRequest, Set<String>> publicFeaturesByClientIdCache;
    private final TLCache<FeatureRequest, Set<String>> featuresByUidCache;
    private final FeatureRequestFactory featureRequestFactory;

    @Autowired
    public FeatureService(FeatureCache featureCache,
                          ClientFeaturesRepository clientFeaturesRepository,
                          ClientService clientService,
                          ShardHelper shardHelper,
                          AgencyClientRelationService agencyClientRelationService,
                          UaasInfoService uaasInfoService,
                          EnvironmentType environmentType,
                          FeatureRequestFactory featureRequestFactory
    ) {
        this.featureCache = featureCache;
        this.clientFeaturesRepository = clientFeaturesRepository;
        this.clientService = clientService;
        this.shardHelper = shardHelper;
        this.agencyClientRelationService = agencyClientRelationService;
        this.uaasInfoService = uaasInfoService;
        this.environmentType = environmentType;
        this.featureRequestFactory = featureRequestFactory;

        featuresByClientIdCache = TLCache.<FeatureRequest, Set<String>>builder()
                .setLoadFunc(this::loadFeaturesForClientId)
                .setLoadAllFunc(this::loadFeaturesForClientIds)
                .setTimeout(5)
                .build();

        publicFeaturesByClientIdCache = TLCache.<FeatureRequest, Set<String>>builder()
                .setLoadFunc(this::loadPublicFeaturesForClientId)
                .setTimeout(5)
                .build();

        featuresByUidCache = TLCache.<FeatureRequest, Set<String>>builder()
                .setLoadFunc(this::loadFeaturesForUid)
                .setLoadAllFunc(this::loadFeaturesForUids)
                .setTimeout(5)
                .build();
    }

    /**
     * Возвращает список доступных фич по uid
     *
     * @param uid uid оператора
     * @return {@link Set<String>} список доступных фич для оператора
     */
    public Set<String> getEnabledForUid(Long uid) {
        return getEnabledForUid(featureRequestFactory.get(uid));
    }

    public Map<ClientId, Set<String>> getEnabled(Set<ClientId> clientIds) {
        Map<ClientId, Set<String>> result = new HashMap<>();
        for (var chunk : Iterables.partition(clientIds, CHUNK_SIZE)) {
            result.putAll(getEnabledForClientIds(mapList(chunk, featureRequestFactory::get)));
        }
        return result;
    }

    /**
     * Включена ли хотя бы одна из фич для переданного clientId
     */
    public boolean anyEnabled(ClientId clientId, Set<FeatureName> featureNames) {
        Set<String> enabledFeatureNames = getEnabledForClientId(clientId);

        return StreamEx.of(featureNames)
                .anyMatch(requiredFeatureName -> enabledFeatureNames.contains(requiredFeatureName.getName()));
    }

    /**
     * Включена ли фича с featureName для переданного uid
     */
    public boolean isEnabled(Long uid, FeatureName featureName) {
        return getEnabledForUid(uid).contains(featureName.getName());
    }

    /**
     * Возвращает список доступных и публичных фич для клиента
     *
     * @param clientId id клиента
     * @return {@link Set<String>} список доступных и публичных фич для клиента
     */
    public Set<String> getPublicForClientId(ClientId clientId) {
        return getPublicForClientId(featureRequestFactory.get(clientId));
    }

    /**
     * Включена ли фича с featureName для переданного clientId
     */
    public boolean isEnabledForClientId(ClientId clientId, FeatureName featureName) {
        return isEnabledForClientId(featureRequestFactory.get(clientId), featureName.getName());
    }

    /**
     * Включена ли фича с featureName для переданного clientId
     */
    public boolean isEnabledForClientId(FeatureRequest featureRequest, String featureName) {
        return isEnabledForFeatureRequests(List.of(featureRequest), featureName).get(featureRequest.getClientId());
    }

    private Map<ClientId, Boolean> isEnabledForFeatureRequests(List<FeatureRequest> featureRequests,
                                                               String featureName) {
        if (featureRequests.isEmpty()) {
            return emptyMap();
        }
        var featuresByClientIds = getEnabledForClientIds(featureRequests);
        var clientIds = featureRequests.stream().map(FeatureRequest::getClientId).collect(toSet());
        return StreamEx.of(clientIds)
                .toMap(clientId -> Optional.ofNullable(featuresByClientIds.get(clientId))
                        .map(features -> features.contains(featureName))
                        .orElse(false));
    }

    /**
     * Проверяет фичу не только в базе, но и в uaas
     * Если список клиектов большой, а фича включается в базе - стоит использовать
     * {@link #isEnabledForClientIdsOnlyFromDb(Set, String)}
     */
    public Map<ClientId, Boolean> isEnabledForClientIds(Set<ClientId> clientIds, FeatureName featureName) {
        return isEnabledForClientIds(clientIds, featureName.getName());
    }

    public Map<ClientId, Boolean> isEnabledForClientIds(Set<ClientId> clientIds, String featureName) {
        Map<ClientId, Boolean> result = new HashMap<>();
        for (var chunk : Iterables.partition(clientIds, CHUNK_SIZE)) {
            List<FeatureRequest> featureRequests =
                    StreamEx.of(chunk).map(featureRequestFactory::get).collect(toList());
            result.putAll(isEnabledForFeatureRequests(featureRequests, featureName));
        }
        return result;
    }

    public Map<ClientId, Boolean> isEnabledForClientIdsOnlyFromDb(Set<ClientId> clientIds, String featureName) {
        Map<ClientId, Boolean> result = new HashMap<>();
        for (var chunk : Iterables.partition(clientIds, CHUNK_SIZE)) {
            List<FeatureRequest> featureRequests =
                    StreamEx.of(chunk).map(clientId ->
                            new FeatureRequest().withClientId(clientId).withOnlyFromDb(true)).collect(toList());
            result.putAll(isEnabledForFeatureRequests(featureRequests, featureName));
        }
        return result;
    }

    /**
     * Включены ли все фичи с featureNames для переданного uid
     */
    public boolean isEnabledForUid(Long uid, Collection<FeatureName> featureNames) {
        return isEnabledForUid(featureRequestFactory.get(uid), featureNames);
    }

    /**
     * Включены ли все фичи с featureNames для переданного uid
     */
    public boolean isEnabledForUid(FeatureRequest featureRequest, Collection<FeatureName> featureNames) {
        return getEnabledForUid(featureRequest).containsAll(mapList(featureNames, FeatureName::getName));
    }

    /**
     * Возвращает все включенные фичи для переданных clientIds
     */
    private Map<ClientId, Set<String>> getEnabledForClientIds(Collection<FeatureRequest> featureRequests) {
        try {
            return EntryStream.of(featuresByClientIdCache.getAll(featureRequests))
                    .mapKeys(FeatureRequest::getClientId)
                    .toMap();
        } catch (ExecutionException e) {
            throw new IllegalStateException("Failed on getting all features for client ids", e);
        }
    }

    /**
     * Возвращает все включенные фичи для переданного clientId
     */
    public Set<String> getEnabledForClientId(FeatureRequest featureRequest) {
        return featuresByClientIdCache.get(featureRequest);
    }

    /**
     * Возвращает все включенные фичи для переданного clientId
     */
    public Set<String> getEnabledForClientId(ClientId clientId) {
        return getEnabledForClientId(featureRequestFactory.get(clientId));
    }

    public Set<String> getEnabledForUid(FeatureRequest featureRequest) {
        return featuresByUidCache.get(featureRequest);
    }

    /**
     * Возвращает все включенные фичи для переданных uids
     */
    public Map<Long, Set<String>> getEnabledForUids(List<Long> uids) {
        Map<Long, Set<String>> result = new HashMap<>();
        for (var chunk : Iterables.partition(uids, CHUNK_SIZE)) {
            result.putAll(getEnabledForUidsFromFeatureRequest(mapList(chunk, featureRequestFactory::get)));
        }
        return result;
    }

    /**
     * Возвращает все включенные фичи для переданных uids
     */
    private Map<Long, Set<String>> getEnabledForUidsFromFeatureRequest(List<FeatureRequest> featureRequests) {
        try {
            return EntryStream.of(featuresByUidCache.getAll(featureRequests))
                    .mapKeys(FeatureRequest::getUid)
                    .toMap();
        } catch (ExecutionException e) {
            throw new IllegalStateException("Failed on getting all features for uids", e);
        }
    }

    /**
     * Возвращает все включенные и публичные фичи для переданного featureRequest
     */
    public Set<String> getPublicForClientId(FeatureRequest featureRequest) {
        return publicFeaturesByClientIdCache.get(featureRequest);
    }

    /**
     * Возвращает все явно включенные и явно выключенные фичи для переданного clientId
     */
    public List<ClientRealFeature> getAllKnownFeaturesForClient(FeatureRequest featureRequest) {
        return getClientFeatureStorage(List.of(featureRequest), featureCache.getCached())
                .getKnownFeatures();
    }

    /**
     * По заданному clientId и списку featureTextIds возвращает статусы этих фич.
     *
     * @param clientId       идентификатор клиента
     * @param featureTextIds список текстовых идентификаторов фич
     */
    public Map<String, FeatureState> getFeatureStates(ClientId clientId, Collection<String> featureTextIds) {
        Map<String, Long> existingFeatures = featureCache.getIdsByTextId(featureTextIds);
        Set<String> enabledFeatures = getEnabledForClientId(clientId);

        Map<String, FeatureState> clientFeatureStatesByTextId = new LinkedHashMap<>();
        featureTextIds.forEach(textId -> {
            FeatureState state = getState(textId, existingFeatures, enabledFeatures);
            clientFeatureStatesByTextId.put(textId, state);
        });
        return clientFeatureStatesByTextId;
    }

    /**
     * @param textId           текстовый идентификатор фичи
     * @param existingFeatures отображение между текстовым и числовым идентификатором фичи
     * @param enabledFeatures  список включенных фичей
     * @return статус фичи или {@link FeatureState#UNKNOWN}, если такой фичи не существует
     * или {@link FeatureState#DISABLED} если ее нет у клиента.
     */
    private FeatureState getState(String textId,
                                  Map<String, Long> existingFeatures,
                                  Set<String> enabledFeatures) {
        if (existingFeatures.get(textId) == null) {
            return FeatureState.UNKNOWN;
        }
        if (enabledFeatures.contains(textId)) {
            return FeatureState.ENABLED;
        }
        return FeatureState.DISABLED;
    }

    /**
     * Пытаемся найти clientId по operatorUid и вычисляем доступ по clientId
     */
    private Map<Long, List<String>> calculateFeaturesStateForUids(List<FeatureRequest> featureRequests,
                                                                  List<Feature> features) {
        Map<Long, FeatureRequest> operatorUidsToRequest =
                featureRequests.stream().collect(toMap(FeatureRequest::getUid, featureRequest -> featureRequest, (u1,
                                                                                                                  u2) -> u1));
        Set<Long> operatorUids = operatorUidsToRequest.keySet();
        Map<Long, Long> uidToClientId = shardHelper.getClientIdsByUids(operatorUids);
        Map<Long, List<Long>> clientIdToUids = EntryStream.of(uidToClientId)
                .invert()
                .grouping();

        var clientIdFeatureRequests =
                featureRequests.stream()
                        .filter(featureRequest -> uidToClientId.containsKey(featureRequest.getUid()))
                        .map(featureRequest -> {
                            var clientId = uidToClientId.get(featureRequest.getUid());
                            return featureRequestFactory.get(ClientId.fromLong(clientId));
                        })
                        .collect(toList());
        //если возможно получаем clientId
        Map<Long, List<String>> featuresForClientsEnabled = EntryStream.of(
                        calculateFeaturesStateForClientIds(clientIdFeatureRequests, features))
                .mapKeys(c -> clientIdToUids.get(c.asLong()))
                .flatMapKeys(Collection::stream)
                .toMap();
        List<Long> uidsWithoutClients = StreamEx.of(operatorUids)
                .nonNull()
                .remove(uidToClientId::containsKey)
                .toList();
        Map<Long, List<String>> featuresForUidsEnabled = new HashMap<>();

        for (Long uid : uidsWithoutClients) {
            List<String> enabledFeatureTextIds = StreamEx.of(features)
                    .mapToEntry(feature -> isFeatureEnabledByPercent(feature, uid))
                    //отдаем, только включенные фичи
                    .filterValues(featureState -> FeatureState.ENABLED == featureState)
                    .keys()
                    .map(Feature::getFeatureTextId)
                    .toList();
            if (!enabledFeatureTextIds.isEmpty()) {
                featuresForUidsEnabled.put(uid, enabledFeatureTextIds);
            }
        }
        featuresForUidsEnabled.putAll(featuresForClientsEnabled);
        return featuresForUidsEnabled;
    }

    /**
     * Вычисляем фичи включенные для clientsIds
     */
    Map<ClientId, List<String>> calculateFeaturesStateForClientIds(Collection<FeatureRequest> featureRequests,
                                                                   List<Feature> features) {
        if (features.isEmpty() || featureRequests.isEmpty()) {
            return Collections.emptyMap();
        }

        //отдаем только включенные фичи
        return StreamEx.of(getClientFeatureStorage(featureRequests, features).getEnabledFeatures())
                .mapToEntry(ClientRealFeature::getClientId,
                        clientRealFeature -> clientRealFeature.getFeature().getFeatureTextId())
                .grouping();
    }

    private ClientFeaturesStorage getClientFeatureStorage(Collection<FeatureRequest> featureRequests,
                                                          List<Feature> features) {
        ClientFeaturesStorage clientFeaturesStorage = new ClientFeaturesStorage(featureRequests, features);

        //фичи из отладочной куки
        addNpFixedFeatures(featureRequests, clientFeaturesStorage);

        //считываем из базы фичи явно включенные или выключенные для клиента
        addManuallyExposedClientsFeatures(clientFeaturesStorage);

        //проверяем роли
        addFeaturesIsEnabledForClientRole(clientFeaturesStorage);

        // фичи из uaas
        if (!environmentType.isSandbox()) { // в песочнице не ходим в AB-шницу
            addUaasFeatures(featureRequests, clientFeaturesStorage);
        }

        //вычисляем фичи включенные для агенств
        addFeaturesIsEnabledForAgency(clientFeaturesStorage);

        //фичи, разыгранные
        addAndSaveFeaturesIsEnabledByPercent(clientFeaturesStorage);

        return clientFeaturesStorage;
    }

    private void addUaasFeatures(Collection<FeatureRequest> featureRequests,
                                 ClientFeaturesStorage clientFeaturesStorage) {
        var possibleFeaturesTextIdsSet =
                featureCache.getCached().stream().map(Feature::getFeatureTextId).collect(toSet());
        var uaasInfoRequests = featureRequests.stream()
                .filter(featureRequest -> !featureRequest.isOnlyFromDb())
                .map(featureRequest -> new UaasInfoRequest()
                        .withClientId(featureRequest.getClientId().asLong())
                        .withHost(featureRequest.getHost())
                        .withUserAgent(featureRequest.getUserAgent())
                        .withIp(featureRequest.getIp())
                        .withYandexUid(featureRequest.getYandexuid())
                        .withYexpCookie(featureRequest.getYexpCookie())
                        .withInterfaceLang(featureRequest.getInterfaceLang())
                        .withEnabledFeatures(getManuallyEnabledFeatures(clientFeaturesStorage,
                                featureRequest.getClientId().asLong())))
                .collect(toList());
        if (!uaasInfoRequests.isEmpty()) {
            uaasInfoService.getInfo(uaasInfoRequests, AllowedFeatures.of(possibleFeaturesTextIdsSet))
                    .forEach(uaasInfoResponse -> addUaasFeaturesToStorage(uaasInfoResponse, clientFeaturesStorage));
        }
    }

    private Set<String> getManuallyEnabledFeatures(ClientFeaturesStorage clientFeaturesStorage, Long clientId) {
        return StreamEx.of(clientFeaturesStorage.getEnabled(ClientId.fromLong(clientId)).getFeatures())
                .map(clientRealFeature -> clientRealFeature.getFeature().getFeatureTextId())
                .toSet();
    }

    private void addUaasFeaturesToStorage(UaasInfoResponse uaasInfoResponse,
                                          ClientFeaturesStorage clientFeaturesStorage) {
        Map<String, ClientRealFeature> featuresWithUnknownStatesForClient =
                clientFeaturesStorage.getUnknown(uaasInfoResponse.getClientId())
                        .getFeatures().stream()
                        .filter(clientRealFeature -> clientRealFeature.getClientId().equals(uaasInfoResponse.getClientId()))
                        .collect(toMap(clientRealFeature -> clientRealFeature.getFeature().getFeatureTextId(),
                                clientRealFeature -> clientRealFeature));
        var featuresForUpdate =
                uaasInfoResponse.getFeatures().stream().filter(featuresWithUnknownStatesForClient::containsKey)
                        .map(featuresWithUnknownStatesForClient::get)
                        .map(clientRealFeature -> new ClientFeature().withState(FeatureState.ENABLED).withClientId(uaasInfoResponse.getClientId()).withId(clientRealFeature.getFeature().getId()))
                        .collect(toList());
        clientFeaturesStorage.update(featuresForUpdate);
        clientFeaturesStorage.setExpBoxes(uaasInfoResponse.getClientId(), uaasInfoResponse.getBoxes(),
                uaasInfoResponse.getBoxesCrypted());
    }

    public FeaturesWithExpBoxes getFeaturesWithExpBoxes(FeatureRequest featureRequest) {
        var storage = getClientFeatureStorage(List.of(featureRequest), featureCache.getCached());
        return storage.getEnabled(featureRequest.getClientId());
    }

    public FeaturesWithExpBoxes getFeaturesWithExpBoxes(FeatureRequest featureRequest, Set<String> wantedFeatures) {
        var storage = getClientFeatureStorage(List.of(featureRequest), featureCache.getCachedByTextId(wantedFeatures));
        return storage.getEnabled(featureRequest.getClientId());
    }

    /**
     * Вычисляем фичи включенные для ролей клиентов
     */
    private void addFeaturesIsEnabledForClientRole(ClientFeaturesStorage clientFeaturesStorage) {
        List<ClientRealFeature> clientFeatureList = clientFeaturesStorage.getUnknownFeatures();
        Set<ClientId> clientIds = listToSet(clientFeatureList, ClientRealFeature::getClientId);
        Map<ClientId, RbacRole> clientIdsRoles = clientService.massGetRolesByClientId(clientIds);

        for (ClientRealFeature clientFeature : clientFeatureList) {
            ClientId clientId = clientFeature.getClientId();
            String roleName = Optional.ofNullable(clientIdsRoles.get(clientId)).orElse(RbacRole.EMPTY).name();
            Feature feature = clientFeature.getFeature();

            FeatureState featureState = isFeatureEnabledForRole(feature, roleName);
            clientFeaturesStorage.update(List.of(new ClientFeature().withClientId(clientId).withId(feature.getId()).withState(featureState)));
        }
    }

    /**
     * Вычисляем фичи включенные для агенств клиентов
     */
    private void addFeaturesIsEnabledForAgency(ClientFeaturesStorage clientFeaturesStorage) {
        List<ClientRealFeature> agencyRealFeatureList = clientFeaturesStorage.getUnknownFeatures()
                .stream().filter(FeatureService::isAgencyFeature).collect(Collectors.toList());

        if (agencyRealFeatureList.isEmpty()) {
            return;
        }

        Set<ClientId> clientIds = listToSet(agencyRealFeatureList, ClientRealFeature::getClientId);

        Collection<AgencyClientRelation> agencyClientRelations =
                agencyClientRelationService.massGetByClients(clientIds);
        Map<ClientId, List<AgencyClientRelation>> clientToAgencyRelationList =
                StreamEx.of(agencyClientRelations).groupingBy(AgencyClientRelation::getClientClientId);

        List<ClientFeature> agencyFeatureList =
                agencyRealFeatureList.stream().flatMap(c -> getClientFeature(clientToAgencyRelationList, c).stream())
                        .collect(Collectors.toList());

        List<ClientFeature> agencyListWithStatuses =
                clientFeaturesRepository.getClientsFeaturesStatus(agencyFeatureList);
        Map<ClientId, List<ClientFeature>> agencyIdToFeature =
                StreamEx.of(agencyListWithStatuses).groupingBy(ClientFeature::getClientId);

        List<ClientFeature> clientFeatureList = new ArrayList<>();
        for (Map.Entry<ClientId, List<AgencyClientRelation>> entry : clientToAgencyRelationList.entrySet()) {
            List<AgencyClientRelation> agencyClientRelationList = entry.getValue();
            if (agencyClientRelationList == null || agencyClientRelationList.isEmpty()) {
                continue;
            }
            ClientId clientId = entry.getKey();
            for (AgencyClientRelation agencyClientRelation : agencyClientRelationList) {
                List<ClientFeature> features = agencyIdToFeature.get(agencyClientRelation.getAgencyClientId());
                if (features == null) {
                    continue;
                }
                for (ClientFeature feature : features) {
                    ClientFeature clientFeature = new ClientFeature();
                    clientFeature.setClientId(clientId);
                    clientFeature.setState(feature.getState());
                    clientFeature.setId(feature.getId());
                    clientFeatureList.add(clientFeature);
                }
            }
        }
        clientFeaturesStorage.update(clientFeatureList);
    }

    private static boolean isAgencyFeature(ClientRealFeature c) {
        return Boolean.TRUE.equals(c.getFeature().getSettings().getIsAgencyFeature());
    }

    private static List<ClientFeature> getClientFeature(Map<ClientId, List<AgencyClientRelation>> clientToAgency,
                                                        ClientRealFeature clientRealFeature) {
        List<AgencyClientRelation> agencyList = clientToAgency.get(clientRealFeature.getClientId());
        if (agencyList == null) {
            return emptyList();
        }
        List<ClientFeature> result = new ArrayList<>();

        for (AgencyClientRelation agency : agencyList) {
            ClientId agencyId = agency.getAgencyClientId();
            result.add(new ClientFeature()
                    .withState(clientRealFeature.getFeatureState())
                    .withClientId(agencyId)
                    .withId(clientRealFeature.getFeature().getId()));
        }
        return result;
    }

    /**
     * Вычисляем фичи, включенные или выключенные через куку np_fixed_features
     */
    private void addNpFixedFeatures(Collection<FeatureRequest> featureRequests,
                                    ClientFeaturesStorage clientFeaturesStorage) {
        if (environmentType == EnvironmentType.PRODUCTION) {
            return;
        }
        List<ClientRealFeature> clientFeatureList = clientFeaturesStorage.getUnknownFeatures();
        for (FeatureRequest featureRequest : featureRequests) {
            Set<String> plusFeatures = new HashSet<>();
            Set<String> minusFeatures = new HashSet<>();
            var npFixedFeatures = featureRequest.getNpFixedFeatures();
            if (npFixedFeatures == null) {
                continue;
            }
            // Изначально куки разделялись запятой, но это вызвало проблемы с обрезанием куки до первой запятой
            // Решили заменить на точку, сейчас поддержано оба варианта, чтобы тесты, использующие запятую, не падали
            // see https://st.yandex-team.ru/DIRECT-141897
            var delimiter = npFixedFeatures.contains(".") ? "\\." : ",";
            String[] parts = npFixedFeatures.split(delimiter);
            for (String p : parts) {
                if (p.startsWith("-")) {
                    minusFeatures.add(p.substring(1));
                } else {
                    plusFeatures.add(p);
                }
            }
            if (plusFeatures.isEmpty() && minusFeatures.isEmpty()) {
                continue;
            }
            for (ClientRealFeature clientFeature : clientFeatureList) {
                ClientId clientId = clientFeature.getClientId();
                Feature feature = clientFeature.getFeature();

                FeatureState featureState = null;
                if (plusFeatures.contains(feature.getFeatureTextId())) {
                    featureState = FeatureState.ENABLED;
                }
                if (minusFeatures.contains((feature.getFeatureTextId()))) {
                    featureState = FeatureState.DISABLED;
                }

                if (featureState != null) {
                    clientFeaturesStorage.update(List.of(new ClientFeature().withClientId(clientId).withState(featureState).withId(feature.getId())));
                }
            }
        }
    }

    /**
     * Вычисляем явно включенные или выключенные для клиентов фичи в базе
     */
    private void addManuallyExposedClientsFeatures(ClientFeaturesStorage clientFeaturesStorage) {
        List<ClientFeature> unknown =
                mapList(clientFeaturesStorage.getUnknownFeatures(), FeatureConverter::toClientFeature);
        var clientFeatures = clientFeaturesRepository.getClientsFeaturesStatus(unknown, false);
        // в clients_features могут быть id фичей, которые удалены из ppcdict, пропускаем при обновлении
        // также могут попасть фичи с уже определённым в addNpFixedFeatures состоянием, их тоже пропускаем
        clientFeaturesStorage.updateIfPresentAndUnknown(clientFeatures);
    }

    /**
     * Разыгрываем и записываем статусы фич для клиентов
     */
    private void addAndSaveFeaturesIsEnabledByPercent(ClientFeaturesStorage clientFeaturesStorage) {
        List<ClientRealFeature> clientFeatureList = clientFeaturesStorage.getUnknownFeatures();
        Set<Long> knownFeaturesBeforePercent =
                clientFeaturesStorage.getKnownFeatures().stream().map(ClientRealFeature::getFeature).map(Feature::getId).collect(toSet());
        for (ClientRealFeature clientFeature : clientFeatureList) {
            ClientId clientId = clientFeature.getClientId();
            Feature feature = clientFeature.getFeature();

            FeatureState featureState = isFeatureEnabledByPercent(feature, clientId.asLong());
            clientFeaturesStorage.update(List.of(new ClientFeature().withClientId(clientId).withState(featureState).withId(feature.getId())));
        }
        List<ClientRealFeature> knownFeaturesByPercent = clientFeaturesStorage.getKnownFeatures()
                .stream()
                .filter(clientRealFeature -> !knownFeaturesBeforePercent.contains(clientRealFeature.getFeature().getId()))
                .collect(Collectors.toUnmodifiableList());
        saveClientsFeaturesState(knownFeaturesByPercent);
    }

    /**
     * Записываем результат розыгрыша фич для клиентов, если включена getIsAccessibleAfterDisabling в настройках фичи
     */
    private void saveClientsFeaturesState(List<ClientRealFeature> clientRealFeatureList) {
        Predicate<ClientRealFeature> featureIsAccessibleAfterDisabling =
                clientRealFeature -> Optional
                        .ofNullable(clientRealFeature.getFeature().getSettings().getIsAccessibleAfterDisabling())
                        .orElse(false);

        List<ClientFeature> clientFeaturesForWriting = StreamEx.of(clientRealFeatureList)
                .filter(featureIsAccessibleAfterDisabling)
                .map(FeatureConverter::toClientFeature)
                .toList();

        clientFeaturesRepository.addClientsFeatures(clientFeaturesForWriting);
    }

    @VisibleForTesting
    static FeatureState isFeatureEnabledByPercent(Feature feature, Long id) {
        Integer percent = feature.getSettings().getPercent();
        if (percent == null) {
            return FeatureState.UNKNOWN;
        } else if (percent <= 0) {
            return FeatureState.DISABLED;
        } else if (percent >= 100) {
            return FeatureState.ENABLED;
        } else {
            BigInteger hash = getMd5HashUtf8(String.valueOf(id + feature.getId()));
            return hash.mod(PERCENT).intValue() < percent ? FeatureState.ENABLED
                    : FeatureState.DISABLED;
        }
    }

    private static FeatureState isFeatureEnabledForRole(Feature feature, String roleName) {
        boolean contains =
                feature.getSettings().getRoles() != null && feature.getSettings().getRoles().contains(roleName);
        return contains ? FeatureState.ENABLED : FeatureState.UNKNOWN;
    }

    public boolean isFeatureExist(FeatureName featureName) {
        return featureCache.getCached().stream()
                .filter(f -> f.getFeatureTextId().equals(featureName.getName()))
                .anyMatch(l -> true);
    }

    public List<String> getAllExistingFeatureTextIds() {
        return mapList(featureCache.getCached(), Feature::getFeatureTextId);
    }

    public void clearCaches() {
        featuresByClientIdCache.clear();
        publicFeaturesByClientIdCache.clear();
        featuresByUidCache.clear();
    }

    @EventListener
    public void onServletRequestHandled(ServletRequestHandledEvent event) {
        logger.trace("clear caches on request handled event");
        clearCaches();
    }

    private Set<String> loadFeaturesForClientId(FeatureRequest key) {
        return calculateFeaturesStateForClientIds(List.of(key), featureCache.getCached()).values()
                .stream()
                .flatMap(Collection::stream)
                .collect(toSet());
    }

    private Map<FeatureRequest, Set<String>> loadFeaturesForClientIds(Collection<FeatureRequest> keys) {
        var featureRequests = StreamEx.of(keys)
                .mapToEntry(FeatureRequest::getClientId, identity())
                .toMap();
        var enabledFeaturesByClientId =
                calculateFeaturesStateForClientIds(featureRequests.values(), featureCache.getCached());

        return StreamEx.of(keys)
                .mapToEntry(identity(), req -> Set.copyOf(enabledFeaturesByClientId.getOrDefault(req.getClientId(),
                        emptyList())))
                .toMap();
    }

    private Set<String> loadPublicFeaturesForClientId(FeatureRequest key) {
        var clientRealFeatures =
                getClientFeatureStorage(List.of(key), featureCache.getCached()).getPublicFeaturesList();

        return listToSet(clientRealFeatures, feature -> feature.getFeature().getFeatureTextId());
    }

    private Set<String> loadFeaturesForUid(FeatureRequest key) {
        return calculateFeaturesStateForUids(List.of(key), featureCache.getCached()).values()
                .stream()
                .flatMap(Collection::stream)
                .collect(toSet());
    }

    private Map<FeatureRequest, Set<String>> loadFeaturesForUids(Collection<FeatureRequest> keys) {
        var featureRequests = StreamEx.of(keys)
                .mapToEntry(FeatureRequest::getUid, identity())
                .toMap();
        var enabledFeaturesByUid =
                calculateFeaturesStateForUids(List.copyOf(featureRequests.values()), featureCache.getCached());

        return StreamEx.of(keys)
                .mapToEntry(identity(), req -> Set.copyOf(enabledFeaturesByUid.getOrDefault(req.getUid(), emptyList())))
                .toMap();
    }
}
