package ru.yandex.direct.core.entity.feature.service;

import java.util.Collection;
import java.util.Map;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import one.util.streamex.StreamEx;

@ParametersAreNonnullByDefault
public class FeatureUtils {

    private FeatureUtils() {
    }

    /**
     * Вспомогательный потоконезависимый кеш для использования внутри FeatureService
     */
    public static class TLCache<K, V> {
        private static final int DEFAULT_MAX_SIZE = 10_000;

        private final ThreadLocal<LoadingCache<K, V>> tls;

        public static <K, V> Builder<K, V> builder() {
            return new Builder<>();
        }

        public TLCache(CacheLoader<K, V> loader, int timeoutInSeconds) {
            this(loader, timeoutInSeconds, DEFAULT_MAX_SIZE);
        }

        public TLCache(CacheLoader<K, V> loader, int timeoutInSeconds, int maxSize) {
            CacheBuilder<Object, Object> builder = CacheBuilder.newBuilder()
                    .concurrencyLevel(1)
                    .maximumSize(maxSize)
                    .expireAfterWrite(timeoutInSeconds, TimeUnit.SECONDS);

            tls = ThreadLocal.withInitial(() -> builder.build(loader));
        }

        public V get(K key) {
            LoadingCache<K, V> cache = tls.get();
            return cache.getUnchecked(key);
        }

        public Map<K, V> getAll(Collection<K> keys) throws ExecutionException {
            LoadingCache<K, V> cache = tls.get();
            return cache.getAll(keys);
        }

        public void clear() {
            tls.remove();
        }

        public static final class Builder<K, V> {
            private Function<K, V> loadFunc;
            private Function<Collection<K>, Map<K, V>> loadAllFunc;
            private int timeoutInSeconds;
            private int maxSize;

            private Builder() {
                loadFunc = key -> {
                    throw new UnsupportedOperationException();
                };
                loadAllFunc = keys -> {
                    throw new UnsupportedOperationException();
                };
                maxSize = DEFAULT_MAX_SIZE;
            }

            public Builder<K, V> setLoadFunc(Function<K, V> load) {
                this.loadFunc = load;
                return this;
            }

            public Builder<K, V> setLoadAllFunc(Function<Collection<K>, Map<K, V>> loadAll) {
                this.loadAllFunc = loadAll;
                return this;
            }

            public Builder<K, V> setTimeout(int timeoutInSeconds) {
                this.timeoutInSeconds = timeoutInSeconds;
                return this;
            }

            public Builder<K, V> setMaxSize(int maxSize) {
                this.maxSize = maxSize;
                return this;
            }

            public TLCache<K, V> build() {
                CacheLoader<K, V> cacheLoader = new CacheLoader<>() {
                    @Override
                    public V load(K key) {
                        return loadFunc.apply(key);
                    }

                    @Override
                    public Map<K, V> loadAll(Iterable<? extends K> keys) {
                        return loadAllFunc.apply(StreamEx.of(keys.iterator()).map(x -> (K) x).toList());
                    }
                };

                return new TLCache<>(cacheLoader, timeoutInSeconds, maxSize);
            }
        }
    }
}
