package ru.yandex.direct.core.entity.feature.service.validation;

import java.util.Collection;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.feature.container.FeatureTextIdToPercent;
import ru.yandex.direct.core.entity.feature.model.Feature;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.feature.FeatureType;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;
import ru.yandex.direct.validation.wrapper.ModelItemValidationBuilder;

import static ru.yandex.direct.validation.constraint.CommonConstraints.inSet;
import static ru.yandex.direct.validation.constraint.NumberConstraints.inRange;

@Service
public class FeaturePercentUpdateValidationService {

    private static final int MAX_PERCENT = 100;
    private static final int MIN_PERCENT = 0;

    public ValidationResult<List<FeatureTextIdToPercent>, Defect> validate(
            List<FeatureTextIdToPercent> featureTextIdToPercentList, Collection<Feature> existFeatures) {
        ListValidationBuilder<FeatureTextIdToPercent, Defect> lvb =
                ListValidationBuilder.of(featureTextIdToPercentList);
        Set<String> featuresWithChangeablePercent = existFeatures.stream()
                .map(Feature::getFeatureTextId)
                .filter(textId -> {
                    var featureName = FeatureName.fromString(textId);
                    return featureName != null && featureName.getFeatureType() == FeatureType.TEMP;
                })
                .collect(Collectors.toSet());
        lvb.checkEachBy(featureToPercent -> validateFeatureToClient(featureToPercent, featuresWithChangeablePercent));
        return lvb.getResult();
    }

    private ValidationResult<FeatureTextIdToPercent, Defect> validateFeatureToClient(
            FeatureTextIdToPercent featureTextIdToPercent,
            Set<String> featuresWithChangeablePercent) {
        ModelItemValidationBuilder<FeatureTextIdToPercent> vb = ModelItemValidationBuilder.of(featureTextIdToPercent);
        vb.item(FeatureTextIdToPercent.TEXT_ID).check(inSet(featuresWithChangeablePercent));
        vb.item(FeatureTextIdToPercent.PERCENT).check(inRange(MIN_PERCENT, MAX_PERCENT));

        return vb.getResult();
    }
}
