package ru.yandex.direct.core.entity.feature.service.validation;

import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import one.util.streamex.StreamEx;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.feature.container.FeatureTextIdToPercent;
import ru.yandex.direct.core.entity.feature.container.FeatureTextIdToRole;
import ru.yandex.direct.core.entity.feature.model.Feature;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.feature.FeatureType;
import ru.yandex.direct.rbac.RbacRole;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.builder.When;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.core.entity.feature.service.validation.FeatureDefects.featureIsAlreadyDisabledForRole;
import static ru.yandex.direct.core.entity.feature.service.validation.FeatureDefects.featureIsAlreadyEnabledForRole;
import static ru.yandex.direct.validation.constraint.CommonConstraints.inSet;
import static ru.yandex.direct.validation.constraint.CommonConstraints.notInSet;

@Service
public class FeatureRoleUpdateValidationService {
    public ValidationResult<List<FeatureTextIdToRole>, Defect> validate(
            List<FeatureTextIdToRole> featureToPercentList, Map<String, Feature> featuresByTextId, boolean isEnabling) {
        Set<String> roles = StreamEx.of(RbacRole.values())
                .map(RbacRole::name)
                .toSet();
        ListValidationBuilder<FeatureTextIdToRole, Defect> lvb =
                ListValidationBuilder.of(featureToPercentList);
        Set<String> featuresWithChangeableRoles = featuresByTextId.keySet().stream()
                .filter(textId -> {
                    var featureName = FeatureName.fromString(textId);
                    return featureName != null && featureName.getFeatureType() == FeatureType.TEMP;
                })
                .collect(Collectors.toSet());
        lvb.checkEachBy(featureToRole -> validateFeatureToClient(featureToRole, featuresByTextId,
                featuresWithChangeableRoles, roles, isEnabling));
        return lvb.getResult();
    }

    private ValidationResult<FeatureTextIdToRole, Defect> validateFeatureToClient(
            FeatureTextIdToRole featureTextIdToRole,
            Map<String, Feature> featuresByTextId,
            Set<String> featuresWithChangeableRoles,
            Set<String> existRoles,
            boolean isEnabling) {
        ItemValidationBuilder<FeatureTextIdToRole, Defect> vb = ItemValidationBuilder.of(featureTextIdToRole);
        vb.item(featureTextIdToRole.getTextId(), FeatureTextIdToPercent.TEXT_ID.name())
                .check(inSet(featuresWithChangeableRoles));
        ItemValidationBuilder<String, Defect> roleName =
                vb.item(featureTextIdToRole.getRoleName(), FeatureTextIdToRole.ROLE_NAME.name());

        roleName.check(inSet(existRoles));
        if (!vb.getResult().hasAnyErrors()) {
            roleName.check(notInSet(featuresByTextId.get(featureTextIdToRole.getTextId()).getSettings().getRoles()),
                    featureIsAlreadyEnabledForRole(), When.isTrue(isEnabling))
                    .check(inSet(featuresByTextId.get(featureTextIdToRole.getTextId()).getSettings().getRoles()),
                            featureIsAlreadyDisabledForRole(),
                            When.isValidAnd(When.isTrue(!isEnabling)));
        }
        return vb.getResult();
    }
}
