package ru.yandex.direct.core.entity.feature.service.validation;

import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import one.util.streamex.EntryStream;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.feature.container.FeatureTextIdToClientIdState;
import ru.yandex.direct.core.entity.feature.model.Feature;
import ru.yandex.direct.core.entity.feature.model.FeatureState;
import ru.yandex.direct.core.validation.defects.RightsDefects;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.rbac.RbacRole;
import ru.yandex.direct.validation.builder.Constraint;
import ru.yandex.direct.validation.builder.ItemValidationBuilder;
import ru.yandex.direct.validation.builder.ListValidationBuilder;
import ru.yandex.direct.validation.result.Defect;
import ru.yandex.direct.validation.result.ValidationResult;

import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.validation.constraint.CommonConstraints.inSet;

@Service
public class SwitchFeatureByClientIdValidationService {
    private final ShardHelper shardHelper;

    @Autowired
    public SwitchFeatureByClientIdValidationService(ShardHelper shardHelper) {
        this.shardHelper = shardHelper;
    }

    public ValidationResult<List<FeatureTextIdToClientIdState>, Defect> validate(
            List<FeatureTextIdToClientIdState> switchFeatureToClientList) {
        Set<Long> existentClientIds =
                getExistingClientIds(mapList(switchFeatureToClientList, FeatureTextIdToClientIdState::getClientId));
        ListValidationBuilder<FeatureTextIdToClientIdState, Defect> lvb =
                ListValidationBuilder.of(switchFeatureToClientList);
        lvb.checkEachBy(featureToUid -> validateFeatureToClient(featureToUid, existentClientIds));
        return lvb.getResult();
    }

    public ValidationResult<List<ClientId>, Defect> validateClientIds(List<ClientId> clientIds) {
        var existingClientIds = getExistingClientIds(clientIds);
        return ListValidationBuilder.<ClientId, Defect>of(clientIds)
                .checkEachBy(clientId -> validateClientId(clientId, existingClientIds))
                .getResult();
    }

    private ValidationResult<ClientId, Defect> validateClientId(ClientId clientId, Set<Long> existingClientIds) {
        var vb = ItemValidationBuilder.<ClientId, Defect>of(clientId);
        vb.item(clientId.asLong(), "clientId")
                .check(inSet(existingClientIds));
        return vb.getResult();
    }

    private Set<Long> getExistingClientIds(List<ClientId> clientIds) {
        return EntryStream.of(shardHelper.getExistingClientIds(mapList(clientIds, ClientId::asLong)))
                .filterValues(s -> s)
                .keys().toSet();
    }

    private ValidationResult<FeatureTextIdToClientIdState, Defect> validateFeatureToClient(
            FeatureTextIdToClientIdState featureTextIdToClientIdState,
            Set<Long> existedClientIds) {
        ItemValidationBuilder<FeatureTextIdToClientIdState, Defect> vb =
                ItemValidationBuilder.of(featureTextIdToClientIdState);
        vb.item(featureTextIdToClientIdState.getClientId().asLong(), FeatureTextIdToClientIdState.CLIENT_ID.name())
                .check(inSet(existedClientIds));
        return vb.getResult();
    }

    /**
     * Проверить, имеет ли роль role право включить/выключить фичи
     *
     * @param featuresByTextId - словарь фич по textId
     * @param role             - роль, для которой проверяются права
     * @param states           - фичи, по которым хотят изменить статусы
     */
    public ValidationResult<List<FeatureTextIdToClientIdState>, Defect> validateRolePermissions(
            Map<String, Feature> featuresByTextId,
            RbacRole role,
            List<FeatureTextIdToClientIdState> states) {
        return ListValidationBuilder.of(states, Defect.class)
                .checkEachBy(state -> validateRolePerm(featuresByTextId, role, state))
                .getResult();
    }

    private ValidationResult<FeatureTextIdToClientIdState, Defect> validateRolePerm(
            Map<String, Feature> featuresByTextId, RbacRole role, FeatureTextIdToClientIdState state) {
        Feature feature = featuresByTextId.get(state.getTextId());
        Optional<Set<String>> allowedRoles = Optional.ofNullable(feature)
                .map(Feature::getSettings)
                .map(s -> state.getState() == FeatureState.ENABLED ? s.getCanEnable()
                        : state.getState() == FeatureState.DISABLED ? s.getCanDisable()
                        : null
                );

        boolean allowed = allowedRoles.map(s -> s.stream().anyMatch(i -> StringUtils.equalsIgnoreCase(i, role.name())))
                .orElse(false);

        ItemValidationBuilder<FeatureTextIdToClientIdState, Defect> vb =
                ItemValidationBuilder.of(state, Defect.class);
        vb.item(state.getState(), FeatureTextIdToClientIdState.STATE.name())
                .check(Constraint.fromPredicate(x -> allowed, RightsDefects.noRights()));
        return vb.getResult();
    }
}
