package ru.yandex.direct.core.entity.feed

import java.math.BigInteger
import java.time.LocalDateTime
import one.util.streamex.EntryStream
import one.util.streamex.StreamEx
import ru.yandex.direct.bmapi.client.model.BmApiErrorCode.BL_ERROR_BUSINESS_TYPE_MISMATCH
import ru.yandex.direct.bmapi.client.model.BmApiErrorCode.BL_ERROR_CATEGORY_NOT_POSITIVE_INTEGER
import ru.yandex.direct.bmapi.client.model.BmApiErrorCode.BL_ERROR_EMPTY_FILE
import ru.yandex.direct.bmapi.client.model.BmApiErrorCode.BL_ERROR_FATAL
import ru.yandex.direct.bmapi.client.model.BmApiErrorCode.BL_ERROR_FEED_DATA_TYPE_MISMATCH
import ru.yandex.direct.bmapi.client.model.BmApiErrorCode.BL_ERROR_FEED_TYPE_MISMATCH
import ru.yandex.direct.bmapi.client.model.BmApiErrorCode.BL_ERROR_FETCH_FAILED
import ru.yandex.direct.bmapi.client.model.BmApiErrorCode.BL_ERROR_FILE_TOO_BIG
import ru.yandex.direct.bmapi.client.model.BmApiErrorCode.BL_ERROR_XML_FATAL
import ru.yandex.direct.bmapi.client.model.BmApiErrorCode.BL_ERROR_XML_NO_CATEGORIES
import ru.yandex.direct.bmapi.client.model.BmApiErrorCode.BL_ERROR_XML_NO_OFFERS
import ru.yandex.direct.bmapi.client.model.BmApiErrorCode.BL_ERROR_YML_NO_MODEL_OR_TAG_NAME
import ru.yandex.direct.bmapi.client.model.BmApiErrorCode.BL_ERROR_YML_NO_URL_TAG
import ru.yandex.direct.bmapi.client.model.BmApiFeedInfoResponse
import ru.yandex.direct.bmapi.client.model.Category
import ru.yandex.direct.core.entity.feed.model.Feed
import ru.yandex.direct.core.entity.feed.model.FeedCategory
import ru.yandex.direct.core.entity.feed.model.FeedHistoryItem
import ru.yandex.direct.core.entity.feed.model.FeedHistoryItemParseResults
import ru.yandex.direct.core.entity.feed.model.FeedHistoryItemParseResultsDefect
import ru.yandex.direct.core.entity.feed.model.FeedSimple
import ru.yandex.direct.core.entity.feed.model.FeedType
import ru.yandex.direct.core.entity.feed.model.UpdateStatus
import ru.yandex.direct.model.AppliedChanges
import ru.yandex.direct.model.ModelChanges
import ru.yandex.direct.utils.JsonUtils

// ошибки которые мы скрываем от клиента (проблема на стороне BannerLand)
// после этих ошибок не добавляется запись об ошибке в таблицу perf_feed_history
// дальнейшая обработка фида должна быть прекращена
val FATAL_BL_ERRORS = setOf(BL_ERROR_FATAL)

// ошибки для клиента (может исправить их изменив фид)
// после этих ошибок в таблицу perf_feed_history добавляется запись об ошибке
// дальнейшая обработка фида должна быть прекращена
val CLIENT_BL_ERRORS = setOf(
    BL_ERROR_FETCH_FAILED,
    BL_ERROR_EMPTY_FILE,
    BL_ERROR_FILE_TOO_BIG,
    BL_ERROR_YML_NO_MODEL_OR_TAG_NAME,
    BL_ERROR_XML_FATAL,
    BL_ERROR_XML_NO_OFFERS,
    BL_ERROR_XML_NO_CATEGORIES,
    BL_ERROR_YML_NO_URL_TAG,
    BL_ERROR_FEED_DATA_TYPE_MISMATCH,
    BL_ERROR_FEED_TYPE_MISMATCH,
    BL_ERROR_BUSINESS_TYPE_MISMATCH,
    BL_ERROR_CATEGORY_NOT_POSITIVE_INTEGER,
)

fun getFeedsAppliedChangesForErrorBLResponse(
    feedsToUpdate: List<FeedSimple>,
    feedIdToErrorBLResponses: Map<Long, BmApiFeedInfoResponse>,
    feedIdsWithIncorrectFeedType: List<Long>,
    now: LocalDateTime
) = StreamEx.of(feedsToUpdate)
    .filter { feedIdToErrorBLResponses.containsKey(it.id) }
    .map { feedSimple: FeedSimple ->
        getErrorAppliedChanges(feedSimple, now)
    }
    .nonNull()
    .append(feedIdsWithIncorrectFeedType.map { feedId ->
        val feedSimple = feedsToUpdate.firstOrNull { it.id == feedId }!!
        getErrorAppliedChanges(feedSimple, now)
    })
    .toList() as List<AppliedChanges<Feed>>

private fun getErrorAppliedChanges(
    feedSimple: FeedSimple,
    now: LocalDateTime
): AppliedChanges<Feed> {
    val changes = ModelChanges(feedSimple.id, Feed::class.java)
    changes.process(now, Feed.LAST_CHANGE)
    changes.process(UpdateStatus.ERROR, Feed.UPDATE_STATUS)
    changes.process(1, Feed.FETCH_ERRORS_COUNT)
    return changes.applyTo(feedSimple as Feed)
}

fun bmApiFeedInfoResponseHasErrors(bmApiFeedInfoResponse: BmApiFeedInfoResponse) =
    !bmApiFeedInfoResponse.errors.isNullOrEmpty()

/**
 * Метод делит хорошие ответы от BmAPI на 2 списка id фидов — те, у которых все хорошо, и их надо обновлять.
 * И те, у которых пришел неизвестный feedType или он изменился(исключение — изменение с null на YandexMarket)
 * Отдельно учитывается, что с ошибкой может прийти feedType=null, это считаем валидным
 */
fun validateFeedIdsToUpdate(
    feedsToUpdate: List<FeedSimple>,
    feedIdToBlFeedInfo: Map<Long, BmApiFeedInfoResponse>
): Pair<List<Long>, List<Long>> {
    val validFeedIds: MutableList<Long> = mutableListOf()
    val invalidFeedIds: MutableList<Long> = mutableListOf()
    feedsToUpdate.forEach { feedSimple ->
        val newType: FeedType? = try {
            FeedType.fromTypedValue(feedIdToBlFeedInfo[feedSimple.id]!!.feedType)
        } catch (e: IllegalArgumentException) {
            //TODO: log
            invalidFeedIds.add(feedSimple.id)
            return@forEach
        }
        if (newType == feedSimple.feedType
            || (newType == FeedType.YANDEX_MARKET && feedSimple.feedType == null)
            || bmApiFeedInfoResponseHasErrors(feedIdToBlFeedInfo[feedSimple.id]!!)
        ) {
            validFeedIds.add(feedSimple.id)
        } else {
            invalidFeedIds.add(feedSimple.id)
        }
    }
    return Pair(validFeedIds, invalidFeedIds)
}

fun getFeedsAppliedChangesForSuccessfulBLResponse(
    feedsToUpdate: List<FeedSimple>,
    feedIdToSuccessefullBLResponses: Map<Long, BmApiFeedInfoResponse>,
    now: LocalDateTime
): List<AppliedChanges<Feed>> = feedsToUpdate
    .filter { feedIdToSuccessefullBLResponses.containsKey(it.id) }
    .mapNotNull { feedSimple: FeedSimple ->
        val changes = ModelChanges(feedSimple.id, Feed::class.java)
        val response = feedIdToSuccessefullBLResponses[feedSimple.id]!!
        changes.process(FeedType.fromTypedValue(response.feedType!!), Feed.FEED_TYPE)
        changes.process(response.totalOffersAmount, Feed.OFFERS_COUNT)
        changes.process(
            EntryStream.of(response.domain).max { o1, o2 -> o1.value.compareTo(o2.value) }.get().key,
            Feed.TARGET_DOMAIN
        )
        changes.process(JsonUtils.toJson(response.feedOfferExamples), Feed.OFFER_EXAMPLES)

        changes.process(now, Feed.LAST_CHANGE)
        changes.process(now, Feed.LAST_REFRESHED)
        changes.process(UpdateStatus.DONE, Feed.UPDATE_STATUS)
        changes.process(0, Feed.FETCH_ERRORS_COUNT)

        changes.applyTo(feedSimple as Feed)
    }

fun getOffersCountForCategoryInFeed(
    bmApiFeedInfoResponse: BmApiFeedInfoResponse,
    category: Category
) = bmApiFeedInfoResponse.categoryIdsToOffersCount?.let { it[category.id] }?.toLong()

fun toDbCategoryId(categoryId: String?): BigInteger =
    if (categoryId == null)
        BigInteger.ZERO
    else
        BigInteger(categoryId)

fun toFeedCategory(feedId: Long, category: Category, bmApiFeedInfoResponse: BmApiFeedInfoResponse): FeedCategory =
    FeedCategory()
        .withFeedId(feedId)
        .withName(category.name)
        .withCategoryId(BigInteger(category.id))
        .withParentCategoryId(toDbCategoryId(category.parentId))
        .withOfferCount(getOffersCountForCategoryInFeed(bmApiFeedInfoResponse, category))
        .withIsDeleted(false)

fun toFeedHistoryItem(feedId: Long, createdAt: LocalDateTime, bmApiFeedInfoResponse: BmApiFeedInfoResponse) =
    FeedHistoryItem().apply {
        this.feedId = feedId
        this.createdAt = createdAt

        offerCount = bmApiFeedInfoResponse.totalOffersAmount
        parseResults = FeedHistoryItemParseResults().apply {
            errors = bmApiFeedInfoResponse.errors?.map {
                FeedHistoryItemParseResultsDefect().apply {
                    code = it.error.code.toLong()
                    messageEn = it.messageEn
                    messageRu = it.messageRu
                }
            }
            warnings = bmApiFeedInfoResponse.warnings?.map {
                FeedHistoryItemParseResultsDefect().apply {
                    code = it.code.toLong()
                    messageEn = it.messageEn
                    messageRu = it.messageRu
                }
            }
        }
    }
