package ru.yandex.direct.core.entity.feed

import ru.yandex.direct.common.log.container.LogType
import ru.yandex.direct.common.log.service.CommonDataLogService
import ru.yandex.direct.core.entity.feature.model.FeatureState
import ru.yandex.direct.core.entity.feed.model.Feed
import ru.yandex.direct.core.entity.feed.model.FeedSimple
import ru.yandex.direct.core.entity.feed.model.FeedType
import ru.yandex.direct.core.entity.feed.model.FeedUsageType
import ru.yandex.direct.core.entity.feed.model.MasterSystem
import ru.yandex.direct.core.entity.feed.model.Source
import ru.yandex.direct.feature.FeatureName
import ru.yandex.direct.model.AppliedChanges
import ru.yandex.direct.model.ModelChanges
import ru.yandex.direct.utils.CollectionUtils.isEmpty
import ru.yandex.direct.utils.DateTimeUtils.moscowDateTimeToEpochSecond
import ru.yandex.direct.utils.model.UrlParts
import java.net.URLDecoder
import java.net.URLEncoder
import java.nio.charset.StandardCharsets
import java.time.Instant
import java.time.LocalDateTime
import java.time.ZoneId
import java.util.EnumSet

const val DEFAULT_REFRESH_INTERVAL = 86_400L // секунд в сутках
const val FAKE_FEED_DOMAIN = "market.feed"
const val FAKE_FEED_URL_PREFIX = "https://$FAKE_FEED_DOMAIN/"
const val FAKE_URL_PARAM = "url"
const val FAKE_SITE_PARAM = "site"
const val MARKET_FEED_ID_URL_PARAM = "market_feed_id"
private const val BUSINESS_ID_URL_PARAM = "business_id"
private const val SHOP_ID_URL_PARAM = "shop_id"
private const val FROM_MARKET_FEED_URL_PARAM = "market"
private const val MARKET_FEED_FORCE_DC_URL_PARAM = "force_dc"

@JvmField
val FEED_FEATURE_TO_FEED_TYPE_MAP = mapOf(
    FeatureName.SEND_TO_MBI_FEED_TYPE_ALIBABA.getName() to FeedType.ALIBABA,
    FeatureName.SEND_TO_MBI_FEED_TYPE_GOOGLEMERCHANT.getName() to FeedType.GOOGLE_MERCHANT,
    FeatureName.SEND_TO_MBI_FEED_TYPE_CRITEOTURKEY.getName() to FeedType.CRITEO_TURKEY,
    FeatureName.SEND_TO_MBI_FEED_TYPE_TURKEYTEKNOSA.getName() to FeedType.TURKEY_TEKNOSA,
    FeatureName.SEND_TO_MBI_FEED_TYPE_ALIEXPRESS.getName() to FeedType.ALIEXPRESS,
    FeatureName.SEND_TO_MBI_FEED_TYPE_TURKEYCLIENT25.getName() to FeedType.TURKEY_CLIENT_25,
    FeatureName.SEND_TO_MBI_FEED_TYPE_YANDEXREALTY.getName() to FeedType.YANDEX_REALTY,
    FeatureName.SEND_TO_MBI_FEED_TYPE_AUTORU.getName() to FeedType.AUTO_RU,
    FeatureName.SEND_TO_MBI_FEED_TYPE_GOOGLEFLIGHTS.getName() to FeedType.GOOGLE_FLIGHTS,
    FeatureName.SEND_TO_MBI_FEED_TYPE_GOOGLETRAVEL.getName() to FeedType.GOOGLE_TRAVEL,
    FeatureName.SEND_TO_MBI_FEED_TYPE_GOOGLEHOTELS.getName() to FeedType.GOOGLE_HOTELS,
    FeatureName.SEND_TO_MBI_FEED_TYPE_TRAVELBOOKING.getName() to FeedType.TRAVEL_BOOKING,
    FeatureName.SEND_TO_MBI_FEED_TYPE_TURKEYJOLLYTUR.getName() to FeedType.TURKEY_JOLLY_TUR,
    FeatureName.SEND_TO_MBI_FEED_TYPE_YANDEXCUSTOM.getName() to FeedType.YANDEX_CUSTOM,
    FeatureName.SEND_TO_MBI_FEED_TYPE_GOOGLECUSTOM.getName() to FeedType.GOOGLE_CUSTOM,
    FeatureName.SEND_TO_MBI_FEED_TYPE_YANDEXMARKET.getName() to FeedType.YANDEX_MARKET
)

/**
 * @param isMarketFeed определяет получен ли фид из маркетной выгрузки в джобе `SyncMarketFeedsJob`
 * @param isForceDC равен `true` - если для фида нужно выгружать оффера из DataCamp'a (по `business_id` и `shop_id`),
 * при этом `srcUrl` в таком случае используется для генерации дерева категорий
 */
fun createFakeFeedUrl(
    businessId: Long?,
    shopId: Long?,
    marketFeedId: Long?,
    srcUrl: String,
    isMarketFeed: Boolean = false,
    isForceDC: Boolean = false
): String = createFakeFeedUrl(businessId, shopId, marketFeedId, srcUrl, FAKE_URL_PARAM, isMarketFeed, isForceDC)

fun createFakeFeedUrl(
    businessId: Long?,
    shopId: Long?,
    marketFeedId: Long?,
    schema: String,
    domain: String
): String {
    val site = "$schema://$domain/"
    return createFakeFeedUrl(businessId, shopId, marketFeedId, site, FAKE_SITE_PARAM, false, false)
}

private fun createFakeFeedUrl(
    businessId: Long?,
    shopId: Long?,
    marketFeedId: Long?,
    url: String,
    urlParam: String,
    isMarketFeed: Boolean,
    isForceDC: Boolean
): String {
    val encodedUrl = URLEncoder.encode(url, StandardCharsets.UTF_8)
    val sb = StringBuilder(encodedUrl.length + 128)
    sb.append("${FAKE_FEED_URL_PREFIX}?${urlParam}=${encodedUrl}")
    appendIdsParts(sb, businessId, shopId, marketFeedId)

    if (isMarketFeed) {
        sb.append("&$FROM_MARKET_FEED_URL_PARAM=$isMarketFeed")
    }

    if (isForceDC) {
        sb.append("&$MARKET_FEED_FORCE_DC_URL_PARAM=$isForceDC")
    }

    return sb.toString()
}

fun unFakeUrlIfNeeded(url: String): String {
    if (!isFakeUrl(url)) {
        return url
    }
    val unFakedUrl = tryGetSrcUrl(url)

    return if (unFakedUrl == null) url else unFakeUrlIfNeeded(unFakedUrl)
}

private fun appendIdsParts(
    sb: StringBuilder,
    businessId: Long?,
    shopId: Long?,
    marketFeedId: Long?
) {
    if (businessId != null) {
        sb.append("&${BUSINESS_ID_URL_PARAM}=${businessId}")
    }
    if (shopId != null) {
        sb.append("&${SHOP_ID_URL_PARAM}=${shopId}")
    }
    if (marketFeedId != null) {
        sb.append("&${MARKET_FEED_ID_URL_PARAM}=${marketFeedId}")
    }
}

fun tryGetSrcUrl(fakeUrl: String) =
    splitFakeUrlParams(fakeUrl)
        ?.firstOrNull { FAKE_URL_PARAM == it[0] || FAKE_SITE_PARAM == it[0] }
        ?.get(1)
        ?.let { URLDecoder.decode(it, StandardCharsets.UTF_8) }

fun getForceDc(fakeUrl: String) =
    splitFakeUrlParams(fakeUrl)
        ?.firstOrNull { MARKET_FEED_FORCE_DC_URL_PARAM == it[0] }
        ?.get(1)
        ?.let { "true" == it }
        ?: false

private fun splitFakeUrlParams(fakeUrl: String) =
    fakeUrl.takeIf { it.startsWith(FAKE_FEED_URL_PREFIX) }
        ?.split("?")
        ?.get(1)
        ?.split("&")
        ?.map { it.split("=") }

fun normalizeSiteUrl(url: String) =
    UrlParts.fromUrl(url)
        .let { "${it.protocol}://${it.domain}/" }

fun toFeedTypes(allowedToSendFeedTypesFeatureStates: Map<String, FeatureState>): Set<FeedType> {
    return allowedToSendFeedTypesFeatureStates
        .asSequence()
        .filter { it.value == FeatureState.ENABLED }
        .map { toFeedType(it.key) }
        .toSet()
}

fun toFeedType(featureName: String): FeedType {
    return FEED_FEATURE_TO_FEED_TYPE_MAP[featureName]
        ?: throw IllegalArgumentException("Unknown feature for feed type: $featureName")
}

fun isFakeUrl(url: String): Boolean {
    return UrlParts.fromUrl(url).domain.equals(FAKE_FEED_DOMAIN)
}

/**
 * Метод проверяющий есть ли изменения в типах использования. NULL и пустой список имеют одинаковый смысл
 */
fun usageTypesEquals(feed: FeedSimple, usageTypes: EnumSet<FeedUsageType>): Boolean {
    val oldUsageTypes = feed.usageTypes
    return oldUsageTypes?.equals(usageTypes) ?: (oldUsageTypes == usageTypes || usageTypes.isEmpty())
}

fun feedWithUsageTypeToAppliedChanges(feedsToUpdateUsage: Map<FeedSimple, EnumSet<FeedUsageType>>)
    : List<AppliedChanges<Feed>> {
    val actualDateTime = Instant.now().atZone(ZoneId.systemDefault()).toLocalDateTime()
    return feedsToUpdateUsage.asSequence()
        .filter { (feed, newUsageTypes) -> !usageTypesEquals(feed, newUsageTypes) }
        .map { (feed, newUsageTypes) ->
            ModelChanges(feed.id, Feed::class.java)
                .process(newUsageTypes, Feed.USAGE_TYPES)
                // устанавливаем время последнего использования, если фид теперь не используется, или null
                .process(actualDateTime.takeIf { isEmpty(newUsageTypes) }, Feed.LAST_USED)
                .applyTo(feed as Feed)
        }.toList()
}

fun logFeedStatusDelete(shard: Int, feedIds: Collection<Long>, commonDataLogService: CommonDataLogService) {
    val now = moscowDateTimeToEpochSecond(LocalDateTime.now())
    feedIds.forEach {
        commonDataLogService.log(
            LogType.FEED_STATUS,
            FeedStatusLogRecord(
                shard,
                null,
                it,
                now,
                "Deleted",
                null,
                null,
                0
            )
        )
    }
}

fun logFeedStatusUpdate(
    shard: Int,
    feeds: Collection<AppliedChanges<Feed>>,
    commonDataLogService: CommonDataLogService
) {
    val now = moscowDateTimeToEpochSecond(LocalDateTime.now())
    feeds.filter { it.changed(Feed.UPDATE_STATUS) }
        .forEach {
            commonDataLogService.log(
                LogType.FEED_STATUS,
                FeedStatusLogRecord(
                    shard,
                    it.model.clientId,
                    it.model.id,
                    now,
                    it.model.updateStatus.name,
                    it.model.source,
                    it.model.masterSystem,
                    it.model.offersCount ?: 0
                )
            )
        }
}

fun logFeedStatus(shard: Int, feeds: Collection<FeedSimple>, commonDataLogService: CommonDataLogService) {
    val now = moscowDateTimeToEpochSecond(LocalDateTime.now())
    feeds.forEach {
        commonDataLogService.log(
            LogType.FEED_STATUS,
            FeedStatusLogRecord(
                shard,
                it.clientId,
                it.id,
                now,
                it.updateStatus.name,
                it.source,
                it.masterSystem,
                it.offersCount ?: 0
            )
        )
    }
}

data class FeedStatusLogRecord(
    val shard: Int,
    val clientId: Long?,
    val feedId: Long,
    val timestamp: Long,
    val status: String,
    val source: Source?,
    val masterSystem: MasterSystem?,
    val offersCount: Long
)
