package ru.yandex.direct.core.entity.feed.converter;

import java.math.BigDecimal;
import java.util.List;
import java.util.Map;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;

import ru.yandex.direct.core.entity.banner.model.ImageSize;
import ru.yandex.direct.core.entity.feed.model.BusinessType;
import ru.yandex.direct.core.entity.feed.model.Feed;
import ru.yandex.direct.core.entity.feed.model.FeedOfferExamples;
import ru.yandex.direct.core.entity.feed.model.FeedSimple;
import ru.yandex.direct.core.entity.feed.model.FeedType;
import ru.yandex.direct.core.entity.feed.model.MasterSystem;
import ru.yandex.direct.core.entity.feed.model.Source;
import ru.yandex.direct.core.entity.feedoffer.converter.FeedOffersToYmlConverter;
import ru.yandex.direct.core.entity.feedoffer.converter.FeedOffersToYmlConverterKt;
import ru.yandex.direct.core.entity.feedoffer.converter.YmlCatalog;
import ru.yandex.direct.core.entity.feedoffer.model.FeedOffer;
import ru.yandex.direct.dbutil.model.BusinessIdAndShopId;
import ru.yandex.direct.model.ModelChanges;
import ru.yandex.direct.utils.JsonUtils;

import static java.util.Collections.emptyMap;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;

@ParametersAreNonnullByDefault
public class FeedConverter {
    private FeedConverter() {
    }

    @Nullable
    public static BusinessIdAndShopId extractBusinessIdAndShopId(FeedSimple feed) {
        return BusinessIdAndShopId.ofNullable(feed.getMarketBusinessId(), feed.getMarketShopId());
    }

    public static Feed convertToManualFeed(String name, List<FeedOffer> offers, Map<String, ImageSize> imageUrlToSize) {
        return new Feed()
                .withName(name)
                .withFeedType(FeedType.YANDEX_MARKET)
                .withBusinessType(BusinessType.RETAIL)
                .withSource(Source.FILE)
                .withFileData(convertFeedOffers(offers))
                .withFilename(name + ".yml")
                .withMasterSystem(MasterSystem.MANUAL)
                .withOfferExamples(JsonUtils.toJson(getOfferExamples(offers, imageUrlToSize)));
    }

    public static ModelChanges<Feed> convertToManualFeedChanges(Long feedId, List<FeedOffer> offers,
                                                                Map<String, ImageSize> imageUrlToSize) {
        return new ModelChanges<>(feedId, Feed.class)
                .processNotNull(convertFeedOffers(offers), Feed.FILE_DATA)
                .processNotNull(JsonUtils.toJson(getOfferExamples(offers, imageUrlToSize)), Feed.OFFER_EXAMPLES);
    }

    private static byte[] convertFeedOffers(List<FeedOffer> offers) {
        YmlCatalog ymlCatalog = FeedOffersToYmlConverter.INSTANCE.convertRetailOffersToYml(offers);
        return FeedOffersToYmlConverterKt.toByteArray(ymlCatalog, false);
    }

    private static FeedOfferExamples getOfferExamples(List<FeedOffer> offers,
                                                      Map<String, ImageSize> imageUrlToSize) {
        if (imageUrlToSize.isEmpty()) {
            return new FeedOfferExamples(emptyMap());
        }

        return new FeedOfferExamples(StreamEx.of(offers)
                .filter(offer -> imageUrlToSize.get(offer.getImages().get(0)) != null)
                .toMap(offer -> offer.getId().toString(), offer -> getOfferExample(offer, imageUrlToSize)));
    }

    private static FeedOfferExamples.Offer getOfferExample(FeedOffer feedOffer,
                                                           Map<String, ImageSize> imageUrlToSize) {
        String imageUrl = feedOffer.getImages().get(0);
        ImageSize imageSize = imageUrlToSize.get(imageUrl);
        FeedOfferExamples.Image image = new FeedOfferExamples.Image(
                imageSize.getWidth(), imageSize.getHeight(), imageUrl, emptyMap());
        String href = feedOffer.getHref();

        return new FeedOfferExamples.Offer(
                href,
                new FeedOfferExamples.Price(
                        feedOffer.getCurrentPrice().toString(),
                        ifNotNull(feedOffer.getOldPrice(), BigDecimal::toString)),
                new FeedOfferExamples.Text(
                        feedOffer.getDescription(),
                        null,
                        feedOffer.getLabel(),
                        feedOffer.getCurrency().toString()),
                new FeedOfferExamples.ClickUrls(href, href, href, href, href, href, href, href),
                new FeedOfferExamples.ImageInfo(image, image, image, image),
                null
        );
    }
}
