package ru.yandex.direct.core.entity.feed.repository;

import java.time.Duration;
import java.time.LocalDateTime;
import java.time.temporal.ChronoUnit;
import java.util.Collection;
import java.util.EnumSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.jetbrains.annotations.NotNull;
import org.jooq.Condition;
import org.jooq.DSLContext;
import org.jooq.Field;
import org.jooq.Record;
import org.jooq.SelectConditionStep;
import org.jooq.SortField;
import org.jooq.TableField;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.common.db.PpcPropertiesSupport;
import ru.yandex.direct.common.log.service.CommonDataLogService;
import ru.yandex.direct.common.util.RepositoryUtils;
import ru.yandex.direct.core.entity.SortOrder;
import ru.yandex.direct.core.entity.container.CampaignIdAndAdGroupIdPair;
import ru.yandex.direct.core.entity.feed.container.FeedQueryFilter;
import ru.yandex.direct.core.entity.feed.container.FeedsOrderBy;
import ru.yandex.direct.core.entity.feed.model.BusinessType;
import ru.yandex.direct.core.entity.feed.model.Feed;
import ru.yandex.direct.core.entity.feed.model.FeedSimple;
import ru.yandex.direct.core.entity.feed.model.FeedStatusesAndMarketIds;
import ru.yandex.direct.core.entity.feed.model.FeedType;
import ru.yandex.direct.core.entity.feed.model.MasterSystem;
import ru.yandex.direct.core.entity.feed.model.Source;
import ru.yandex.direct.core.entity.feed.model.StatusMBIEnabled;
import ru.yandex.direct.core.entity.feed.model.StatusMBISynced;
import ru.yandex.direct.core.entity.feed.model.UpdateStatus;
import ru.yandex.direct.core.entity.feed.processing.FeedOrderByField;
import ru.yandex.direct.dbschema.ppc.enums.FeedsBusinessType;
import ru.yandex.direct.dbschema.ppc.enums.FeedsMasterSystem;
import ru.yandex.direct.dbschema.ppc.enums.FeedsSource;
import ru.yandex.direct.dbschema.ppc.enums.FeedsStatusmbienabled;
import ru.yandex.direct.dbschema.ppc.enums.FeedsStatusmbisynced;
import ru.yandex.direct.dbschema.ppc.enums.FeedsUpdateStatus;
import ru.yandex.direct.dbschema.ppc.enums.PhrasesAdgroupType;
import ru.yandex.direct.dbutil.QueryWithoutIndex;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.dbutil.wrapper.DslContextProvider;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder;
import ru.yandex.direct.jooqmapperhelper.InsertHelper;
import ru.yandex.direct.jooqmapperhelper.UpdateHelper;
import ru.yandex.direct.model.AppliedChanges;
import ru.yandex.direct.multitype.entity.LimitOffset;
import ru.yandex.direct.utils.CollectionUtils;
import ru.yandex.direct.utils.crypt.Encrypter;

import static java.time.LocalDateTime.now;
import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static java.util.Collections.singletonList;
import static java.util.stream.Collectors.toList;
import static org.jooq.impl.DSL.not;
import static org.jooq.impl.DSL.row;
import static ru.yandex.direct.common.db.PpcPropertyNames.SEND_NEW_FEEDS_IN_BL_FROM_JOBS;
import static ru.yandex.direct.common.util.RepositoryUtils.nullSafeWriter;
import static ru.yandex.direct.core.entity.feed.FeedUtilsKt.logFeedStatus;
import static ru.yandex.direct.dbschema.ppc.Tables.ADGROUPS_DYNAMIC;
import static ru.yandex.direct.dbschema.ppc.Tables.ADGROUPS_PERFORMANCE;
import static ru.yandex.direct.dbschema.ppc.Tables.CAMPAIGNS;
import static ru.yandex.direct.dbschema.ppc.Tables.FEEDS;
import static ru.yandex.direct.dbschema.ppc.Tables.PHRASES;
import static ru.yandex.direct.dbschema.ppc.tables.AdgroupsText.ADGROUPS_TEXT;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty;
import static ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property;
import static ru.yandex.direct.utils.CollectionUtils.isNotEmpty;
import static ru.yandex.direct.utils.CommonUtils.coalesce;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Репозиторий для работы с фидам. (Товарный фид — систематизированный и структурированный список данных
 * о товарных предложениях или услугах.)
 * {@see https://yandex.ru/support/direct/dynamic-text-ads/feeds.html}
 */
@Repository
@ParametersAreNonnullByDefault
public class FeedRepository {
    public static final int DAYS_AFTER_CAMPAIGN_STOP_TO_SEND_FEED = 7;
    private final ShardHelper shardHelper;
    private final DslContextProvider dslContextProvider;
    private final Encrypter encrypter;
    private final FeedSupplementaryDataRepository feedSupplementaryDataRepository;
    private final JooqMapperWithSupplier<Feed> mapper;
    private final PpcPropertiesSupport ppcPropertiesSupport;
    private final CommonDataLogService commonDataLogService;
    private static final Condition SENT_TO_MBI = FEEDS.MARKET_SHOP_ID.isNotNull()
            .or(FEEDS.STATUS_MBI_SYNCED.eq(FeedsStatusmbisynced.Yes));

    public FeedRepository(Encrypter encrypter,
                          ShardHelper shardHelper,
                          DslContextProvider dslContextProvider,
                          FeedSupplementaryDataRepository feedSupplementaryDataRepository,
                          PpcPropertiesSupport ppcPropertiesSupport,
                          CommonDataLogService commonDataLogService) {
        this.encrypter = encrypter;
        this.shardHelper = shardHelper;
        this.dslContextProvider = dslContextProvider;
        this.feedSupplementaryDataRepository = feedSupplementaryDataRepository;
        this.ppcPropertiesSupport = ppcPropertiesSupport;
        this.commonDataLogService = commonDataLogService;

        mapper = JooqMapperWithSupplierBuilder.builder(Feed::new)
                .map(property(Feed.ID, FEEDS.FEED_ID))
                .map(property(Feed.CLIENT_ID, FEEDS.CLIENT_ID))
                .map(property(Feed.MARKET_BUSINESS_ID, FEEDS.MARKET_BUSINESS_ID))
                .map(property(Feed.MARKET_SHOP_ID, FEEDS.MARKET_SHOP_ID))
                .map(property(Feed.MARKET_FEED_ID, FEEDS.MARKET_FEED_ID))
                .map(convertibleProperty(Feed.STATUS_MBI_SYNCED, FEEDS.STATUS_MBI_SYNCED,
                        StatusMBISynced::fromSource, FeedMappings::toStatusMbiSynced))
                .map(convertibleProperty(Feed.STATUS_MBI_ENABLED, FEEDS.STATUS_MBI_ENABLED,
                        StatusMBIEnabled::fromSource, FeedMappings::toStatusMbiEnabled))
                .map(convertibleProperty(Feed.FEED_TYPE, FEEDS.FEED_TYPE,
                        FeedType::fromTypedValue, nullSafeWriter(FeedType::getTypedValue)))
                .map(convertibleProperty(Feed.BUSINESS_TYPE, FEEDS.BUSINESS_TYPE,
                        BusinessType::fromSource, FeedMappings::toBusinessType))
                .map(convertibleProperty(Feed.SOURCE, FEEDS.SOURCE,
                        Source::fromSource, Source::toSource))
                .map(property(Feed.NAME, FEEDS.NAME))
                .map(property(Feed.URL, FEEDS.URL))
                .map(property(Feed.FILENAME, FEEDS.FILENAME))
                .map(property(Feed.LOGIN, FEEDS.LOGIN))
                .map(convertibleProperty(Feed.PLAIN_PASSWORD, FEEDS.ENCRYPTED_PASSWORD,
                        this::decryptPassword, this::encryptPassword))
                .map(property(Feed.EMAIL, FEEDS.EMAIL))
                .map(property(Feed.REFRESH_INTERVAL, FEEDS.REFRESH_INTERVAL))
                .map(convertibleProperty(Feed.IS_REMOVE_UTM, FEEDS.IS_REMOVE_UTM,
                        RepositoryUtils::booleanFromLong, FeedMappings::toIsRemoveUtm))
                .map(convertibleProperty(Feed.UPDATE_STATUS, FEEDS.UPDATE_STATUS,
                        UpdateStatus::fromSource, FeedMappings::toUpdateStatus))
                .map(property(Feed.CACHED_FILE_HASH, FEEDS.CACHED_FILE_HASH))
                .map(convertibleProperty(Feed.FETCH_ERRORS_COUNT, FEEDS.FETCH_ERRORS_COUNT,
                        p -> p, RepositoryUtils::nullToZero))
                .map(property(Feed.OFFERS_COUNT, FEEDS.OFFERS_COUNT))
                .map(convertibleProperty(Feed.LAST_CHANGE, FEEDS.LAST_CHANGE,
                        p -> p, p -> nvl(p, now())))
                .map(property(Feed.LAST_REFRESHED, FEEDS.LAST_REFRESHED))
                .map(convertibleProperty(Feed.MASTER_SYSTEM, FEEDS.MASTER_SYSTEM,
                        MasterSystem::fromSource, FeedMappings::toMasterSystem))
                .map(property(Feed.TARGET_DOMAIN, FEEDS.TARGET_DOMAIN))
                .map(property(Feed.OFFER_EXAMPLES, FEEDS.OFFER_EXAMPLES))
                .map(convertibleProperty(Feed.USAGE_TYPES, FEEDS.USAGE_TYPE, FeedMappings::feedUsageTypeFromDbFormat,
                        FeedMappings::feedUsageTypeToDbFormat))
                .map(property(Feed.LAST_USED, FEEDS.LAST_USED))
                .map(property(Feed.SHOP_NAME, FEEDS.SHOP_NAME))
                .build();
    }

    /**
     * Добавление фидов
     *
     * @param feeds коллекция фидов для добавления
     * @return список идентификаторов фидов в порядке feeds
     */
    public List<Long> add(int shard, Collection<Feed> feeds) {
        // Фиды отправляемые в MBI получают ID ещё в операции.
        List<Feed> feedsWithoutId = feeds.stream().filter(f -> f.getId() == null).collect(toList());
        List<Long> ids = shardHelper.generateFeedIds(feedsWithoutId.size());
        StreamEx.of(feedsWithoutId).zipWith(ids.stream())
                .forKeyValue(Feed::setId);

        new InsertHelper<>(dslContextProvider.ppc(shard), FEEDS)
                .addAll(mapper, feeds)
                .executeIfRecordsAdded();
        var result = mapList(feeds, Feed::getId);
        logFeedStatus(shard, feeds, commonDataLogService);
        return result;
    }

    /**
     * Полчуить ID фидов у которых проставлен usage_type
     */
    public List<Long> getFeedIdsWithMarketShopId(int shard, @Nullable Collection<Long> ids, Long minFeedId,
                                                 Integer limit) {
        Condition condition = FEEDS.MARKET_SHOP_ID.isNotNull()
                .and(FEEDS.FEED_ID.gt(minFeedId));

        if (isNotEmpty(ids)) {
            condition.and(FEEDS.FEED_ID.in(ids));
        }

        return dslContextProvider.ppc(shard)
                .select(FEEDS.FEED_ID)
                .from(FEEDS)
                .where(condition)
                .orderBy(FEEDS.FEED_ID)
                .limit(limit)
                .fetch(FEEDS.FEED_ID);
    }

    private List<Feed> get(int shard, Condition condition,
                           @Nullable List<FeedsOrderBy> orders,
                           @Nullable LimitOffset limitOffset, Set<Field<?>> fieldsToRead) {
        SelectConditionStep<Record> selectConditionStep = dslContextProvider.ppc(shard)
                .select(fieldsToRead)
                .from(FEEDS)
                .where(condition);
        if (orders != null && !orders.isEmpty()) {
            List<SortField<?>> sortFields = orders.stream()
                    .map(l -> {
                        TableField<?, ?> field = l.getField().getTypedValue();
                        SortField<?> sortField;
                        if (l.getOrder() == SortOrder.DESC) {
                            sortField = field.desc();
                        } else {
                            sortField = field.asc();
                        }
                        return sortField;
                    }).collect(toList());
            selectConditionStep.orderBy(sortFields);
        } else {
            selectConditionStep
                    .orderBy(FEEDS.FEED_ID.asc());
        }
        if (limitOffset != null) {
            selectConditionStep
                    .offset(limitOffset.offset())
                    .limit(limitOffset.limit());
        }
        return selectConditionStep
                .fetch(mapper::fromDb);
    }

    public List<Feed> get(int shard, ClientId clientId, Collection<Long> ids) {
        Condition condition = FEEDS.FEED_ID.in(ids)
                .and(FEEDS.CLIENT_ID.eq(clientId.asLong()));
        return get(shard, condition, null, null, mapper.getFieldsToRead());
    }

    public List<FeedStatusesAndMarketIds> getFeedStatusesAndMarketIds(int shard, Collection<Long> feedIds) {
        return dslContextProvider.ppc(shard)
                .select(mapper.getFieldsToRead(FeedStatusesAndMarketIds.allModelProperties()))
                .from(FEEDS)
                .where(FEEDS.FEED_ID.in(feedIds))
                .fetchStream()
                .map(record -> {
                    var feed = mapper.fromDb(record);
                    return (FeedStatusesAndMarketIds) feed;
                })
                .collect(Collectors.toList());
    }

    public Map<Long, Long> getMarketShopIdByFeedId(int shard, Collection<Long> feedIds) {
        if (feedIds.isEmpty()) {
            return emptyMap();
        }
        return dslContextProvider.ppc(shard)
                .select(FEEDS.FEED_ID, FEEDS.MARKET_SHOP_ID)
                .from(FEEDS)
                .where(FEEDS.FEED_ID.in(feedIds).and(FEEDS.MARKET_SHOP_ID.isNotNull()))
                .fetchMap(FEEDS.FEED_ID, FEEDS.MARKET_SHOP_ID);
    }

    public List<Feed> get(int shard, Collection<Long> ids) {
        Condition condition = FEEDS.FEED_ID.in(ids);
        return get(shard, condition, null, null, mapper.getFieldsToRead());
    }

    public List<FeedSimple> getSimple(int shard, Collection<Long> ids) {
        return getSimple(shard, FEEDS.FEED_ID.in(ids), null, null);
    }

    public List<FeedSimple> getSimple(int shard, ClientId clientId, FeedQueryFilter feedQueryFilter) {
        return getSimple(shard, getCondition(clientId, feedQueryFilter), feedQueryFilter.getOrders(),
                feedQueryFilter.getLimitOffset());
    }

    public List<FeedSimple> getSimple(int shard, Condition condition) {
        return getSimple(shard, condition, null, null);
    }

    private List<FeedSimple> getSimple(int shard, Condition condition,
                                       @Nullable List<FeedsOrderBy> orders,
                                       @Nullable LimitOffset limitOffset) {
        var feeds = get(shard, condition, orders, limitOffset, mapper.getFieldsToRead(FeedSimple.allModelProperties()));
        return StreamEx.of(feeds)
                .map(feed -> (FeedSimple) feed)
                .toList();
    }

    public List<FeedSimple> getSimpleUpdatingFeeds(int shard, Collection<Long> feedIds) {
        return getSimple(shard, FEEDS.FEED_ID.in(feedIds).and(FEEDS.UPDATE_STATUS.eq(FeedsUpdateStatus.Updating)));
    }

    public void setFeedsStatus(int shard, UpdateStatus updateStatus, Collection<FeedSimple> feeds) {
        if (CollectionUtils.isEmpty(feeds)) {
            return;
        }
        var feedIds = StreamEx.of(feeds)
                .map(FeedSimple::getId)
                .toList();
        logFeedStatus(shard, feeds, commonDataLogService);
        dslContextProvider.ppc(shard)
                .update(FEEDS)
                .set(FEEDS.UPDATE_STATUS, UpdateStatus.toSource(updateStatus))
                .where(FEEDS.FEED_ID.in(feedIds))
                .execute();
    }

    public List<Feed> get(int shard, ClientId clientId, FeedQueryFilter feedQueryFilter) {
        return get(shard, getCondition(clientId, feedQueryFilter), feedQueryFilter.getOrders(),
                feedQueryFilter.getLimitOffset(), mapper.getFieldsToRead());
    }

    private static Condition getCondition(ClientId clientId, FeedQueryFilter feedQueryFilter) {
        Condition condition = FEEDS.CLIENT_ID.eq(clientId.asLong());
        if (feedQueryFilter.getFeedIds() != null) {
            condition = condition.and(FEEDS.FEED_ID.in(feedQueryFilter.getFeedIds()));
        }
        if (feedQueryFilter.getBusinessIdsAndShopIds() != null) {
            condition = condition.and(row(FEEDS.MARKET_BUSINESS_ID, FEEDS.MARKET_SHOP_ID).in(
                    mapList(feedQueryFilter.getBusinessIdsAndShopIds(),
                            id -> row(id.getBusinessId(), id.getShopId()))));
        }
        if (feedQueryFilter.getUpdateStatuses() != null) {
            List<FeedsUpdateStatus> feedsUpdateStatuses =
                    mapList(feedQueryFilter.getUpdateStatuses(), UpdateStatus::toSource);
            condition = condition.and(FEEDS.UPDATE_STATUS.in(feedsUpdateStatuses));
        }
        if (feedQueryFilter.getSources() != null) {
            List<FeedsSource> feedsSources = mapList(feedQueryFilter.getSources(), Source::toSource);
            condition = condition.and(FEEDS.SOURCE.in(feedsSources));
        }
        if (feedQueryFilter.getTypes() != null) {
            List<FeedsBusinessType> feedsTypes = mapList(feedQueryFilter.getTypes(), BusinessType::toSource);
            condition = condition.and(FEEDS.BUSINESS_TYPE.in(feedsTypes));
        }
        if (feedQueryFilter.getNames() != null) {
            condition = condition.and(FEEDS.NAME.in(feedQueryFilter.getNames()));
        }
        if (feedQueryFilter.getTargetDomains() != null) {
            condition = condition.and(FEEDS.TARGET_DOMAIN.in(feedQueryFilter.getTargetDomains()));
        }
        String searchBy = feedQueryFilter.getSearchBy();
        if (searchBy != null) {
            Condition searchCondition = FEEDS.NAME.contains(searchBy);
            Long asId = tryParseLong(searchBy);
            if (asId != null && asId > 0L) {
                searchCondition = searchCondition.or(FEEDS.FEED_ID.eq(asId));
            }
            condition = condition.and(searchCondition);
        }
        return condition;
    }

    private static Long tryParseLong(String value) {
        try {
            return Long.parseLong(value);
        } catch (NumberFormatException e) {
            return null;
        }
    }

    /**
     * Возвращает упрощенные модели всех фидов, загружаемых через DataCamp. В том числе:
     * - фиды, созданные в Директе и отправленные в MBI;
     * - фиды, созданные во внешних системах и загруженные из них в Директ.
     */
    @QueryWithoutIndex("Вызывается только из jobs")
    public List<FeedSimple> getAllDataCampFeedsSimple(int shard) {
        return getSimple(shard, SENT_TO_MBI);
    }

    public Map<Long, FeedsUpdateStatus> getStatusesById(int shard, ClientId clientId, Collection<Long> feedIds) {
        return dslContextProvider.ppc(shard)
                .select(FEEDS.FEED_ID, FEEDS.UPDATE_STATUS)
                .from(FEEDS)
                .where(FEEDS.CLIENT_ID.eq(clientId.asLong()).and(FEEDS.FEED_ID.in(feedIds)))
                .fetchMap(FEEDS.FEED_ID, FEEDS.UPDATE_STATUS);
    }

    public void delete(int shard, Collection<Long> feedIds) {
        var context = dslContextProvider.ppc(shard);
        feedSupplementaryDataRepository.deleteFeedVendors(context, feedIds);
        feedSupplementaryDataRepository.deleteFeedCategories(context, feedIds);
        feedSupplementaryDataRepository.deleteFeedHistory(context, feedIds);
        deleteFeedsById(context, feedIds);
    }

    private void deleteFeedsById(DSLContext dslContext, Collection<Long> feedIds) {
        dslContext
                .deleteFrom(FEEDS)
                .where(FEEDS.FEED_ID.in(feedIds))
                .execute();
    }


    /**
     * @param shard    номер шарда
     * @param clientId идентификатор клиента
     * @return количество фидов клиента, загружаемых через Баннерленд, т.е. без фидов загружаемых через DataCamp
     */
    public int getBlCount(int shard, ClientId clientId) {
        Condition condition = FEEDS.CLIENT_ID.eq(clientId.asLong())
                .and(FEEDS.MASTER_SYSTEM.eq(FeedsMasterSystem.direct))
                .and(not(SENT_TO_MBI));
        return dslContextProvider.ppc(shard)
                .selectCount()
                .from(FEEDS)
                .where(condition)
                .fetchOne(0, int.class);
    }

    public void update(int shard, List<AppliedChanges<Feed>> appliedChanges) {
        DSLContext context = dslContextProvider.ppc(shard);
        update(context, appliedChanges);
    }

    public void update(DSLContext context, List<AppliedChanges<Feed>> appliedChanges) {
        new UpdateHelper<>(context, FEEDS.FEED_ID)
                .processUpdateAll(mapper, appliedChanges)
                .execute();
    }

    private String decryptPassword(@Nullable byte[] encrypted) {
        return encrypted == null ? null : encrypter.decryptText(new String(encrypted));
    }

    private byte[] encryptPassword(@Nullable String text) {
        return text == null ? null : encrypter.encryptText(text).getBytes();
    }

    public List<FeedSimple> getFeedsToSendToMBI(int shard) {
        return getSimple(shard, FEEDS.STATUS_MBI_SYNCED.eq(FeedsStatusmbisynced.No));
    }

    public List<FeedSimple> getFeedsSimpleWithSpecifiedLastUsed(int shard, LocalDateTime lastUsedFrom,
                                                                LocalDateTime lastUsedTo) {
        Condition condition = SENT_TO_MBI.and(FEEDS.LAST_USED.between(lastUsedFrom, lastUsedTo));
        return getSimple(shard, condition, null, null);
    }

    /**
     * Получить id фидов используемых в группах
     * В одной группе может быть не более одного фида, интересуют только смартовые, динамические и текстовые группы
     * Если фида в группе нет - в мапе не будет соответствующей записи
     *
     * @param shard      - шард
     * @param adGroupIds - идентификаторы групп
     * @return мапа adGroupId -> feedId
     */
    public Map<Long, Long> getFeedIdsByAdGroupIds(int shard, Collection<Long> adGroupIds) {
        return dslContextProvider.ppc(shard)
                .select(PHRASES.PID, ADGROUPS_PERFORMANCE.FEED_ID, ADGROUPS_DYNAMIC.FEED_ID, ADGROUPS_TEXT.FEED_ID)
                .from(PHRASES)
                .leftJoin(ADGROUPS_PERFORMANCE).on(ADGROUPS_PERFORMANCE.PID.eq(PHRASES.PID))
                .leftJoin(ADGROUPS_DYNAMIC).on(ADGROUPS_DYNAMIC.PID.eq(PHRASES.PID))
                .leftJoin(ADGROUPS_TEXT).on(ADGROUPS_TEXT.PID.eq(PHRASES.PID))
                .where(PHRASES.PID.in(adGroupIds).and(PHRASES.ADGROUP_TYPE.in(EnumSet.of(
                                PhrasesAdgroupType.base,
                                PhrasesAdgroupType.performance,
                                PhrasesAdgroupType.dynamic))
                        ).and(ADGROUPS_PERFORMANCE.FEED_ID.isNotNull()
                                .or(ADGROUPS_DYNAMIC.FEED_ID.isNotNull()
                                        .or(ADGROUPS_TEXT.FEED_ID.isNotNull())))
                ).fetchMap(record -> record.get(PHRASES.PID), record -> coalesce(
                        record.get(ADGROUPS_PERFORMANCE.FEED_ID),
                        record.get(ADGROUPS_DYNAMIC.FEED_ID),
                        record.get(ADGROUPS_TEXT.FEED_ID))
                );
    }

    public Map<CampaignIdAndAdGroupIdPair, Long> getCampaignIdAdGroupIdByFeedsInPerformance(int shard,
                                                                                            Collection<Long> feedIds) {
        return dslContextProvider.ppc(shard).select(ADGROUPS_PERFORMANCE.PID, CAMPAIGNS.CID,
                        ADGROUPS_PERFORMANCE.FEED_ID)
                .from(ADGROUPS_PERFORMANCE)
                .join(PHRASES).on(PHRASES.PID.eq(ADGROUPS_PERFORMANCE.PID))
                .join(CAMPAIGNS).on(CAMPAIGNS.CID.eq(PHRASES.CID))
                .where(ADGROUPS_PERFORMANCE.FEED_ID.in(feedIds))
                .fetchMap(record -> new CampaignIdAndAdGroupIdPair()
                                .withCampaignId(record.get(CAMPAIGNS.CID))
                                .withAdGroupId(record.get(ADGROUPS_PERFORMANCE.PID)),
                        record -> record.get(ADGROUPS_PERFORMANCE.FEED_ID));
    }

    public Map<CampaignIdAndAdGroupIdPair, Long> getCampaignIdAdGroupIdByFeedsInDynamic(int shard,
                                                                                        Collection<Long> feedIds) {
        return dslContextProvider.ppc(shard).select(ADGROUPS_DYNAMIC.PID, ADGROUPS_DYNAMIC.FEED_ID, CAMPAIGNS.CID)
                .from(ADGROUPS_DYNAMIC)
                .join(PHRASES).on(PHRASES.PID.eq(ADGROUPS_DYNAMIC.PID))
                .join(CAMPAIGNS).on(CAMPAIGNS.CID.eq(PHRASES.CID))
                .where(ADGROUPS_DYNAMIC.FEED_ID.in(feedIds))
                .fetchMap(record -> new CampaignIdAndAdGroupIdPair()
                                .withCampaignId(record.get(CAMPAIGNS.CID))
                                .withAdGroupId(record.get(ADGROUPS_DYNAMIC.PID)),
                        record -> record.get(ADGROUPS_DYNAMIC.FEED_ID));
    }

    @Nullable
    public FeedSimple getManuallyAddedFeed(int shard, ClientId clientId) {
        Condition condition = FEEDS.CLIENT_ID.eq(clientId.asLong())
                .and(FEEDS.MASTER_SYSTEM.eq(FeedsMasterSystem.manual));

        List<FeedsOrderBy> order = singletonList(
                new FeedsOrderBy()
                        .withField(FeedOrderByField.ID)
                        .withOrder(SortOrder.ASC)
        );

        LimitOffset limitOffset = new LimitOffset(1, 0);

        List<FeedSimple> feeds = getSimple(shard, condition, order, limitOffset);
        if (feeds.isEmpty()) {
            return null;
        }
        return feeds.get(0);
    }

    @Nullable
    public Feed getShopInShopFeedByUrl(int shard, ClientId clientId, String url) {
        Condition condition = FEEDS.CLIENT_ID.eq(clientId.asLong())
                .and(FEEDS.MASTER_SYSTEM.eq(FeedsMasterSystem.shop_in_shop))
                .and(FEEDS.URL.eq(url));

        LimitOffset limitOffset = new LimitOffset(1, 0);

        return get(shard, condition, null, limitOffset, mapper.getFieldsToRead()).stream()
                .findFirst()
                .orElse(null);
    }

    public void setFeedStatusMbiEnabled(Long uid, Long feedId, boolean enabled) {
        int shard = shardHelper.getShardByUserId(uid);
        dslContextProvider.ppc(shard)
                .update(FEEDS)
                .set(FEEDS.STATUS_MBI_ENABLED, enabled ? FeedsStatusmbienabled.Yes : FeedsStatusmbienabled.No)
                .where(FEEDS.FEED_ID.eq(feedId))
                .execute();
    }

    /**
     * Метод достает из шарда [limit] фидов в статусе New, со старейшей датой feed.LastChange, для отправки в BmAPI
     * Фиды с source=SITE должны быть зарегистрированы в MBI до отправки в BmAPI
     *
     * @param shard шард из которого доставать фиды
     * @param limit сколько фидов доставать
     * @return список упрощенных представлений фидов
     */
    @NotNull
    public List<FeedSimple> getNewFeedsToSendToBmAPI(int shard, int limit) {
        // проверка проперти, для переключения между джобой и скриптом для новых фидов
        Optional<Boolean> optionalFeatureId = ppcPropertiesSupport
                .get(SEND_NEW_FEEDS_IN_BL_FROM_JOBS, Duration.of(5, ChronoUnit.MINUTES))
                .find();
        if (optionalFeatureId.isEmpty() || !optionalFeatureId.get()) {
            return emptyList();
        }

        Condition condition = FEEDS.UPDATE_STATUS.eq(FeedMappings.toUpdateStatus(UpdateStatus.NEW))
                .andNot(FEEDS.SOURCE.eq(FeedsSource.site).and(FEEDS.STATUS_MBI_SYNCED.eq(FeedsStatusmbisynced.No)));
        List<FeedsOrderBy> orderBy =
                List.of(new FeedsOrderBy().withField(FeedOrderByField.LAST_CHANGE).withOrder(SortOrder.ASC));
        LimitOffset limitOffset = LimitOffset.limited(limit);
        return getSimple(shard, condition, orderBy, limitOffset);
    }

}
