package ru.yandex.direct.core.entity.feed.repository

import org.jooq.DSLContext
import org.jooq.Record
import org.jooq.types.ULong
import org.springframework.stereotype.Repository
import ru.yandex.direct.common.jooqmapperex.ReaderWriterBuildersEx.booleanProperty
import ru.yandex.direct.common.jooqmapperex.ReaderWriterBuildersEx.compressedMediumblobJsonProperty
import ru.yandex.direct.common.util.RepositoryUtils
import ru.yandex.direct.core.entity.feed.model.FeedCategory
import ru.yandex.direct.core.entity.feed.model.FeedHistoryItem
import ru.yandex.direct.core.entity.feed.model.FeedHistoryItemParseResults
import ru.yandex.direct.core.entity.feed.model.FeedVendor
import ru.yandex.direct.dbschema.ppc.Tables
import ru.yandex.direct.dbutil.wrapper.DslContextProvider
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier
import ru.yandex.direct.jooqmapper.JooqMapperWithSupplierBuilder
import ru.yandex.direct.jooqmapper.ReaderWriterBuilders.convertibleProperty
import ru.yandex.direct.jooqmapper.ReaderWriterBuilders.property
import ru.yandex.direct.jooqmapperhelper.InsertHelper
import ru.yandex.direct.jooqmapperhelper.JooqUpdateBuilder
import ru.yandex.direct.model.AppliedChanges
import java.math.BigInteger
import javax.annotation.ParametersAreNonnullByDefault

/**
 * Репозиторий для работы с дополнительными таблицами фидов — история, категории, вендоры
 */
@Repository
@ParametersAreNonnullByDefault
class FeedSupplementaryDataRepository(
    private val dslContextProvider: DslContextProvider,
) {
    private val feedVendorMapper: JooqMapperWithSupplier<FeedVendor>
    private val feedHistoryItemMapper: JooqMapperWithSupplier<FeedHistoryItem>
    private val feedCategoryMapper: JooqMapperWithSupplier<FeedCategory>

    init {
        feedVendorMapper = createFeedVendorMapper()
        feedHistoryItemMapper = createFeedHistoryItemMapper()
        feedCategoryMapper = createFeedCategoryMapper()
    }

    private fun createFeedCategoryMapper(): JooqMapperWithSupplier<FeedCategory> {
        return JooqMapperWithSupplierBuilder.builder { FeedCategory() }
            .map(property(FeedCategory.ID, Tables.PERF_FEED_CATEGORIES.ID))
            .map(convertibleProperty(
                FeedCategory.CATEGORY_ID,
                Tables.PERF_FEED_CATEGORIES.CATEGORY_ID,
                { unLong: ULong -> RepositoryUtils.bigIntegerFromULong(unLong) })
            { bigInteger: BigInteger -> RepositoryUtils.bigIntegerToULong(bigInteger) })
            .map(property(FeedCategory.FEED_ID, Tables.PERF_FEED_CATEGORIES.FEED_ID))
            .map(
                booleanProperty(FeedCategory.IS_DELETED, Tables.PERF_FEED_CATEGORIES.IS_DELETED)
            )
            .map(property(FeedCategory.NAME, Tables.PERF_FEED_CATEGORIES.NAME))
            .map(property(FeedCategory.OFFER_COUNT, Tables.PERF_FEED_CATEGORIES.OFFERS_COUNT))
            .map(convertibleProperty(
                FeedCategory.PARENT_CATEGORY_ID,
                Tables.PERF_FEED_CATEGORIES.PARENT_CATEGORY_ID,
                { unLong: ULong -> RepositoryUtils.bigIntegerFromULong(unLong) })
            { bigInteger: BigInteger -> RepositoryUtils.bigIntegerToULong(bigInteger) })
            .build()
    }

    private fun createFeedHistoryItemMapper(): JooqMapperWithSupplier<FeedHistoryItem> {
        return JooqMapperWithSupplierBuilder.builder { FeedHistoryItem() }
            .map(property(FeedHistoryItem.ID, Tables.PERF_FEED_HISTORY.ID))
            .map(property(FeedHistoryItem.FEED_ID, Tables.PERF_FEED_HISTORY.FEED_ID))
            .map(property(FeedHistoryItem.CREATED_AT, Tables.PERF_FEED_HISTORY.CREATED_AT))
            .map(property(FeedHistoryItem.OFFER_COUNT, Tables.PERF_FEED_HISTORY.OFFERS_COUNT))
            .map(
                compressedMediumblobJsonProperty(
                    FeedHistoryItem.PARSE_RESULTS,
                    Tables.PERF_FEED_HISTORY.PARSE_RESULTS_JSON_COMPRESSED,
                    FeedHistoryItemParseResults::class.java
                )
            )
            .build()
    }

    private fun createFeedVendorMapper(): JooqMapperWithSupplier<FeedVendor> {
        return JooqMapperWithSupplierBuilder.builder { FeedVendor() }
            .map(property(FeedVendor.VENDOR_ID, Tables.PERF_FEED_VENDORS.ID))
            .map(property(FeedVendor.FEED_ID, Tables.PERF_FEED_VENDORS.FEED_ID))
            .map(property(FeedVendor.NAME, Tables.PERF_FEED_VENDORS.NAME))
            .build()
    }

    fun getFeedCategories(shard: Int, feedIds: List<Long>): List<FeedCategory> {
        return dslContextProvider.ppc(shard)
            .select(feedCategoryMapper.fieldsToRead)
            .from(Tables.PERF_FEED_CATEGORIES)
            .where(Tables.PERF_FEED_CATEGORIES.FEED_ID.`in`(feedIds))
            .fetch { record: Record -> feedCategoryMapper.fromDb(record) }
    }

    fun updateCategories(dslContext: DSLContext, changes: List<AppliedChanges<FeedCategory>>) {
        val ub = JooqUpdateBuilder(Tables.PERF_FEED_CATEGORIES.ID, changes)
        ub.processProperty(FeedCategory.NAME, Tables.PERF_FEED_CATEGORIES.NAME)
        ub.processProperty(
            FeedCategory.IS_DELETED,
            Tables.PERF_FEED_CATEGORIES.IS_DELETED
        ) { isDeleted -> if (isDeleted) 1 else 0 }
        ub.processProperty(FeedCategory.OFFER_COUNT, Tables.PERF_FEED_CATEGORIES.OFFERS_COUNT)
        ub.processProperty(
            FeedCategory.PARENT_CATEGORY_ID,
            Tables.PERF_FEED_CATEGORIES.PARENT_CATEGORY_ID
        ) { parentId -> RepositoryUtils.bigIntegerToULong(parentId) }

        dslContext
            .update(Tables.PERF_FEED_CATEGORIES)
            .set(ub.values)
            .where(Tables.PERF_FEED_CATEGORIES.ID.`in`(ub.changedIds))
            .execute();
    }

    fun markFeedCategoriesDeleted(dslContext: DSLContext, categoriesToDelete: List<FeedCategory>) {
        val categoryIds = categoriesToDelete.map { it.id }
        dslContext
            .update(Tables.PERF_FEED_CATEGORIES)
            .set(Tables.PERF_FEED_CATEGORIES.IS_DELETED, 1)
            .where(Tables.PERF_FEED_CATEGORIES.ID.`in`(categoryIds))
            .execute()
    }

    fun addCategories(dslContext: DSLContext, categoriesToAdd: List<FeedCategory>) {
        InsertHelper(dslContext, Tables.PERF_FEED_CATEGORIES)
            .addAll(feedCategoryMapper, categoriesToAdd)
            .executeIfRecordsAdded()
    }

    fun getFeedCategoriesByFeedId(shard: Int, feedIds: Collection<Long>): Map<Long, List<FeedCategory>> {
        return dslContextProvider.ppc(shard)
            .select(feedCategoryMapper.fieldsToRead)
            .from(Tables.PERF_FEED_CATEGORIES)
            .where(Tables.PERF_FEED_CATEGORIES.FEED_ID.`in`(feedIds))
            .fetch { record: Record -> feedCategoryMapper.fromDb(record) }
            .groupBy { it.feedId }
    }

    fun getFeedVendorsByFeedId(shard: Int, feedIds: Collection<Long>): Map<Long, List<FeedVendor>> {
        return dslContextProvider.ppc(shard)
            .select(feedVendorMapper.fieldsToRead)
            .from(Tables.PERF_FEED_VENDORS)
            .where(Tables.PERF_FEED_VENDORS.FEED_ID.`in`(feedIds))
            .fetch { record: Record -> feedVendorMapper.fromDb(record) }
            .groupBy { it.feedId }
    }

    fun deleteFeedCategories(dslContext: DSLContext, feedIds: Collection<Long>) {
        dslContext
            .deleteFrom(Tables.PERF_FEED_CATEGORIES)
            .where(Tables.PERF_FEED_CATEGORIES.FEED_ID.`in`(feedIds))
            .execute()
    }

    fun deleteFeedVendors(dslContext: DSLContext, feedIds: Collection<Long>) {
        dslContext
            .deleteFrom(Tables.PERF_FEED_VENDORS)
            .where(Tables.PERF_FEED_VENDORS.FEED_ID.`in`(feedIds))
            .execute()
    }

    fun deleteFeedHistory(dslContext: DSLContext, feedIds: Collection<Long>) {
        dslContext
            .deleteFrom(Tables.PERF_FEED_HISTORY)
            .where(Tables.PERF_FEED_HISTORY.FEED_ID.`in`(feedIds))
            .execute()
    }

    fun addFeedVendors(dslContext: DSLContext, vendorsByFeedIds: Map<Long, List<String>>) {
        val feedVendors: MutableList<FeedVendor> = mutableListOf()
        vendorsByFeedIds.entries.forEach { (feedId, vendorsList) ->
            feedVendors.addAll(vendorsList.map {
                FeedVendor()
                    .withFeedId(feedId)
                    .withName(it)
            })
        }
        InsertHelper(dslContext, Tables.PERF_FEED_VENDORS)
            .addAll(feedVendorMapper, feedVendors)
            .executeIfRecordsAdded()
    }

    /**
     * Под капотом забирается вся история фидов с момента их создания.
     * Не стоит делать сильно массовые запросы.
     */
    fun getLatestFeedHistoryItems(shard: Int, feedIds: Collection<Long>): Map<Long, FeedHistoryItem> {
        val historyByFeedId = getFeedHistoryItems(shard, feedIds)
        return historyByFeedId
            .filterValues { it.isNotEmpty() }
            .mapValues { it.value.maxByOrNull { feedHistoryItem -> feedHistoryItem.id }!! }
    }

    /**
     * Данных может быть очень много, хранится вся история фидов с момента их создания.
     * Не стоит делать сильно массовые запросы.
     */
    fun getFeedHistoryItems(shard: Int, feedIds: Collection<Long>): Map<Long, List<FeedHistoryItem>> {
        return if (feedIds.isEmpty()) {
            emptyMap()
        } else dslContextProvider.ppc(shard)
            .select(feedHistoryItemMapper.fieldsToRead)
            .from(Tables.PERF_FEED_HISTORY)
            .where(Tables.PERF_FEED_HISTORY.FEED_ID.`in`(feedIds))
            .fetchGroups(Tables.PERF_FEED_HISTORY.FEED_ID, feedHistoryItemMapper::fromDb)
    }

    fun addHistoryItems(dslContext: DSLContext, historyItemsToInsert: MutableList<FeedHistoryItem>) {
        InsertHelper(dslContext, Tables.PERF_FEED_HISTORY)
            .addAll(feedHistoryItemMapper, historyItemsToInsert)
            .executeIfRecordsAdded()
    }
}
